package com.agilex.soap;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.SocketTimeoutException;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLStreamHandler;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLSession;
import javax.xml.soap.SOAPConnection;
import javax.xml.soap.SOAPConnectionFactory;
import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPMessage;

import com.agilex.soap.exceptions.SoapServiceException;
import com.agilex.soap.exceptions.SoapTimeoutException;

/**
 * Wraps a Http Connection and supports sending a request to a SOAP endpoint.
 */
public class Service {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(Service.class);

	private URL url;

    /**
     * Creates an instance of a Service connection.  This should be the only constructor because we want
     * to require both a connectionTimeout and readTimeout.
     * @param endPoint          URL to send SOAP request.
     * @param connectTimeout    A SocketTimeout will be thrown if connection to the endpoint is not made in this
     *                          many milliseconds.
     * @param readTimeout       A SocketTimeout will be thrown if the endpoint does not return the complete response
     *                          in this many milliseconds.
     */
	public Service(String endPoint, int connectTimeout, int readTimeout) {
		createEndpoint(endPoint, connectTimeout, readTimeout);
	}

	private void createEndpoint(String endPoint, final int connectTimeout, final int readTimeout) {
		if (logger.isDebugEnabled())
			logger.debug("Creating connection to endpoint: " + endPoint + " with connectTimeout of: " + connectTimeout + " and readTimeout of: " + readTimeout);

		try {
			url = new URL(null, endPoint, new URLStreamHandler() {
				@Override
				protected URLConnection openConnection(URL url) throws IOException {
					URL target = null;
					
					if (url.toString().startsWith("https")){
						target = new URL(null, url.toString(), new sun.net.www.protocol.https.Handler());
					}else{
						target = new URL(url.toString());
					}		
					
					URLConnection connection = target.openConnection();
					connection.setConnectTimeout(connectTimeout);
					connection.setReadTimeout(readTimeout);

					logger.info("URL Connection Class impl: " + connection.getClass().getName());
					
					if (connection instanceof sun.net.www.protocol.https.HttpsURLConnectionImpl){
						logger.info("Overiding the sun.net.www.protocol.https.HttpsURLConnectionImpl with Hostname Verifier");
						
						((sun.net.www.protocol.https.HttpsURLConnectionImpl) connection).setHostnameVerifier(new HostnameVerifier() {
					        public boolean verify(String s, SSLSession sslSession) {
					        	logger.info(" Its a valid host");
					            return true;
					        }
					    });
					}
					
					return (connection);
				}
			});
		} catch (MalformedURLException e) {
			throw new SoapServiceException(endPoint + " is not valid.", e);
		}

	}

    /**
     * Sends a SOAP request message to an endpoint.
     * @param request - SOAP message to send to endpoint specified in constructor.
     * @return SOAPMessage - SOAP response message from endpoint specified in constructor.
     * @throws SoapTimeoutException is the connection or read response actions timeout or
     *         a SoapServiceException for any other exception generated during the SOAP call.
     *         An exception is NOT throw if the message contains a fault.  The consumer must
     *         process SOAP faults.
     */
	public SOAPMessage send(SOAPMessage request) {
		if (url == null)
			throw new SoapServiceException("Send aborted.  URL was either not set or not valid.");

		// log response & processing time

		SOAPMessage response = null;
		SOAPConnection connection = createConnection();

		try {
			response = connection.call(request, url);
		} catch (SOAPException e) {
			if (containsTimeoutException(e))
				throw new SoapTimeoutException("Timeout occurred ", e);
			else
				throw new SoapServiceException("Error sending message ", e);
		} finally {
            closeConnection(connection);
        }

		return response;
	}

	private SOAPConnection createConnection() {
		SOAPConnection connection = null;

		try {
			connection = SOAPConnectionFactory.newInstance().createConnection();
		} catch (SOAPException e) {
			throw new SoapServiceException("Unable to create a SOAP connection object.", e);
		}
		return connection;
	}

    private void closeConnection(SOAPConnection connection) {
        if (connection == null)
            return;

        try {
            connection.close();
        } catch (SOAPException e) {
            throw new SoapServiceException("Error closing SOAP connection", e);
        }
    }

    private boolean containsTimeoutException(SOAPException e) {
        boolean hasTimeout = false;
        Throwable cause = e.getCause();

        while (cause != null) {
            if (cause instanceof SocketTimeoutException) {
                hasTimeout = true;
                break;
            }

            cause = cause.getCause();
        }

        return hasTimeout;
    }
}
