package com.agilex.soap;

import javax.xml.soap.*;
import javax.xml.transform.stream.StreamSource;

import com.agilex.soap.exceptions.*;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;

/**
 * Adapter for Java's SOAPMessage class.
 */
public class Message {
	private final SOAPMessage message;
	private Body body;
	private Header header;
	private boolean hasHeader = true;

	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(Message.class);

	public Message() {
		message = createMessage();
		initMessage();
	}

	private void initMessage() {
		try {
			message.setProperty(SOAPMessage.WRITE_XML_DECLARATION, "true");
			message.setProperty(SOAPMessage.CHARACTER_SET_ENCODING, "utf-8");
		} catch (SOAPException e) {
			throw new SoapMessageException("Unable to set xml declaration in message.");
		}
	}

	public Message(SOAPMessage soapMessage) {
		message = soapMessage;
	}

    public Message(String soapMessage) {
        message = createMessage();
        toSoapMessage(soapMessage, message);
    }

    private SOAPMessage createMessage() {
        try {
            return MessageFactory.newInstance().createMessage();
        } catch (Throwable t) {
            throw new SoapMessageException("Message factory was unable to create a SOAP message", t);
        }
    }

    private void toSoapMessage(String soapText, SOAPMessage message) {
        SOAPPart soapPart = message.getSOAPPart();

        byte[] buffer = soapText.getBytes();
        ByteArrayInputStream stream = new ByteArrayInputStream(buffer);
        StreamSource source = new StreamSource(stream);

        try {
            soapPart.setContent(source);
        } catch (SOAPException e) {
            throw new SoapMessageException("Unable to covert SOAP message string to SOAPMessage.", e);
        }
    }

	public void addNamespace(String prefix, String namespace) {
		try {
			message.getSOAPPart().getEnvelope().addNamespaceDeclaration(prefix, namespace);
		} catch (SOAPException e) {
			throw new SoapMessageException("Unable to add namespace to SOAP Envelope.");
		}
	}

	/**
	 * @return Body - SOAP message body
	 * @throws SoapMessageException
	 *             - if the body can not be retrieved from the SOAP message
	 */
	public Body getBody() {
		if (body == null) {
			try {
				body = new Body(message.getSOAPBody());
			} catch (Throwable t) {
				throw new SoapMessageException("Unable to get SOAP body from message.", t);
			}
		}
		return body;
	}

	public Header getHeader() {
		if(!hasHeader)
			throw new SoapMessageException("Header has already been explicitly detached");
		
		if (header == null) {
			try {
				header = new Header(message.getSOAPHeader(), message.getMimeHeaders());
			} catch (Throwable t) {
				throw new SoapMessageException("Unable to get SOAP header from message", t);
			}
		}

		return header;
	}

	/**
	 * Some SOAP messages do not require a SOAP header so it can be removed.
	 * 
	 * @return Message - this instance of Message
	 * @throws SoapMessageException
	 *             - if the header can not be retrieved from the SOAP message
	 */
	public Message removeHeader() {
		if (!hasHeader)
			return this;

		try {
			SOAPHeader header = message.getSOAPHeader();
			header.detachNode();
			this.header = null;
			this.hasHeader = false;
		} catch (Throwable t) {
			throw new SoapMessageException("Unable to get SOAP header.", t);
		}

		return this;
	}

	/**
	 * Sends a SOAP message to a service.
	 * 
	 * @param service
	 *            - service to send SOAP message to.
	 * @return Message - the response from the service
	 * @throws SoapServiceException
	 *             - if service call timeouts or service is not available
	 */
	public Message send(Service service) {
		if (logger.isDebugEnabled()) {
			logger.debug("Request message follows.");
			logger.debug(toString(message));
		}

		SOAPMessage response = service.send(message);

		if (logger.isDebugEnabled()) {
			logger.debug("Response message follows.");
			logger.debug(toString(response));
		}

		return new Message(response);
	}

	@Override
	public String toString() {
		return toString(message);
	}

	private String toString(SOAPMessage msg) {
		ByteArrayOutputStream byteArrayOS = new ByteArrayOutputStream();
		try {
			msg.writeTo(byteArrayOS);
		} catch (Exception e) {
			return "Unable to serialize message.";
		}
		return new String(byteArrayOS.toByteArray());
	}
}
