package com.agilex.soap;

import com.agilex.soap.exceptions.SoapMessageException;

import javax.xml.soap.*;
import java.util.*;

/**
 * Adapter for Java's SOAPElement class.
 */
public class Element {
    private final SOAPElement element;

    public Element(SOAPElement element) {
        this.element = element;
    }

    /**
     * Creates and adds a child element to this element.
     *
     * @param name - local name of the element
     * @return Element - new child element of this element
     * @throws SoapMessageException - if unable to create a child SOAP element
     */
    public Element addChild(String name) {
        try {
            SOAPElement newElement = element.addChildElement(name);
            return new Element(newElement);
        } catch (Throwable t) {
            throw new SoapMessageException("Unable to create a child SOAP element.", t);
        }
     }

    /**
     * Creates and adds a child element to this element.
     *
     * @param name - local name of the element
     * @param prefix - namespace prefix
     * @param namespace - namespace for new element
     * @return Element - new child element of this element
     * @throws SoapMessageException - if unable to create a child SOAP element
     */
    public Element addChild(String name, String prefix, String namespace) {
        try {
            SOAPElement newElement = element.addChildElement(Factory.createName(name, prefix, namespace));
            return new Element(newElement);
        } catch (Throwable t) {
            throw new SoapMessageException("Unable to create a child SOAP element.", t);
        }
    }

    /**
     * Creates and adds a child element to this element. Adds an attribute to the child element.
     *
     * @param name - local name of the element
     * @param attribute - attribute name
     * @param attributeValue - attribute value
     * @return Element - new child element of this element
     * @throws SoapMessageException - if unable to create a child SOAP element
     */
    public Element addChildWithAttribute(String name, String attribute, String attributeValue) {
        Element newElement = addChild(name);
        newElement.addAttribute(attribute, attributeValue);

        return newElement;
    }

    /**
     * Creates and adds a child element to this element. Adds multiple attribute to the child element.
     * The attributes and attributeValues arrays should have the same number of elements.
     *
     * @param name - local name of the element
     * @param attributes - array of attribute names
     * @param attributeValues - array of attribute values
     * @return Element - new child element of this element
     * @throws SoapMessageException - if unable to create a child SOAP element, if one of the attribute arrays
     *                                is null or the length of the attribute arrays are different
     */
    public Element addChildWithAttributes(String name, String attributes[], String attributeValues[]) {
        Element newElement = addChild(name);
        newElement.addAttributes(attributes, attributeValues);

        return newElement;
    }

    /**
     * Adds a single attribute to this element.
     *
     * @param attribute - attribute name
     * @param value - attribute value
     * @return Element - this element
     */
    public Element addAttribute(String attribute, String value)  {
        element.setAttribute(attribute, value);

        return this;
    }

    /**
     * Adds multiple attributes to this element. The attributes and attributeValues arrays should have the
     * same number of elements.
     *
     * @param attributes - array of attribute names
     * @param attributeValues - array of attribute values
     * @return Element - this element
     * @throws SoapMessageException - if one of the attribute arrays is null or the length of the
     *                                attribute arrays are different
     */
    public Element addAttributes(String[] attributes, String[] attributeValues)  {
        if (attributes == null || attributeValues == null)
            throw new SoapMessageException("Attributes and attribute values must not be null.");

        if (attributes.length != attributeValues.length)
            throw new SoapMessageException("Unable to add attributes to element. " +
                    "There must be an equal number of attributes and attribute values.  ");
//                    "The following attributes and values were passed in: " +
//                    Arrays.toString(attributes) + " and " + Arrays.toString(attributeValues));

        for (int i = 0; i < attributes.length; i++)
            element.setAttribute(attributes[i], attributeValues[i]);

        return this;
    }

    /**
     * Adds a namespace to this element without a prefix.
     *
     * @param namespace - namespace to add to this element
     * @return Element - this element
     * @throws SoapMessageException - if the namespace can not be added to this element
     */
    public Element addNamespace(String namespace) {
        return addNamespace("", namespace);
    }

    /**
     * Adds a namespace with a prefix to this element.
     *
     * @param prefix - namespace prefix
     * @param namespace  - namespace to add to this element
     * @return Element - this element
     * @throws SoapMessageException - if the namespace can not be added to this element
     */
    public Element addNamespace(String prefix, String namespace) {
        try {
            element.addNamespaceDeclaration(prefix, namespace);
            return this;
        } catch (Throwable t) {
            throw new SoapMessageException("Unable to add namespace to element.", t);
        }
    }

    /**
     * Adds text to this element. Ex. <Element>txt would be here</Element>
     *
     * @param txt - text to add to this element
     * @return Element - this element
     * @throws SoapMessageException - if the text can not be added to this element
     */
    public Element addElementValue(String txt) {
        try {
            element.addTextNode(txt);
            return this;
        } catch (Throwable t) {
            throw new SoapMessageException("Unable to add text to element.", t);
        }
    }

    /**
     * Get an attribute value.
     *
     * @param name - attribute to get value of
     * @return String - return attribute value or Null if attribute not found
     */
    public String getAttributeValue(String name) {
        return element.getAttributeValue(Factory.createName(name));
    }

    /**
     * Get text for this element.   Ex. <Element>txt would be here</Element>
     *
     * @return String - text for this element if there is a Text node or the text of the first immediate child of this with a
     * Test node.  Null is return if no Text node is found.
     */
    public String getElementValue() {
        return element.getValue();
    }

    /**
     * Returns a collection of child elements that have the same name.
     *
     * @param nodeName - name of child Elements to filter on
     * @param prefix - namespace prefix of element
     * @param nameSpace - namespace of element
     * @return Elements - All the child Elements for this Element with a name that matches the nodeName parameter
     * @throws SoapMessageException - if a node in nodeNames parameter is not found in the hierarchy
     */
    public Elements getChildren(String nodeName, String prefix, String nameSpace) {
        Iterator iterator = getChildren(element, Factory.createName(nodeName, prefix, nameSpace));

        return new Elements(iterator);
    }

    /**
     * This method can be used to select child Element of this Elements.  This method will use the
     * first Element found for Element name in nodeNames.
     *
     * @param nodeName - name of child Element to find
     * @param prefix - namespace prefix of elements
     * @param nameSpace - namespace of elements
     * @return Element - child Element at nodeNames[last]
     * @throws SoapMessageException - if a node in nodeNames parameter is not found in the hierarchy
     */
    public Element getChild(String nodeName, String prefix, String nameSpace) {

        Iterator iterator = getChildren(element, Factory.createName(nodeName, prefix, nameSpace));
        SOAPElement childElement = getNext(iterator);

        return new Element(childElement);
    }

    /**
     * This method can be used to select a nested Element in a hierarchy of Elements.  This method will use the
     * first Element found for Element name in nodeNames. This method only works for a hierarchy of Elements in the
     * sames namespace (and same prefix).
     *
     * @param nodeNames - list of names to drill down into a hierarchy of Elements with nodeNames[0] being the
     *                    first Element that is a child of this Element.
     * @param prefix - namespace prefix of elements
     * @param nameSpace - namespace of elements
     * @return Element - child Element at nodeNames[last]
     * @throws SoapMessageException - if a node in nodeNames parameter is not found in the hierarchy
     */
    public Element getChild(String[] nodeNames, String prefix, String nameSpace) {
        SOAPElement loopElement = element;

        for(String nodeName : nodeNames) {
            Iterator iterator = getChildren(loopElement, Factory.createName(nodeName, prefix, nameSpace));
            loopElement = getNext(iterator);
        }

        return new Element(loopElement);
    }

    private Iterator getChildren(SOAPElement element,  Name name) {
        return element.getChildElements(name);
    }

    private SOAPElement getNext(Iterator iterator) {
        if (iterator.hasNext())
            return (SOAPElement) iterator.next();
        else
            throw new SoapMessageException("There are no elements under the current element");
    }
}
