package com.agilexhealthcare.mobilehealthplatform.restservice.diet;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.fail;

import java.net.URI;
import java.util.UUID;

import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriBuilder;

import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.Test;
import org.springframework.http.HttpStatus;

import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.DietEntries;
import com.agilex.healthcare.mobilehealthplatform.domain.DietEntry;
import com.agilex.healthcare.mobilehealthplatform.domain.LinkTitles;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.MealType;
import com.agilex.healthcare.utility.ModeHelper;
import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.UniformInterfaceException;

public class DietResourceTest {
	private static final String VALID_VALUE = "10";
	private static final String INVALID_VALUE = "-1";

	private static MobileHealthClientTestVersion client;
	PatientIdentifier integrationPatientIdentifier = new PatientIdentifier("EDIPI", "INT-TEST-MOCK");

	private static MobileHealthClientTestVersion noDataClient;
	PatientIdentifier identifierNoData = new PatientIdentifier("EDIPI", "NO-DATA-MOCK");

	@BeforeClass
	public static void checkWhetherTestsAreRunningInCIHMode() {
		Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev", "dev"));

		client = TestHelper.createMobileHealthClient("zztest.patient75", "pass", "oauth");
		noDataClient = TestHelper.createMobileHealthClient("zztest.patient76", "pass", "oauth");
	}

	@AfterClass
	public static void tearDown() {
		if (client != null) {
			client.getJerseyClient().destroy();
			client = null;
		}
		if (noDataClient != null) {
			noDataClient.getJerseyClient().destroy();
			noDataClient = null;
		}
	}
	
	@Test
	public void fetchNoDataForInvalidPatientIdentifier() {
		URI mealsUri = TestHelper.getPatientDataUri(identifierNoData, LinkTitles.PatientDiet, "zztest.patient76");
		DietEntries mealEntries = getMeals(noDataClient.getJerseyClient(), mealsUri);
		assertEquals(0, mealEntries.size());
	}

	@Test
	public void addMealEntryAndRetrieve() {
		ensurePatientDoesNotAlreadyHaveMeals(client.getJerseyClient(), integrationPatientIdentifier);

		URI mealsUri = getPatientMealsUri(integrationPatientIdentifier);

		DietEntry savedMealEntry = createAndSaveMealEntry(integrationPatientIdentifier, "1/1/2012", "");

		DietEntries mealEntries = getMeals(client.getJerseyClient(), mealsUri);
		assertEquals(1, mealEntries.size());
		assertEquals(savedMealEntry.getId(), mealEntries.get(0).getId());
		
		cleanup(client.getJerseyClient(), savedMealEntry);
	}

	@Test
	public void addMealEntryAndRetrieveSingleDate() {

		ensurePatientDoesNotAlreadyHaveMeals(client.getJerseyClient(), integrationPatientIdentifier);

		createAndSaveMealEntry(integrationPatientIdentifier, "1/1/2012", "");
		createAndSaveMealEntry(integrationPatientIdentifier, "1/2/2012", "");
		createAndSaveMealEntry(integrationPatientIdentifier, "1/3/2012", "");

		URI baseMealsUri = getPatientMealsUri(integrationPatientIdentifier);
		URI mealsUri = UriBuilder.fromUri(baseMealsUri.toString()).queryParam("entryDate", "1/2/2012").build();
		DietEntries mealEntries = getMeals(client.getJerseyClient(), mealsUri);
		assertEquals(1, mealEntries.size());
		assertEquals(DateHelper.parseDate("1/2/2012"), mealEntries.get(0).getEntryDate());
		
		cleanup(client.getJerseyClient(), baseMealsUri);
	}

	@Test
	public void entryDateNotCaseSensitive() {
		PatientIdentifier patientId = integrationPatientIdentifier;

		ensurePatientDoesNotAlreadyHaveMeals(client.getJerseyClient(), patientId);

		createAndSaveMealEntry(patientId, "1/1/2012", "");
		createAndSaveMealEntry(patientId, "1/2/2012", "");
		createAndSaveMealEntry(patientId, "1/3/2012", "");

		URI baseMealsUri = getPatientMealsUri(patientId);
		URI mealsUri = UriBuilder.fromUri(baseMealsUri.toString()).queryParam("entrydate", "1/2/2012").build();
		DietEntries mealEntries = getMeals(client.getJerseyClient(), mealsUri);
		assertEquals(1, mealEntries.size());
		assertEquals(DateHelper.parseDate("1/2/2012"), mealEntries.get(0).getEntryDate());
		
		cleanup(client.getJerseyClient(), baseMealsUri);
	}

	@Test
	public void addAndModifyMeal() {
		PatientIdentifier patientId = integrationPatientIdentifier;

		ensurePatientDoesNotAlreadyHaveMeals(client.getJerseyClient(), patientId);

		URI mealsUri = getPatientMealsUri(patientId);

		DietEntry savedMealEntry = createAndSaveMealEntry(patientId, "1/1/2012", "");
		savedMealEntry.setEntryDate(DateHelper.parseDate("1/2/2012"));

		assertNotNull(savedMealEntry.getSelfUri());
		DietEntry modifiedMealEntry = client.getJerseyClient().resource(savedMealEntry.getSelfUri()).type(MediaType.APPLICATION_XML).accept("application/xml").put(DietEntry.class, savedMealEntry);

		DietEntries mealEntries = getMeals(client.getJerseyClient(), mealsUri);
		assertEquals(1, mealEntries.size());

		assertEquals(savedMealEntry.getSelfUri(), modifiedMealEntry.getSelfUri());
		assertEquals(modifiedMealEntry.getSelfUri(), mealEntries.get(0).getSelfUri());
		
		cleanup(client.getJerseyClient(), savedMealEntry);
	}

	@Test
	public void retrieveMeals() {
		PatientIdentifier patientId = integrationPatientIdentifier;

		ensurePatientDoesNotAlreadyHaveMeals(client.getJerseyClient(), patientId);

		createAndSaveMealEntry(patientId, "1/1/2012", "");
		createAndSaveMealEntry(patientId, "1/2/2012", "");
		createAndSaveMealEntry(patientId, "1/3/2012", "");
		createAndSaveMealEntry(patientId, "1/4/2012", "");
		createAndSaveMealEntry(patientId, "1/5/2012", "");
		createAndSaveMealEntry(patientId, "1/6/2012", "");

		URI mealsUri = getPatientMealsUri(patientId);
		mealsUri = UriBuilder.fromUri(mealsUri.toString()).build();
		DietEntries mealEntries = getMeals(client.getJerseyClient(), mealsUri);
		assertEquals(6, mealEntries.size());
		
		cleanup(client.getJerseyClient(), mealEntries);
	}

	@Test
	public void retrieveByDate() {
		PatientIdentifier patientId = integrationPatientIdentifier;

		ensurePatientDoesNotAlreadyHaveMeals(client.getJerseyClient(), patientId);

		createAndSaveMealEntry(patientId, "1/1/2012", "");
		createAndSaveMealEntry(patientId, "1/2/2012", "");
		createAndSaveMealEntry(patientId, "1/3/2012", "");
		createAndSaveMealEntry(patientId, "1/4/2012", "");
		createAndSaveMealEntry(patientId, "1/5/2012", "");
		createAndSaveMealEntry(patientId, "1/6/2012", "");

		URI baseMealsUri = getPatientMealsUri(patientId);
		URI mealsUri = UriBuilder.fromUri(baseMealsUri.toString()).queryParam("startDate", "1/2/2012").queryParam("endDate", "1/4/2012").build();
		DietEntries mealEntries = getMeals(client.getJerseyClient(), mealsUri);
		assertEquals(3, mealEntries.size());
		
		cleanup(client.getJerseyClient(), baseMealsUri);
	}

	@Test
	public void testLargeNote() {
		PatientIdentifier patientId = integrationPatientIdentifier;
		
		String note = "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa" + "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
				+ "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa";

		DietEntry shouldNotFail = createAndSaveMealEntry(patientId, "1/1/2012", note);
		assertEquals(note, shouldNotFail.getNotes());
		
		cleanup(client.getJerseyClient(), shouldNotFail);
	}
	
	@Test
	public void addDeleteTest() {
		PatientIdentifier patientId = integrationPatientIdentifier;
		ensurePatientDoesNotAlreadyHaveMeals(client.getJerseyClient(), patientId);

		URI mealsUri = getPatientMealsUri(patientId);

		DietEntry savedMealEntry = createAndSaveMealEntry(patientId, "1/1/2012", "");
		savedMealEntry.setEntryDate(DateHelper.parseDate("1/2/2012"));
		savedMealEntry.setNotes("Updated Meal");

		client.getJerseyClient().resource(savedMealEntry.getSelfUri()).type(MediaType.APPLICATION_XML).accept("application/xml").put(DietEntries.class, savedMealEntry);
		DietEntries mealEntries = getMeals(client.getJerseyClient(), mealsUri);
		assertEquals(1, mealEntries.size());
		assertEquals("Updated Meal", mealEntries.get(0).getNotes());

		client.getJerseyClient().resource(savedMealEntry.getSelfUri()).delete();
		mealEntries = getMeals(client.getJerseyClient(), mealsUri);
		assertEquals(0, mealEntries.size());
	}

	@Test
	public void mealFailValidation() {
		PatientIdentifier patientId = integrationPatientIdentifier;
		try {
			createAndSaveInvalidMealEntry(patientId, "1/1/2012");
			fail("expected validation exception");
		} catch (UniformInterfaceException e) {
			assertEquals(HttpStatus.BAD_REQUEST.value(), e.getResponse().getStatus());
		}
	}

	private DietEntry createAndSaveMealEntry(PatientIdentifier patientIdentifier, String entryDate, String notes) {
		DietEntry mealEntry = createValidMealEntry(patientIdentifier, entryDate, notes);
		URI mealsUri = getPatientMealsUri(patientIdentifier);
		DietEntry savedMealEntry = client.getJerseyClient().resource(mealsUri).type(MediaType.APPLICATION_XML).accept("application/xml").post(DietEntry.class, mealEntry);
		return savedMealEntry;

	}

	private DietEntry createValidMealEntry(PatientIdentifier patientIdentifier, String entryDate, String notes) {
		DietEntry mealEntry = new DietEntry();
		mealEntry.setEntryDate(DateHelper.parseDate(entryDate));
		mealEntry.setPatientIdentifier(patientIdentifier);
		mealEntry.setCalories(VALID_VALUE);
		mealEntry.setFat(VALID_VALUE);
		mealEntry.setCarbs(VALID_VALUE);
		mealEntry.setProtein(VALID_VALUE);
		mealEntry.setMealType(MealType.LUNCH);
		mealEntry.setNotes(notes);

		return mealEntry;
	}

	private DietEntry createAndSaveInvalidMealEntry(PatientIdentifier patientIdentifier, String entryDate) {
		DietEntry mealEntry = createInvalidMealEntry(patientIdentifier, entryDate);
		URI mealsUri = getPatientMealsUri(patientIdentifier);
		DietEntry savedMealEntry = client.getJerseyClient().resource(mealsUri).type(MediaType.APPLICATION_XML).accept("application/xml").post(DietEntry.class, mealEntry);
		return savedMealEntry;
	}

	private DietEntry createInvalidMealEntry(PatientIdentifier patientIdentifier, String entryDate) {
		DietEntry mealEntry = new DietEntry();
		mealEntry.setId(UUID.randomUUID().toString());
		mealEntry.setEntryDate(DateHelper.parseDate(entryDate));
		mealEntry.setPatientIdentifier(patientIdentifier);
		mealEntry.setCalories(INVALID_VALUE);
		mealEntry.setFat(INVALID_VALUE);
		mealEntry.setCarbs(INVALID_VALUE);
		mealEntry.setProtein(INVALID_VALUE);

		return mealEntry;
	}

	private URI getPatientMealsUri(PatientIdentifier patientIdentifier) {
		return TestHelper.getPatientDataUri(patientIdentifier, LinkTitles.PatientDiet, "zztest.patient75");
	}

	private void ensurePatientDoesNotAlreadyHaveMeals(Client jerseyClient, PatientIdentifier patientIdentifier) {
		URI mealsUri = getPatientMealsUri(patientIdentifier);
		DietEntries mealEntries = getMeals(jerseyClient, mealsUri);
		assertEquals(0, mealEntries.size());
	}

	private DietEntries getMeals(Client jerseyClient, URI mealsUri) {
		DietEntries mealsEntries = jerseyClient.resource(mealsUri).get(DietEntries.class);
		return mealsEntries;
	}
	
	public void cleanup(Client jerseyClient, URI uri) {
		DietEntries meals = getMeals(jerseyClient, uri);
		cleanup(jerseyClient, meals);
	}
	
	public void cleanup(Client jerseyClient, DietEntry dietEntry){
		jerseyClient.resource(dietEntry.getSelfUri()).delete();
	}
	
	public void cleanup(Client jerseyClient, DietEntries dietEntries) {
		for(DietEntry entry : dietEntries) {
			cleanup(jerseyClient, entry);
		}
	}

}
