package com.agilexhealthcare.mobilehealthplatform.restservice.dailyevent;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.fail;

import java.net.URI;

import javax.ws.rs.core.UriBuilder;

import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.Test;
import org.springframework.http.HttpStatus;

import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.DailyEvent;
import com.agilex.healthcare.mobilehealthplatform.domain.DailyEvents;
import com.agilex.healthcare.mobilehealthplatform.domain.LinkTitles;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.ModeHelper;
import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.UniformInterfaceException;

public class DailyEventResourceTest {
	private static final String INTEGRATIONTEST_USER = "zztest.patient75";
	private static MobileHealthClientTestVersion client;
	PatientIdentifier integrationPatientIdentifier = new PatientIdentifier("EDIPI", "INT-TEST-MOCK");

	private static MobileHealthClientTestVersion noDataClient;
	PatientIdentifier identifierNoData = new PatientIdentifier("EDIPI", "NO-DATA-MOCK");

	@BeforeClass
	public static void checkWhetherTestsAreRunningInCIHMode() {
		Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev", "dev"));
		client = TestHelper.createMobileHealthClient(INTEGRATIONTEST_USER, "pass", "oauth");
		noDataClient = TestHelper.createMobileHealthClient("zztest.patient76", "pass", "oauth");
	}
	
	@AfterClass
	public static void destroy() {
		if (client != null) {
			client.getJerseyClient().destroy();
			client = null;
		}
		if (noDataClient != null) {
			noDataClient.getJerseyClient().destroy();
			noDataClient = null;
		}
	}

	@Test
	public void retrieve0() {
		PatientIdentifier patientIdentifier = identifierNoData;
		ensurePatientDoesNotAlreadyHaveDailyEvents(noDataClient.getJerseyClient(), patientIdentifier, "zztest.patient76");

		URI eventsURI = getPatientDailyEventsUri(patientIdentifier, "zztest.patient76");
		DailyEvents dailyEvents = getDailyEvents(noDataClient.getJerseyClient(), eventsURI);
		assertEquals(0, dailyEvents.size());
	}

	@Test
	public void addDailyEventAndRetrieve() {
		PatientIdentifier patientIdentifier = integrationPatientIdentifier;

		ensurePatientDoesNotAlreadyHaveDailyEvents(client.getJerseyClient(), patientIdentifier, INTEGRATIONTEST_USER);

		URI eventsUri = getPatientDailyEventsUri(patientIdentifier, INTEGRATIONTEST_USER);

		DailyEvent savedEvent = createAndSaveDailyEvent(patientIdentifier, "1/1/2012", INTEGRATIONTEST_USER);

		DailyEvents dailyEvents = getDailyEvents(client.getJerseyClient(), eventsUri);
		assertEquals(1, dailyEvents.size());
		assertEquals(savedEvent.getId(), dailyEvents.get(0).getId());
		
		cleanup(client.getJerseyClient(), dailyEvents);
	}

	@Test
	public void addDailyEventAndRetrieveSingleDate() {
		PatientIdentifier patientIdentifier = integrationPatientIdentifier;

		ensurePatientDoesNotAlreadyHaveDailyEvents(client.getJerseyClient(), patientIdentifier, INTEGRATIONTEST_USER);

		createAndSaveDailyEvent(patientIdentifier, "1/1/2012", INTEGRATIONTEST_USER);
		createAndSaveDailyEvent(patientIdentifier, "1/2/2012", INTEGRATIONTEST_USER);
		createAndSaveDailyEvent(patientIdentifier, "1/3/2012", INTEGRATIONTEST_USER);

		URI baseEventUri = getPatientDailyEventsUri(patientIdentifier, INTEGRATIONTEST_USER);
		URI eventUri = UriBuilder.fromUri(baseEventUri.toString()).queryParam("entryDate", "1/2/2012").build();
		DailyEvents dailyEvents = getDailyEvents(client.getJerseyClient(), eventUri);
		assertEquals(1, dailyEvents.size());
		assertEquals(DateHelper.parseDate("1/2/2012"), dailyEvents.get(0).getEntryDate());
		
		cleanup(client.getJerseyClient(), baseEventUri);
	}

	@Test
	public void entryDateNotCaseSensitive() {
		PatientIdentifier patientIdentifier = integrationPatientIdentifier;

		ensurePatientDoesNotAlreadyHaveDailyEvents(client.getJerseyClient(), patientIdentifier, INTEGRATIONTEST_USER);

		createAndSaveDailyEvent(patientIdentifier, "1/1/2012", INTEGRATIONTEST_USER);
		createAndSaveDailyEvent(patientIdentifier, "1/2/2012", INTEGRATIONTEST_USER);
		createAndSaveDailyEvent(patientIdentifier, "1/3/2012", INTEGRATIONTEST_USER);

		URI baseEventsUri = getPatientDailyEventsUri(patientIdentifier, INTEGRATIONTEST_USER);
		URI eventsUri = UriBuilder.fromUri(baseEventsUri.toString()).queryParam("entryDate", "1/2/2012").build();
		DailyEvents dailyEvents = getDailyEvents(client.getJerseyClient(), eventsUri);
		assertEquals(1, dailyEvents.size());
		assertEquals(DateHelper.parseDate("1/2/2012"), dailyEvents.get(0).getEntryDate());
		
		cleanup(client.getJerseyClient(), baseEventsUri);
	}

	@Test
	public void addAndModifyDailyEvent() {
		PatientIdentifier patientIdentifier = integrationPatientIdentifier;

		ensurePatientDoesNotAlreadyHaveDailyEvents(client.getJerseyClient(), patientIdentifier, INTEGRATIONTEST_USER);

		URI dailyEventsUri = getPatientDailyEventsUri(patientIdentifier, INTEGRATIONTEST_USER);

		DailyEvent savedDailyEvent = createAndSaveDailyEvent(patientIdentifier, "10/1/2011", INTEGRATIONTEST_USER);
		savedDailyEvent.setEntryDate(DateHelper.parseDate("10/2/2011"));
		savedDailyEvent.setNotes("some test notes");

		DailyEvent modifiedDailyEvent = client.getJerseyClient().resource(savedDailyEvent.getSelfUri()).put(DailyEvent.class, savedDailyEvent);

		assertEquals(modifiedDailyEvent.getId(), savedDailyEvent.getId());

		DailyEvents dailyEvents = getDailyEvents(client.getJerseyClient(), dailyEventsUri);
		assertEquals(1, dailyEvents.size());
		assertEquals(savedDailyEvent.getPatientId(), modifiedDailyEvent.getPatientId());
		assertEquals(savedDailyEvent.getSelfUri(), modifiedDailyEvent.getSelfUri());
		assertEquals(modifiedDailyEvent.getSelfUri(), dailyEvents.get(0).getSelfUri());
		
		cleanup(client.getJerseyClient(), dailyEvents);
	}

	@Test
	public void verifySaveLongNotes() {
		PatientIdentifier patientIdentifier = integrationPatientIdentifier;

		DailyEvent dailyEvent = new DailyEvent();
		dailyEvent.setEntryDate(DateHelper.parseDate("1/1/2012"));
		dailyEvent.setPatientIdentifier(patientIdentifier);
		dailyEvent.setNotes(stringOfLength(100));
		dailyEvent.setTitle("good title");

		URI dailyEventsUri = getPatientDailyEventsUri(patientIdentifier, INTEGRATIONTEST_USER);
		DailyEvent savedDailyEvent = client.getJerseyClient().resource(dailyEventsUri).post(DailyEvent.class, dailyEvent);
		assertNotNull(savedDailyEvent);
		assertEquals(dailyEvent.getNotes(), savedDailyEvent.getNotes());
		
		cleanup(client.getJerseyClient(), savedDailyEvent);
	}

	@Test(expected = UniformInterfaceException.class)
	public void failOnLongTitle() {
		PatientIdentifier patientIdentifier = integrationPatientIdentifier;

		DailyEvent dailyEvent = new DailyEvent();
		dailyEvent.setEntryDate(DateHelper.parseDate("1/1/2012"));
		dailyEvent.setPatientIdentifier(patientIdentifier);
		dailyEvent.setNotes(stringOfLength(100));
		dailyEvent.setTitle(stringOfLength(60));

		URI dailyEventsUri = getPatientDailyEventsUri(patientIdentifier, INTEGRATIONTEST_USER);
		client.getJerseyClient().resource(dailyEventsUri).post(DailyEvent.class, dailyEvent);
		fail("Should have errored prior to this point");
	}

	@Test
	public void retrieveDailyEvents() {
		PatientIdentifier patientIdentifier = integrationPatientIdentifier;

		ensurePatientDoesNotAlreadyHaveDailyEvents(client.getJerseyClient(), patientIdentifier, INTEGRATIONTEST_USER);

		createAndSaveDailyEvent(patientIdentifier, "1/1/2012", INTEGRATIONTEST_USER);
		createAndSaveDailyEvent(patientIdentifier, "1/2/2012", INTEGRATIONTEST_USER);
		createAndSaveDailyEvent(patientIdentifier, "1/3/2012", INTEGRATIONTEST_USER);
		createAndSaveDailyEvent(patientIdentifier, "1/4/2012", INTEGRATIONTEST_USER);
		createAndSaveDailyEvent(patientIdentifier, "1/5/2012", INTEGRATIONTEST_USER);
		createAndSaveDailyEvent(patientIdentifier, "1/6/2012", INTEGRATIONTEST_USER);

		URI dailyEventsUri = getPatientDailyEventsUri(patientIdentifier, INTEGRATIONTEST_USER);
		dailyEventsUri = UriBuilder.fromUri(dailyEventsUri.toString()).build();
		DailyEvents dailyEvents = getDailyEvents(client.getJerseyClient(), dailyEventsUri);
		assertEquals(6, dailyEvents.size());
		
		cleanup(client.getJerseyClient(), dailyEvents);
	}

	@Test
	public void retrieveByDate() {
		PatientIdentifier patientIdentifier = integrationPatientIdentifier;

		ensurePatientDoesNotAlreadyHaveDailyEvents(client.getJerseyClient(), patientIdentifier, INTEGRATIONTEST_USER);

		createAndSaveDailyEvent(patientIdentifier, "1/1/2012", INTEGRATIONTEST_USER);
		createAndSaveDailyEvent(patientIdentifier, "1/2/2012", INTEGRATIONTEST_USER);
		createAndSaveDailyEvent(patientIdentifier, "1/3/2012", INTEGRATIONTEST_USER);
		createAndSaveDailyEvent(patientIdentifier, "1/4/2012", INTEGRATIONTEST_USER);
		createAndSaveDailyEvent(patientIdentifier, "1/5/2012", INTEGRATIONTEST_USER);
		createAndSaveDailyEvent(patientIdentifier, "1/6/2012", INTEGRATIONTEST_USER);

		URI baseDailyEventsUri = getPatientDailyEventsUri(patientIdentifier, INTEGRATIONTEST_USER);
		URI dailyEventsUri = UriBuilder.fromUri(baseDailyEventsUri.toString()).queryParam("startDate", "1/2/2012").queryParam("endDate", "1/4/2012").build();
		DailyEvents dailyEvents = getDailyEvents(client.getJerseyClient(), dailyEventsUri);
		assertEquals(3, dailyEvents.size());
		
		cleanup(client.getJerseyClient(), baseDailyEventsUri);
	}

	private DailyEvent createAndSaveDailyEvent(PatientIdentifier patientIdentifier, String entryDate, String userName) {
		DailyEvent dailyEvent = createValidDailyEvent(patientIdentifier, entryDate);
		URI dailyEventsUri = getPatientDailyEventsUri(patientIdentifier, userName);
		DailyEvent savedDailyEvent = client.getJerseyClient().resource(dailyEventsUri).post(DailyEvent.class, dailyEvent);
		return savedDailyEvent;

	}

	private DailyEvent createValidDailyEvent(PatientIdentifier patientIdentifier, String entryDate) {
		DailyEvent dailyEvent = new DailyEvent();
		dailyEvent.setEntryDate(DateHelper.parseDate(entryDate));
		dailyEvent.setPatientIdentifier(patientIdentifier);
		dailyEvent.setNotes("these are good notes");
		dailyEvent.setTitle("good title");

		return dailyEvent;
	}

	private DailyEvent createAndSaveInvalidDailyEvent(PatientIdentifier patientIdentifier, String entryDate) {
		DailyEvent dailyEventEntry = createInvalidDailyEvent(patientIdentifier, entryDate);
		URI dailyEventsUri = getPatientDailyEventsUri(patientIdentifier, INTEGRATIONTEST_USER);
		DailyEvent savedDailyEvent = client.getJerseyClient().resource(dailyEventsUri).post(DailyEvent.class, dailyEventEntry);
		return savedDailyEvent;

	}

	private DailyEvent createInvalidDailyEvent(PatientIdentifier patientIdentifier, String entryDate) {
		DailyEvent dailyEvent = new DailyEvent();
		dailyEvent.setEntryDate(DateHelper.parseDate(entryDate));
		dailyEvent.setPatientIdentifier(patientIdentifier);
		dailyEvent.setNotes("");

		return dailyEvent;
	}

	private URI getPatientDailyEventsUri(PatientIdentifier patientIdentifier, String userName) {
		return TestHelper.getPatientDataUri(patientIdentifier, LinkTitles.DailyEvents, userName);
	}

	private void ensurePatientDoesNotAlreadyHaveDailyEvents(Client jerseyClient, PatientIdentifier patientIdentifier, String userName) {
		URI dailyEventsUri = getPatientDailyEventsUri(patientIdentifier, userName);
		DailyEvents dailyEvents = getDailyEvents(jerseyClient, dailyEventsUri);
		assertEquals(0, dailyEvents.size());
	}

	private DailyEvents getDailyEvents(Client jerseyClient, URI dailyEventsUri) {
		DailyEvents dailyEvents = jerseyClient.resource(dailyEventsUri).get(DailyEvents.class);
		return dailyEvents;
	}

	@Test
	public void addDeleteTest() {
		PatientIdentifier patientIdentifier = integrationPatientIdentifier;

		ensurePatientDoesNotAlreadyHaveDailyEvents(client.getJerseyClient(), patientIdentifier, INTEGRATIONTEST_USER);

		URI dailyEventsUri = getPatientDailyEventsUri(patientIdentifier, INTEGRATIONTEST_USER);

		DailyEvents dailyEvents = getDailyEvents(client.getJerseyClient(), dailyEventsUri);
		assertEquals(0, dailyEvents.size());

		DailyEvent savedDailyEvent = createAndSaveDailyEvent(patientIdentifier, "1/1/2012", INTEGRATIONTEST_USER);

		dailyEvents = getDailyEvents(client.getJerseyClient(), dailyEventsUri);
		assertEquals(1, dailyEvents.size());

		client.getJerseyClient().resource(savedDailyEvent.getSelfUri()).delete();
		dailyEvents = getDailyEvents(client.getJerseyClient(), dailyEventsUri);
		assertEquals(0, dailyEvents.size());
	}

	@Test
	public void dailyEventFailValidation() {
		PatientIdentifier patientIdentifier = integrationPatientIdentifier;

		try {
			createAndSaveInvalidDailyEvent(patientIdentifier, "1/1/2012");
			fail("expected validation exception");
		} catch (UniformInterfaceException e) {
			assertEquals(HttpStatus.BAD_REQUEST.value(), e.getResponse().getStatus());
		}
	}

	private String stringOfLength(int length) {
		StringBuilder sb = new StringBuilder();
		for (int i = 0; i < length; i++)
			sb.append("x");
		return sb.toString();
	}
	
	public void cleanup(Client jerseyClient, URI uri) {
		DailyEvents dailyEvents = jerseyClient.resource(uri).get(DailyEvents.class);
		cleanup(jerseyClient, dailyEvents);
	}
	
	public void cleanup(Client jerseyClient, DailyEvent dailyEvent){
		jerseyClient.resource(dailyEvent.getSelfUri()).delete();
	}
	
	public void cleanup(Client jerseyClient, DailyEvents dailyEvents) {
		for(DailyEvent entry : dailyEvents) {
			cleanup(jerseyClient, entry);
		}
	}

}
