package com.agilexhealth.mobilehealthplatform.restservice.healthadvocaterequest;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.net.URI;

import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.HealthAdvocateRequest;
import com.agilex.healthcare.mobilehealthplatform.domain.HealthAdvocateRequests;
import com.agilex.healthcare.mobilehealthplatform.domain.HealthAdvocateStatusCode;
import com.agilex.healthcare.mobilehealthplatform.domain.LinkTitles;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.ModeHelper;

public class HealthAdvocateRequestResourceTest {

	
	String defaultIcn = "D123401";
	PatientIdentifier patientIdentifier = new PatientIdentifier("EDIPI", defaultIcn);

	private static MobileHealthClientTestVersion client;
	
	@BeforeClass
	public static void checkWhetherTestsAreRunningInCIHMode() {
		Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev", "dev"));
		client = TestHelper.createMobileHealthClient();
	}

	@AfterClass
	public static void destroy() {
		if (client != null) {
			client.getJerseyClient().destroy();
			client = null;
		}
	}
	
	@Before
	public void before() {
	}

	@Test
	public void postNewHAR() {
		HealthAdvocateRequest newRequest = getSampleNewRequest();
		URI healthAdvocateRequestsURI = this.generateHealthAdvocateRequestsURI();

		HealthAdvocateRequest healthAdvocateRequest = client.getJerseyClient().resource(healthAdvocateRequestsURI).post(HealthAdvocateRequest.class, newRequest);
		assertNotNull( healthAdvocateRequest);
		
		assertNotNull(healthAdvocateRequest.getUniqueId());
		assertNotNull(healthAdvocateRequest.getHealthAdvocateEDIPI());
		assertNotNull(healthAdvocateRequest.getUriLinkKey());
		assertTrue(DateHelper.calculateDeltaInDays(DateHelper.getToday(), healthAdvocateRequest.getLastActionDate()) < 1);
		assertTrue(DateHelper.calculateDeltaInDays(DateHelper.getToday(), healthAdvocateRequest.getRequestDate()) < 1);
		assertNull(healthAdvocateRequest.getStatus());
		assertEquals("Tim", healthAdvocateRequest.getFirstName());
		
	}

	@Test
	public void testPut() {
		HealthAdvocateRequest newRequest = getSampleNewRequest();
		
		URI healthAdvocateRequestsURI = this.generateHealthAdvocateRequestsURI();
		HealthAdvocateRequest createdHealthAdvocateRequest = client.getJerseyClient().resource(healthAdvocateRequestsURI).post(HealthAdvocateRequest.class, newRequest);
		assertNotNull( createdHealthAdvocateRequest);	
		
		createdHealthAdvocateRequest.setStatus(HealthAdvocateStatusCode.SUBMITTED);
		
		URI putUri = createdHealthAdvocateRequest.getSelfUri();
		
		HealthAdvocateRequest updatedhealthAdvocateRequest = client.getJerseyClient().resource(putUri).put(HealthAdvocateRequest.class, createdHealthAdvocateRequest);
		assertNotNull( updatedhealthAdvocateRequest);	
		assertNotNull(updatedhealthAdvocateRequest.getLastActionDate());

		assertEquals(HealthAdvocateStatusCode.SUBMITTED, updatedhealthAdvocateRequest.getStatus());
	}
	
	@Test
	public void verifyFetchHAR() {
		
		HealthAdvocateRequest newRequest = getSampleNewRequest();
		URI healthAdvocateRequestsURI = this.generateHealthAdvocateRequestsURI();

		HealthAdvocateRequest healthAdvocateRequest = client.getJerseyClient().resource(healthAdvocateRequestsURI).post(HealthAdvocateRequest.class, newRequest);
		assertNotNull( healthAdvocateRequest);
		
		HealthAdvocateRequests healthAdvocateRequests = this.getHAR();
		assertNotNull(healthAdvocateRequests);
		
		assertTrue(checkStatusSorting(healthAdvocateRequests));
	}
	
	private HealthAdvocateRequests getHAR() {

		URI healthAdvocateRequestsURI = this.generateHealthAdvocateRequestsURI();
		HealthAdvocateRequests healthAdvocateRequests = client.getJerseyClient().resource(healthAdvocateRequestsURI).get(HealthAdvocateRequests.class);
	
		return healthAdvocateRequests;
	}

	private URI generateHealthAdvocateRequestsURI() {
		return TestHelper.getPatientDataUri(patientIdentifier, LinkTitles.HealthAdvocateRequest, "zztest.patient01");
	}
	
	private HealthAdvocateRequest getSampleNewRequest() {
		HealthAdvocateRequest r = new HealthAdvocateRequest();
		r.setAddressFirstLine("33 House Lane");
		r.setAddressSecondLine("Apt B");
		r.setCity("Chantilly");
		r.setEmail("testEmail6@testEmail.com");
		r.setFirstName("Tim");
		r.setLastName("Chapin");
		r.setPatientId("D123401");
		r.setState("VA");
		r.setZipCode("22222");	
		r.setPatientName("Patient Test");
		
		return r;
	}
	
	private boolean checkStatusSorting(HealthAdvocateRequests requests) {
		if (requests.getSize() >= 2) {
			for (int i=1; i < requests.getSize(); i++) {
				if (areSorted(requests.get(i), requests.get(i-1)) == false) {
					return false;
				}
			}
		}
		
		return true;
	}
	
	private boolean areSorted(HealthAdvocateRequest next, HealthAdvocateRequest previous) {
		boolean check = true;
		
		if (translateStatusOrdering(next.getStatus()) < translateStatusOrdering(previous.getStatus())) {
			check = false;
		}
	
		if (translateStatusOrdering(next.getStatus()) == translateStatusOrdering(previous.getStatus()) && next.getLastActionDate().compareTo(previous.getLastActionDate()) > 0 ) {
			check = false;
		}
		
		return check;
	}
	
	private Integer translateStatusOrdering (String status) {
		Integer statusOrder = 6;
		
		if (HealthAdvocateStatusCode.SUBMITTED.equalsIgnoreCase(status)) {
			statusOrder = 1;
		} else if (HealthAdvocateStatusCode.ACCEPTED.equalsIgnoreCase(status)) {
			statusOrder = 2;
		} else if (HealthAdvocateStatusCode.DECLINED.equalsIgnoreCase(status)) {
			statusOrder = 3;
		} else if (HealthAdvocateStatusCode.CANCELLED.equalsIgnoreCase(status)) {
			statusOrder = 4;
		} else if (HealthAdvocateStatusCode.REVOKED.equalsIgnoreCase(status)) {
			statusOrder = 5;
		}
		
		return statusOrder;
	}
}
;
