package com.agilex.healthcare.testutility;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.net.URI;
import java.util.Date;

import javax.ws.rs.core.UriBuilder;

import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.agilex.healthcare.mobilehealthplatform.client.AuthenticationInfo;
import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.XpathHelper;

public class PatientDataResourceFieldTestRunner {
	private MobileHealthClientTestVersion client = TestHelper.createMobileHealthClient("zztest.patient01", "pass", "oauth");
	private String xpathOnPatientsToDataLink = null;
	private String dataElementPatientId = null;
	private Node retrievedData = null;
	private String xpathOnRetrieveToDataList;
	private String xpathOnDataListToTestDataElement;

	public PatientDataResourceFieldTestRunner(String patientId, String xpathOnPatientsToDataLink, String xpathOnRetrieveToDataList, String xpathOnDataListToTestDataElement) {
		System.out.println("initializing test runner");
		this.dataElementPatientId = patientId;
		this.xpathOnPatientsToDataLink = xpathOnPatientsToDataLink;
		this.xpathOnRetrieveToDataList = xpathOnRetrieveToDataList;
		this.xpathOnDataListToTestDataElement = xpathOnDataListToTestDataElement;
	}

	private Node getPatient(String userName) {
		return getPatient(this.dataElementPatientId, userName);
	}

	private Node getPatient(String patientId, String userName) {
		URI patientUri = UriBuilder.fromPath(IntegrationTestConfiguration.getMobileHealthServerBaseUri()).path("patient/EDIPI/").path(patientId).build();
		MobileHealthClientTestVersion restClient = TestHelper.createMobileHealthClient(userName, "pass", "oauth");
		Document patient = restClient.getAsXml(patientUri);
		assertNotNull("failed to retrieve patient [id=" + patientId + "]", patientId);
		return patient;
	}

	private String getXpathOnPatientsToDataUri() {
		return this.xpathOnPatientsToDataLink + "/@href";
	}

	private String getXpathOnRetrieveDataToTestDataElement() {
		return xpathOnRetrieveToDataList + xpathOnDataListToTestDataElement;
	}

	private String getXpathOnRetrieveToDataList() {
		return xpathOnRetrieveToDataList;
	}

	public Node retrieveData() {
		Node patient = getPatient("zztest.patient01");
		String dataUri = XpathHelper.getString(patient, this.getXpathOnPatientsToDataUri());
		NullCheckerAssert.assertNotNullish("failed to determine data uri from patient [" + this.getXpathOnPatientsToDataUri() + "]", dataUri);
		Document retrievedDatas = client.getAsXml(dataUri);
		Node retrievedData = XpathHelper.getNode(retrievedDatas, this.getXpathOnRetrieveDataToTestDataElement());
		return retrievedData;
	}

	public NodeList retrieveDataList() {
		String dataUri = XpathHelper.getString(getPatient("zztest.patient01"), this.getXpathOnPatientsToDataUri());
		Document retrievedDatas = client.getAsXml(dataUri);
		NodeList retrievedDataList = XpathHelper.getNodeList(retrievedDatas, this.getXpathOnRetrieveToDataList());
		return retrievedDataList;
	}

	private Node getRetrievedData() {
		if (this.retrievedData == null) {
			this.retrievedData = retrieveData();
		}
		return this.retrievedData;
	}

	public void validateFieldTest(Date expectedValue, String expectedDateFormat, String xPathToField) {
		String expectedDateAsText = null;
		if (expectedValue != null)
			expectedDateAsText = DateHelper.format(expectedValue, expectedDateFormat);
		validateFieldTest(expectedDateAsText, xPathToField);
	}

	public void validateFieldTest(boolean expectedValue, String xPathToField) {
		validateFieldTest(Boolean.toString(expectedValue), xPathToField);
	}

	public void validateFieldTest(String expectedValue, String xPathToField) {
		Node retrievedData = getRetrievedData();
		assertNotNull("Failed to retrieve data", retrievedData);
		String retrievedValue = XpathHelper.getString(retrievedData, xPathToField);

		assertEqualIgnoreNull(expectedValue, retrievedValue);
	}

	private void assertEqualIgnoreNull(String expected, String actual) {
		if (NullChecker.isNullish(expected))
			expected = null;
		if (NullChecker.isNullish(actual))
			actual = null;
		assertEquals(expected, actual);
	}

	public void validate x c C u tTest(int expectedCount) {
		assertEquals(expectedCount, retrieveDataList().getLength());
	}

	public void validateAtleastCountTest(int atLeastCount) {
		assertTrue(retrieveDataList().getLength() >= atLeastCount);
	}

	public void noDataReturnedTest() {

		String userName = "zztest.patient76";
		AuthenticationInfo authenticationInfo = new AuthenticationInfo(userName, "pass", "oauth");
		MobileHealthClientTestVersion noDataClient = TestHelper.createMobileHealthClient(authenticationInfo);
		
		PatientIdentifier patientIdentifier = new PatientIdentifier("EDIPI", "NO-DATA-MOCK");
		Patient patient = PatientLoader.loadPatient(patientIdentifier, userName);
		String healthyPatientId = patient.getId();
		String dataUri = XpathHelper.getString(getPatient(healthyPatientId, userName), this.getXpathOnPatientsToDataUri());
		Document retrievedDatas = noDataClient.getAsXml(dataUri);
		NodeList retrievedDataList = XpathHelper.getNodeList(retrievedDatas, this.getXpathOnRetrieveToDataList());
		assertEquals(0, retrievedDataList.getLength());
		
		noDataClient.getJerseyClient().destroy();
	}

}
