package com.agilex.healthcare.mobilehealthplatform.restservice.vitals;

import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.net.URI;

import javax.ws.rs.core.UriBuilder;

import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.GraphData;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalEntries;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalTableData;
import com.agilex.healthcare.mobilehealthplatform.domain.code.VitalSectionCode;
import com.agilex.healthcare.testutility.PatientLoader;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.ModeHelper;

public class VitalsResourceTest {

	private static Patient testPatient;


	private static MobileHealthClientTestVersion client;
	
	@BeforeClass
	public static void checkWhetherTestsAreRunningInCIHMode() {
		Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev"));
		testPatient = PatientLoader.loadTestPatient();
		client = TestHelper.createMobileHealthClient();
	}

	@AfterClass
	public static void destroy() {
		if (client != null) {
			client.getJerseyClient().destroy();
			client = null;
		}
	}
	
	@Test
	public void verifyFetchManyByBloodPressureSection() {
		VitalEntries tempVitals = fetchManyBySection(testPatient.getLongitudinalVitalsUri(), VitalSectionCode.BloodPressure);
		assertTrue("Should have 1 or more bp entries - longitudinal", tempVitals.size() > 0);
	}
	
	@Test
	public void verifyFetchManyByTemperatureSection() {
		VitalEntries tempVitals = fetchManyBySection(testPatient.getLongitudinalVitalsUri(), VitalSectionCode.Temperature);
		assertTrue("Should have 1 or more temp entries - longitudinal", tempVitals.size() > 0);
	}
	
	@Test
	public void verifyFetchManyByPulseOxSection() {
		VitalEntries pulseOxVitals = fetchManyBySection(testPatient.getLongitudinalVitalsUri(), VitalSectionCode.PulseOximetry);
		assertTrue("Should have 1 or more temp entries - longitudinal", pulseOxVitals.size() > 0);
	}
	
	@Test
	public void verifyFetchManyByHeightSection() {
		VitalEntries pulseOxVitals = fetchManyBySection(testPatient.getLongitudinalVitalsUri(), VitalSectionCode.Height);
		assertTrue("Should have 1 or more height entries - longitudinal", pulseOxVitals.size() > 0);
	}
	
	@Test
	public void verifyFetchManyByWeightSection() {
		VitalEntries weightVitals = fetchManyBySection(testPatient.getLongitudinalVitalsUri(), VitalSectionCode.Weight);
		assertTrue("Should have 1 or more weight entries - longitudinal", weightVitals.size() > 0);
	}
	
	@Test
	public void verifyFetchManyByRespirationSection() {
		VitalEntries respVitals = fetchManyBySection(testPatient.getLongitudinalVitalsUri(), VitalSectionCode.Respiration);
		assertTrue("Should have 1 or more respiration entries - longitudinal", respVitals.size() > 0);
	}
	
	public VitalEntries fetchManyBySection(URI vitalsUri, String vitalSectionCode) {
		URI uri = UriBuilder.fromUri(vitalsUri).queryParam("section", vitalSectionCode).build();
		return client.getJerseyClient().resource(uri).get(VitalEntries.class);
	}

	@Test
	public void verifyVitalTableDataRetrieval() {
		URI providerEnteredVitalsTableDataUri = testPatient.getProviderEnteredVitalsTableDataUri();
		VitalTableData vitalTableData = client.getJerseyClient().resource(providerEnteredVitalsTableDataUri).get(VitalTableData.class);
		assertTrue(vitalTableData.size() > 0);
	}

	@Test
	public void verifyVitalTableDataForDateRange() {
		URI providerEnteredVitalsTableDataUri = testPatient.getProviderEnteredVitalsTableDataUri();
		providerEnteredVitalsTableDataUri = UriBuilder.fromUri(providerEnteredVitalsTableDataUri).queryParam("startDate", "12/02/2011").queryParam("endDate", "01/01/2020").build();

		VitalTableData vitalTableData = client.getJerseyClient().resource(providerEnteredVitalsTableDataUri).get(VitalTableData.class);
		assertTrue("Graph Data should have more than 0 records", vitalTableData.size() > 0);
	}

	@Test
	public void verifyVitalTableDataAreSortedInDescendingOrder() {
		URI providerEnteredVitalsTableDataUri = testPatient.getProviderEnteredVitalsTableDataUri();
		VitalTableData vitalTableData = client.getJerseyClient().resource(providerEnteredVitalsTableDataUri).get(VitalTableData.class);

		for (int i = 0; i < vitalTableData.size() - 1; i++) {
			if (vitalTableData.get(i).getEntryTime().compareTo(vitalTableData.get(i + 1).getEntryTime()) <= 0) {
				fail("Data is not sorted");
			}
		}
	}

	@Test
	public void verifyVitalGraphDataRetrieval() {
		URI providerEnteredVitalsGraphDataUri = testPatient.getProviderEnteredVitalsGraphDataUri();
		UriBuilder uriBuilder = UriBuilder.fromUri(providerEnteredVitalsGraphDataUri);
		uriBuilder.queryParam("section", "BP");
		providerEnteredVitalsGraphDataUri = uriBuilder.build();

		GraphData graphData = client.getJerseyClient().resource(providerEnteredVitalsGraphDataUri).get(GraphData.class);
		assertNotNull(graphData.getDataSeriesCollection().size() > 0);
	}

}
