package com.agilex.healthcare.mobilehealthplatform.restservice.tooltrackingresult;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;

import java.net.URI;
import java.util.Date;

import org.junit.After;
import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Ignore;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.AtomLink;
import com.agilex.healthcare.mobilehealthplatform.domain.LinkTitles;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.ToolTrackingResult;
import com.agilex.healthcare.mobilehealthplatform.domain.ToolTrackingResults;
import com.agilex.healthcare.mobilehealthplatform.domain.code.ToolTrackingCode;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.ModeHelper;


public class ToolTrackingResultResourceTest {

	private static MobileHealthClientTestVersion client;
	
	@BeforeClass
	public static void checkWhetherTestsAreRunningInCIHMode() {
		Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev", "dev"));
		client = TestHelper.createMobileHealthClient();
	}

	@AfterClass
	public static void destroy() {
		if (client != null) {
			client.getJerseyClient().destroy();
			client = null;
		}
	}
	
	PatientIdentifier integrationTestPatient = new PatientIdentifier("EDIPI", "D123401");
	String translatedId = "D123401";
	ToolTrackingResult baseResult;
	
	@Before
	public void before() {
		baseResult = saveTool();
	}
	
	@After
	public void after() {
		deleteTool(baseResult);
	}
	
	@Test
	public void verifyBaseLinks() {
		ToolTrackingResults results = client.getJerseyClient().resource(generateLink()).get(ToolTrackingResults.class);
		
		assertNotNull(results.getLink().getLinkByTitle(ToolTrackingCode.ACTIVITY_PACING));
		assertNotNull(results.getLink().getLinkByTitle(ToolTrackingCode.PLAN_PLEASANT_ACTIVITY));
		assertNotNull(results.getLink().getLinkByTitle(ToolTrackingCode.DEEP_BREATHING));
		assertNotNull(results.getLink().getLinkByTitle(ToolTrackingCode.DISTRACTION));
		assertNotNull(results.getLink().getLinkByTitle(ToolTrackingCode.SNAP));
		assertNotNull(results.getLink().getLinkByTitle(ToolTrackingCode.MUSCLE_RELAXATION));
		assertNotNull(results.getLink().getLinkByTitle(ToolTrackingCode.SLEEP_TIPS));
		assertNotNull(results.getLink().getLinkByTitle(ToolTrackingCode.VISUALIZATION));
	}
	
	@Test
	public void verifyFetchMostRecent() {
		ToolTrackingResults results = client.getJerseyClient().resource(generateLink()).get(ToolTrackingResults.class);
		
		AtomLink distractionLink = results.getLink().getLinkByTitle(ToolTrackingCode.DISTRACTION);
		
		ToolTrackingResult mostRecent = client.getJerseyClient().resource(distractionLink.getHref()).get(ToolTrackingResult.class);
		assertNotNull(mostRecent);
	}
	
	@Test
	public void verifyFetchSingleById() {
		ToolTrackingResult result = client.getJerseyClient().resource(baseResult.getSelfUri()).get(ToolTrackingResult.class);
		
		assertNotNull(result);
		assertEquals(baseResult.getUniqueId(), result.getUniqueId());
		assertEquals(baseResult.getTool(), result.getTool());
		assertEquals(baseResult.getFrequency(), result.getFrequency());
		assertEquals(baseResult.getUsefulness(), result.getUsefulness());
		assertEquals(baseResult.getSuccessRate(), result.getSuccessRate());
		assertEquals(baseResult.getDateTracked(), result.getDateTracked());
	}
	
	@Test
	public void verifySave() {
		assertNotNull(baseResult);
		assertNotNull(baseResult.getUniqueId());
		assertEquals(baseResult.getPatientId(), translatedId);
	}
	
	@Ignore
	@Test
	public void verifyUpdate() {
		baseResult.setTool(ToolTrackingCode.SNAP);
		
		ToolTrackingResult updated = client.getJerseyClient().resource(baseResult.getSelfUri()).put(ToolTrackingResult.class, baseResult);
		
		assertEquals(updated.getUniqueId(), baseResult.getUniqueId());
		assertEquals(ToolTrackingCode.SNAP, updated.getTool());
	}

	public ToolTrackingResult generateToolTrackingResult() {
		ToolTrackingResult result = new ToolTrackingResult();
		result.setFrequency("Every so often");
		result.setTool(ToolTrackingCode.DISTRACTION);
		result.setPatientIdentifier(integrationTestPatient);
		result.setUsefulness(40);
		result.setSuccessRate(100);
		result.setDateTracked(new Date());
		
		return result;
	}
	
	public ToolTrackingResult saveTool() {
		ToolTrackingResult result = generateToolTrackingResult();
		
		return client.getJerseyClient().resource(generateLink()).post(ToolTrackingResult.class, result);
	}
	
	private void deleteTool(ToolTrackingResult result) {
		client.getJerseyClient().resource(result.getSelfUri()).delete();
	}
	
	public URI generateLink() {
		return TestHelper.getPatientDataUri(integrationTestPatient, LinkTitles.ToolTrackingResults, "zztest.patient01");
	}
}
