package com.agilex.healthcare.mobilehealthplatform.restservice.securemessage;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.net.URI;

import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.UriBuilder;

import org.junit.*;

import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.AtomLink;
import com.agilex.healthcare.mobilehealthplatform.domain.AtomLinks;
import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessage;
import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessageCode;
import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessageUser;
import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessages;
import com.agilex.healthcare.testutility.IntegrationTestConfiguration;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.ModeHelper;
import com.sun.jersey.api.client.UniformInterfaceException;
@Ignore
public class SecureMessageResourceTest {
	
	SecureMessageUser defaultUser;
	String defaultIcn = "D123401";
	String defaultToId = "979";
	
	private static MobileHealthClientTestVersion client;
	
	@BeforeClass
	public static void checkWhetherTestsAreRunningInCIHMode() {
		Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev", "dev"));
		client = TestHelper.createMobileHealthClient();
	}

	@AfterClass
	public static void destroy() {
		if (client != null) {
			client.getJerseyClient().destroy();
			client = null;
		}
	}
	
	
	@Before
	public void before() {
		URI secureMessageUserUri = generateSecureMessageUserURI(defaultIcn);
		this.defaultUser = client.getJerseyClient().resource(secureMessageUserUri).get(SecureMessageUser.class);
	}
	
	@Test
	public void verifyFetchInbox() {
		SecureMessages secureMessages = this.getMessagesByFolderId(SecureMessageCode.FOLDER_ID_INBOX.toString());
		assertNotNull(secureMessages);
		SecureMessage secureMessage = secureMessages.get(0);
		
		assertEquals("Inbox", secureMessage.getFolder());
		assertEquals(SecureMessageCode.FOLDER_ID_INBOX.toString(), secureMessage.getFolderId());
	}
	
	@Test
	public void verifyFetchDrafts() {
		SecureMessages secureMessages = this.getMessagesByFolderId(SecureMessageCode.FOLDER_ID_DRAFT.toString());
		assertNotNull(secureMessages);
		
		SecureMessage secureMessage = secureMessages.get(0);
		assertEquals("Drafts", secureMessage.getFolder());
		assertEquals(SecureMessageCode.FOLDER_ID_DRAFT.toString(), secureMessage.getFolderId());
	}
	
	@Test
	public void verifyFetchDeleted() {
		SecureMessages secureMessages = this.getMessagesByFolderId(SecureMessageCode.FOLDER_ID_DELETED.toString());
		assertNotNull(secureMessages);
		
		SecureMessage secureMessage = secureMessages.get(0);
		assertEquals("Deleted", secureMessage.getFolder());
		assertEquals(SecureMessageCode.FOLDER_ID_DELETED.toString(), secureMessage.getFolderId());
	}
	
	@Test
	public void verifyFetchSent() {
		SecureMessages secureMessages = this.getMessagesByFolderId(SecureMessageCode.FOLDER_ID_SENT.toString());
		assertNotNull(secureMessages);
		
		SecureMessage secureMessage = secureMessages.get(0);
		assertEquals(SecureMessageCode.FOLDER_NAME_SENT, secureMessage.getFolder());
		assertEquals(SecureMessageCode.FOLDER_ID_SENT.toString(), secureMessage.getFolderId());
	}
	
	@Test
	public void verifySend() {
		SecureMessage secureMessage = generateBaseMessage();
		secureMessage.setPatientIdentifier(defaultUser.getPatientIdentifier());
		
		URI sendSecureMessageUri = generateSendSecureMessageURI(defaultUser.getPatientId());
		SecureMessage sentMessage = client.getJerseyClient().resource(sendSecureMessageUri).type(MediaType.APPLICATION_XML).accept("application/xml").post(SecureMessage.class, secureMessage);
		
		assertNotNull(sentMessage);
		assertNotNull(sentMessage.getAddresseeId());
		assertNotNull(sentMessage.getUniqueId());
		assertEquals("SM Resource Test Subject", sentMessage.getSubject());
		
		assertTrue(isInFolder(sentMessage.getUniqueId(), SecureMessageCode.FOLDER_ID_SENT.toString()));
	}
	
	@Test
	public void verifyReply() {
		SecureMessage existingMessage = get0thMessageOfFolder(SecureMessageCode.FOLDER_ID_INBOX.toString());
		
		assertNotNull(existingMessage);
		
		SecureMessage reply = generateBaseMessage();
		reply.setSubject("Reply Subject");
		reply.setMessageBody("Reply Body");
		reply.setReplyToMessageId(existingMessage.getUniqueId());
		
		URI replyUri = existingMessage.getLink().getUriByTitle("reply");
		SecureMessage messageReply = client.getJerseyClient().resource(replyUri).type(MediaType.APPLICATION_XML).accept("application/xml").post(SecureMessage.class, reply);
		assertNotNull(messageReply);
		assertNotNull(messageReply.getAddresseeId());
		assertNotNull(messageReply.getAddresseeOplock());

		assertTrue(isInFolder(messageReply.getUniqueId(), SecureMessageCode.FOLDER_ID_SENT.toString()));
	}
	
	@Test
	public void verifySaveDraft() {
		SecureMessage draftMessage = generateBaseMessage();
		URI draftUri = generateDraftSecureMessageURI(defaultUser.getPatientId());
		
		// always want to have a net gain in draft messages overall, sending 3 for good measure
		SecureMessage savedDraft = client.getJerseyClient().resource(draftUri).post(SecureMessage.class, draftMessage);
		client.getJerseyClient().resource(draftUri).type(MediaType.APPLICATION_XML).accept("application/xml").post(SecureMessage.class, draftMessage);
		client.getJerseyClient().resource(draftUri).type(MediaType.APPLICATION_XML).accept("application/xml").post(SecureMessage.class, draftMessage);
		assertNotNull(savedDraft.getAddresseeId());
		assertNotNull(savedDraft.getUniqueId());
		assertEquals("SM Resource Test Subject", savedDraft.getSubject());
		assertEquals("SM Resource Test Message Body", savedDraft.getMessageBody());
		assertTrue(isInFolder(savedDraft.getUniqueId(), SecureMessageCode.FOLDER_ID_DRAFT.toString()));
	}
	
	@Test
	public void verifySendDraft() {
		SecureMessage draftMessage = get0thMessageOfFolder(SecureMessageCode.FOLDER_ID_DRAFT.toString());
		draftMessage.setToId(this.defaultToId);
		draftMessage.setMailGroup(this.defaultToId);
		
		URI draftSecureMessageUri = generateDraftSecureMessageURI(defaultUser.getPatientId());
		draftSecureMessageUri = UriBuilder.fromUri(draftSecureMessageUri).path("id").path(draftMessage.getUniqueId()).build();
		
		SecureMessage sentDraft = client.getJerseyClient().resource(draftSecureMessageUri).type(MediaType.APPLICATION_XML).accept("application/xml").put(SecureMessage.class, draftMessage);
		assertNotNull(sentDraft);
		
		// TODO validate sent date
		assertTrue(isInFolder(sentDraft.getUniqueId(), SecureMessageCode.FOLDER_ID_SENT.toString()));
	}
	
	@Test
	public void verifyMove() {
		SecureMessage inboxMessage = get0thMessageOfFolder(SecureMessageCode.FOLDER_ID_INBOX.toString());
		
		assertEquals(inboxMessage.getFolderId(), SecureMessageCode.FOLDER_ID_INBOX.toString());
		
		URI baseMessageUri = generateSendSecureMessageURI(defaultUser.getPatientId());
		URI moveToDeletedUri = UriBuilder.fromUri(baseMessageUri).path("id").path(inboxMessage.getUniqueId()).path("move").build();
		inboxMessage.setFolderId(SecureMessageCode.FOLDER_ID_DELETED.toString());
		
		SecureMessage messageInDeletedFolder = client.getJerseyClient().resource(moveToDeletedUri).type(MediaType.APPLICATION_XML).accept("application/xml").post(SecureMessage.class, inboxMessage);
		assertNotNull(messageInDeletedFolder);
		assertEquals(SecureMessageCode.FOLDER_ID_DELETED.toString(), messageInDeletedFolder.getFolderId());
		
		// Move back to preserve data
		messageInDeletedFolder.setFolderId(SecureMessageCode.FOLDER_ID_INBOX.toString());
		client.getJerseyClient().resource(moveToDeletedUri).post(SecureMessage.class, messageInDeletedFolder);
	}
	
	@Test
	public void verifyDeleteDraft() {
		SecureMessages draftMessages = getMessagesByFolderId(SecureMessageCode.FOLDER_ID_DRAFT.toString());
		SecureMessage draftMessage = draftMessages.get(0);
		
		assertEquals(draftMessage.getFolderId(), SecureMessageCode.FOLDER_ID_DRAFT.toString());
		
		URI draftMessageUri = generateDraftSecureMessageURI(defaultUser.getPatientId());
		draftMessageUri = UriBuilder.fromUri(draftMessageUri).path("id").path(draftMessage.getUniqueId()).build();
		
		client.getJerseyClient().resource(draftMessageUri).delete();
		
		draftMessages = null;
		draftMessages = getMessagesByFolderId(SecureMessageCode.FOLDER_ID_DRAFT.toString());
		
		for(SecureMessage message : draftMessages) {
			if(draftMessage.getUniqueId().equals(message.getUniqueId())) {
				fail();
			}
		}
	}
	
	@Test
	public void verifyRead() {
		SecureMessage sentMessage = get0thMessageOfFolder(SecureMessageCode.FOLDER_ID_SENT.toString());
		
		assertEquals(sentMessage.getFolderId(), SecureMessageCode.FOLDER_ID_SENT.toString());
		
		URI readUri = sentMessage.getLink().getUriByTitle("read");
		SecureMessage readMessage = client.getJerseyClient().resource(readUri).type(MediaType.APPLICATION_XML).accept("application/xml").put(SecureMessage.class, sentMessage);
		
		assertEquals(Integer.parseInt(sentMessage.getAddresseeOplock())+1, Integer.parseInt(readMessage.getAddresseeOplock()));
		assertNotNull(readMessage.getMessageBody());
	}
	
	@Test
	public void verifyReadTwice() {
		SecureMessage sentMessage = get0thMessageOfFolder(SecureMessageCode.FOLDER_ID_SENT.toString());
		
		assertEquals(sentMessage.getFolderId(), SecureMessageCode.FOLDER_ID_SENT.toString());
		
		URI readUri = sentMessage.getLink().getUriByTitle("read");
		SecureMessage readMessage = client.getJerseyClient().resource(readUri).type(MediaType.APPLICATION_XML).accept("application/xml").put(SecureMessage.class, sentMessage);
		
		readUri = readMessage.getLink().getUriByTitle("read");
		SecureMessage readAgain = client.getJerseyClient().resource(readUri).type(MediaType.APPLICATION_XML).accept("application/xml").put(SecureMessage.class, readMessage);
		
		assertEquals(Integer.parseInt(readMessage.getAddresseeOplock())+1, Integer.parseInt(readAgain.getAddresseeOplock()));
		assertNotNull(readMessage.getMessageBody());
	}
	
	@Test
	public void verifyReadHasFoldersSet() {
		SecureMessage messageToRead = get0thMessageOfFolder(SecureMessageCode.FOLDER_ID_DRAFT.toString());
		
		SecureMessage readMessage = readMessage(messageToRead);
		
		assertNotNull(readMessage.getFolder());
		assertNotNull(readMessage.getFolderId());
	}
	
	@Test
	public void verifyUserLinks() {
		AtomLinks link = defaultUser.getLink();
		AtomLink foldersUri = link.getLinkByTitle("folders");
		assertNotNull(foldersUri);
		
		AtomLink contactsUri = link.getLinkByTitle("contacts");
		assertNotNull(contactsUri);
		
		AtomLink sendUri = link.getLinkByTitle("send");
		assertNotNull(sendUri);
		
		AtomLink saveDraftUri = link.getLinkByTitle("save-draft");
		assertNotNull(saveDraftUri);
	}
	
	@Test
	public void verifyInboxMessageLinks() {
		SecureMessages inboxMessages = getMessagesByFolderId(SecureMessageCode.FOLDER_ID_INBOX.toString());
		SecureMessage inboxMessage = inboxMessages.get(0);
		
		verifyDefaultLinks(inboxMessage);
		AtomLink moveLink = inboxMessage.getLink().getLinkByTitle("move");
		assertNotNull(moveLink);
		AtomLink deleteLink = inboxMessage.getLink().getLinkByTitle("delete");
		assertNotNull(deleteLink);
	}
	
	@Test
	public void verifySentMessageLinks() {
		SecureMessages sentMessages = getMessagesByFolderId(SecureMessageCode.FOLDER_ID_SENT.toString());
		SecureMessage sentMessage = sentMessages.get(0);
		
		verifyDefaultLinks(sentMessage);
		AtomLink deleteLinkShouldNotExistForSent = sentMessage.getLink().getLinkByTitle("delete");
		assertNull(deleteLinkShouldNotExistForSent);
	}
	
	@Test
	public void verifyDraftMessageLinks() {
		SecureMessages draftMessages = getMessagesByFolderId(SecureMessageCode.FOLDER_ID_DRAFT.toString());
		SecureMessage draftMessage = draftMessages.get(0);
		
		AtomLink readLink = draftMessage.getLink().getLinkByTitle("read");
		assertNotNull(readLink);
		AtomLink saveLink = draftMessage.getLink().getLinkByTitle("save-draft");
		assertNotNull(saveLink);
		AtomLink sendLink = draftMessage.getLink().getLinkByTitle("send");
		assertNotNull(sendLink);
		AtomLink deleteLink = draftMessage.getLink().getLinkByTitle("delete");
		assertNotNull(deleteLink);
	}
	
	@Test
	public void verifyDraftMessageLinksAfterRead() {
		SecureMessages draftMessages = getMessagesByFolderId(SecureMessageCode.FOLDER_ID_DRAFT.toString());
		SecureMessage draftMessageToRead = draftMessages.get(0);
		
		URI readUri = draftMessageToRead.getLink().getUriByTitle("read");
		SecureMessage readMessage = client.getJerseyClient().resource(readUri).type(MediaType.APPLICATION_XML).accept("application/xml").put(SecureMessage.class, draftMessageToRead);
		
		AtomLink readLink = readMessage.getLink().getLinkByTitle("read");
		assertNotNull(readLink);
		AtomLink saveLink = readMessage.getLink().getLinkByTitle("save-draft");
		assertNotNull(saveLink);
		AtomLink sendLink = readMessage.getLink().getLinkByTitle("send");
		assertNotNull(sendLink);
		AtomLink deleteLink = readMessage.getLink().getLinkByTitle("delete");
		assertNotNull(deleteLink);
	}
	
	@Test
	public void verifyDeletedMessageLinks() {
		SecureMessages deletedMessages = getMessagesByFolderId(SecureMessageCode.FOLDER_ID_DELETED.toString());
		SecureMessage deletedMessage = deletedMessages.get(0);
		
		
		verifyDefaultLinks(deletedMessage);
		AtomLink moveLink = deletedMessage.getLink().getLinkByTitle("move");
		assertNotNull(moveLink);
	}
	
	@Test
	public void verifyNoncachedPdf() {
		boolean test = false;

		SecureMessage messageToRead = get0thMessageOfFolder(SecureMessageCode.FOLDER_ID_DRAFT.toString());
		URI pdfUri = messageToRead.getLink().getUriByTitle("print");
		
		try {
			client.getJerseyClient().resource(pdfUri).accept("application/pdf").get(Response.class);
		} catch (UniformInterfaceException e) {
			if (Status.PRECONDITION_FAILED.getStatusCode() == e.getResponse().getStatus()) {
				test = true;
			}
		}
		
		assertTrue(test);
	}
	
	SecureMessage readMessage(SecureMessage messageToRead) {
		URI readUri = messageToRead.getLink().getUriByTitle("read");
		return client.getJerseyClient().resource(readUri).type(MediaType.APPLICATION_XML).accept("application/xml").put(SecureMessage.class, messageToRead);
	}
	
	public void verifyDefaultLinks(SecureMessage message) {
		AtomLink readLink = message.getLink().getLinkByTitle("read");
		assertNotNull(readLink);
		AtomLink replyLink = message.getLink().getLinkByTitle("reply");
		assertNotNull(replyLink);
	}
	
	private URI generateSecureMessageUserURI(String userIcn) {
		String secureMessageUserUriAsString = String.format("%s/securemessage-service/EDIPI/%s/secure-message-user", IntegrationTestConfiguration.getMobileHealthServerBaseUri(), userIcn);
		return UriBuilder.fromUri(secureMessageUserUriAsString).build();
	}
	
	private URI generateSendSecureMessageURI(String userId) {
		String secureMessageUserUriAsString = String.format("%s/securemessage-service/SM/%s/messages", IntegrationTestConfiguration.getMobileHealthServerBaseUri(), userId);
		return UriBuilder.fromUri(secureMessageUserUriAsString).build();
	}
	
	private URI generateDraftSecureMessageURI(String userId) {
		String secureMessageUserUriAsString = String.format("%s/securemessage-service/SM/%s/drafts/messages", IntegrationTestConfiguration.getMobileHealthServerBaseUri(), userId);
		return UriBuilder.fromUri(secureMessageUserUriAsString).build();
	}
	
	private SecureMessages getMessagesByFolderId(String folderId) {
		URI folderUri = constructFolderUri(folderId);
		folderUri = UriBuilder.fromUri(folderUri).queryParam("pageNumber", "1").queryParam("pageSize", "250").build();
		SecureMessages secureMessages = client.getJerseyClient().resource(folderUri).get(SecureMessages.class);
		return secureMessages;
	}
	
	private boolean isInFolder(String messageId, String folderId) {
		SecureMessages secureMessages = getMessagesByFolderId(folderId);
		
		for(SecureMessage message : secureMessages) {
			if(messageId.equalsIgnoreCase(message.getUniqueId())) {
				return true;
			}
		}
		
		return false;
	}
	
	SecureMessage get0thMessageOfFolder(String folderId) {
		SecureMessages messages = getMessagesByFolderId(folderId);
		SecureMessage message = messages.get(0);
	
		return message;
	}
	
	private URI constructFolderUri(String folderId) {
		String folderUri = String.format("%s/securemessage-service/SM/%s/folders/id/%s/messages", IntegrationTestConfiguration.getMobileHealthServerBaseUri(), defaultUser.getPatientId(), folderId);
		return UriBuilder.fromUri(folderUri).build();
	}

	private SecureMessage generateBaseMessage() {
		SecureMessage secureMessage = new SecureMessage();
		
		secureMessage.setSubject("SM Resource Test Subject");
		secureMessage.setMessageBody("SM Resource Test Message Body");
		secureMessage.setSubjectCategory("0");
		secureMessage.setMailGroup(this.defaultToId);
		secureMessage.setFromId(defaultUser.getPatientId());
		secureMessage.setToId(this.defaultToId);
		
		return secureMessage;
	}
}
