package com.agilex.healthcare.mobilehealthplatform.restservice.radiology;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.net.URI;
import java.util.Date;

import javax.ws.rs.core.UriBuilder;

import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.LinkTitles;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.RadiologyResult;
import com.agilex.healthcare.mobilehealthplatform.domain.RadiologyResults;
import com.agilex.healthcare.mobilehealthplatform.domain.RadiologyTest;
import com.agilex.healthcare.mobilehealthplatform.domain.RadiologyTests;
import com.agilex.healthcare.testutility.PatientLoader;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.ModeHelper;

public class RadiologyResourceTest {
	Patient patient = PatientLoader.loadTestLabPatient();

	private static MobileHealthClientTestVersion client;
	
	@BeforeClass
	public static void checkWhetherTestsAreRunningInCIHMode() {
		Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev", "dev"));
		client = TestHelper.createMobileHealthClient();
	}

	@AfterClass
	public static void destroy() {
		if (client != null) {
			client.getJerseyClient().destroy();
			client = null;
		}
	}
	
	@Test
	public void getZeroResults() {
		URI uri = getRadiologyResultsUri();
		uri = UriBuilder.fromUri(uri).queryParam("date", "1/1/1800").build();
		RadiologyResults zeroResults = client.getJerseyClient().resource(uri).get(RadiologyResults.class);
		assertEquals(0, zeroResults.size());
	}

	@Test
	public void retrieveResults() {
		URI uri = getRadiologyResultsUri();
		RadiologyResults radResults = client.getJerseyClient().resource(uri).get(RadiologyResults.class);
		assertTrue(radResults.size() > 0);
	}

	@Test
	public void getResultsByEntryDate() {
		URI uri = getRadiologyResultsUri();
		uri = UriBuilder.fromUri(uri).queryParam("date", "6/3/2009").build();
		RadiologyResults radResults = client.getJerseyClient().resource(uri).get(RadiologyResults.class);
		assertTrue(radResults.size() > 0);
		for (RadiologyResult radResult : radResults) {
			assertDatePartsEquals("6/3/2009", radResult.getResultedDate());
		}
	}

	@Test
	public void getResultsByDateRange() {
		URI uri = getRadiologyResultsUri();
		uri = UriBuilder.fromUri(uri).queryParam("startDate", "1/1/2000").queryParam("enddate", "1/1/2001").build();
		RadiologyResults radResults = client.getJerseyClient().resource(uri).get(RadiologyResults.class);
		assertTrue(radResults.size() > 0);
		for (RadiologyResult radResult : radResults) {
			assertTrue(radResult.getResultedDate().compareTo(DateHelper.parseDate("1/1/2000")) >= 0);
			assertTrue(radResult.getResultedDate().compareTo(DateHelper.parseDate("1/1/2001")) <= 0);
		}
	}

	@Test
	public void getZeroTests() {
		URI uri = getRadiologyTestsUri();
		uri = UriBuilder.fromUri(uri).queryParam("date", "1/1/1800").build();
		RadiologyTests radTests = client.getJerseyClient().resource(uri).get(RadiologyTests.class);
		assertEquals(0, radTests.size());
	}

	@Test
	public void getTests() {
		URI uri = getRadiologyTestsUri();
		RadiologyTests radTests = client.getJerseyClient().resource(uri).get(RadiologyTests.class);
		assertTrue(radTests.size() > 0);
	}

	@Test
	public void getTestsByEntryDate() {
		URI uri = getRadiologyTestsUri();
		uri = UriBuilder.fromUri(uri).queryParam("date", "6/3/2009").build();
		RadiologyTests radTests = client.getJerseyClient().resource(uri).get(RadiologyTests.class);
		assertTrue(radTests.size() > 0);
		for (RadiologyTest radTest : radTests) {
			for (RadiologyResult radResult : radTest.getRadiologyResults()) {
				assertDatePartsEquals("6/3/2009", radResult.getResultedDate());
			}
		}
	}

	@Test
	public void getTestsByDateRange() {
		URI uri = getRadiologyTestsUri();
		uri = UriBuilder.fromUri(uri).queryParam("startDate", "1/1/2000").queryParam("enddate", "1/1/2001").build();
		RadiologyTests radTests = client.getJerseyClient().resource(uri).get(RadiologyTests.class);
		assertTrue(radTests.size() > 0);
		for (RadiologyTest radTest : radTests) {
			for (RadiologyResult radResult : radTest.getRadiologyResults()) {
				assertTrue(radResult.getResultedDate().compareTo(DateHelper.parseDate("1/1/2000")) >= 0);
				assertTrue(radResult.getResultedDate().compareTo(DateHelper.parseDate("1/1/2001")) <= 0);
			}
		}
	}

	private void assertDatePartsEquals(String expected, Date actual) {
		assertEquals(getDatePart(DateHelper.parseDate(expected)), getDatePart(actual));
	}

	private Date getDatePart(Date dt) {
		return DateHelper.parseDate(DateHelper.formatDate(dt));
	}

	private URI getRadiologyTestsUri() {
		return patient.getLink().getUriByTitle(LinkTitles.PatientRadiologyTests);
	}

	private URI getRadiologyResultsUri() {
		return patient.getLink().getUriByTitle(LinkTitles.PatientRadiologyResults);
	}

}
