package com.agilex.healthcare.mobilehealthplatform.restservice.patients;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.Test;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.testutility.IntegrationTestConfiguration;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.ModeHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.XpathHelper;
import com.sun.jersey.api.client.Client;

public class PatientSearchViaXmlTest {
	private static final String escaped_space = "%20";
	private String patientSearchURI = IntegrationTestConfiguration.getMobileHealthServerBaseUri() + "/patients";

	private static MobileHealthClientTestVersion client;
	
	@BeforeClass
	public static void setup() {
		Assume.assumeTrue(ModeHelper.isMode("va-provider-dev"));
		client = TestHelper.createMobileHealthClient();
	}

	@AfterClass
	public static void destroy() {
		if(client != null){
			client.getJerseyClient().destroy();
		}
	}
	
	@Test
	public void validateBasicPatientSearch() {
		String lastNameSearchCriteria = IntegrationTestConfiguration.getTestPatient().getLastName();
		String searchUri = patientSearchURI + "?lastname=" + lastNameSearchCriteria;
		Document result = client.getAsXml(searchUri);
		NodeList patientsNode = XpathHelper.getNodeList(result, "//m:patients/m:patient");
		assertEquals("No search results for " + lastNameSearchCriteria, true, patientsNode.getLength() >= 1);
		for (int i = 0; i < patientsNode.getLength(); i++) {
			Node patient = patientsNode.item(i);
			String patientLastName = XpathHelper.getString(patient, "lastName");
			if (NullChecker.isNotNullish(patientLastName))
				patientLastName = patientLastName.toLowerCase();
			assertEquals(lastNameSearchCriteria.toLowerCase(), patientLastName);
		}
	}

	private Node getPatientByLastName(String lastname) {
		NodeList patients = getPatientsByLastName(lastname);
		return patients.item(0);
	}

	private NodeList getPatientsByLastName(String lastname) {
		Client jerseyClient = TestHelper.createJerseyClient();
		com.sun.jersey.api.client.WebResource resourceHandle = jerseyClient.resource(IntegrationTestConfiguration.getMobileHealthServerBaseUri()).path("patients");
		resourceHandle = resourceHandle.queryParam("lastname", lastname);
		Document resultDocument = resourceHandle.get(Document.class);
		NodeList patients = XpathHelper.getNodeList(resultDocument, "//m:patient");
		return patients;
	}

	@Test
	public void validateNoMatchReturnsEmptyCollection() {
		String searchParamName = "XYZ";
		NodeList patients = getPatientsByLastName(searchParamName);
		assertEquals(0, patients.getLength());
	}

	@Test
	public void validateOneMatchReturnsCollectionWithSizeOne() {
		String searchParamName = IntegrationTestConfiguration.getTestPatient().getLastName();
		NodeList patients = getPatientsByLastName(searchParamName);
		assertEquals(1, patients.getLength());
	}

	@Test
	public void validateSearchReturnsLastName() {
		String lastname = IntegrationTestConfiguration.getTestPatient().getLastName();
		Node patient = getPatientByLastName(lastname);
		String value = XpathHelper.getString(patient, "lastName");
		String expected = IntegrationTestConfiguration.getTestPatient().getLastName();
		assertEquals(expected.toLowerCase(), value.toLowerCase());
	}

	@Test
	public void validateSearchReturnsFirstName() {
		String lastname = IntegrationTestConfiguration.getTestPatient().getLastName();
		Node patient = getPatientByLastName(lastname);
		String value = XpathHelper.getString(patient, "firstName");
		String expected = IntegrationTestConfiguration.getTestPatient().getFirstName();
		assertTrue(TestHelper.beginsWith(value, expected));
	}

	@Test
	public void validatePatientSearchResultsAreInNamespace() {
		String lastNameSearchCriteria = IntegrationTestConfiguration.getTestPatient().getLastName();
		String searchUri = patientSearchURI + "?lastname=" + lastNameSearchCriteria;
		Document result = client.getAsXml(searchUri);
		System.out.println(result.getNamespaceURI());

		Node patients = result.getFirstChild();
		System.out.println(patients.getNamespaceURI());

		patients = XpathHelper.getNode(result, "/m:patients");
		System.out.println(patients.getNamespaceURI());
	}

	@Test
	public void validateBasicPatientSearchUsingSearchStringAsLastName() {
		String lastNameSearchCriteria = IntegrationTestConfiguration.getTestPatient().getLastName();
		String searchUri = patientSearchURI + "?searchString=" + lastNameSearchCriteria;
		Document result = client.getAsXml(searchUri);
		NodeList patientsNode = XpathHelper.getNodeList(result, "//m:patients/m:patient");
		assertEquals(1, patientsNode.getLength());
	}

	@Test
	public void validateBasicPatientSearchUsingSearchStringAsLastNameCommaFirstName() {
		String lastNameSearchCriteria = IntegrationTestConfiguration.getTestPatient().getLastName();
		String firstNameSearchCriteria = IntegrationTestConfiguration.getTestPatient().getFirstName();
		String searchUri = patientSearchURI + "?searchString=" + lastNameSearchCriteria + "," + firstNameSearchCriteria;
		Document result = client.getAsXml(searchUri);
		NodeList patientsNode = XpathHelper.getNodeList(result, "//m:patients/m:patient");
		assertEquals(1, patientsNode.getLength());
	}

	@Test
	public void validateBasicPatientSearchUsingSearchStringSsn() {
		String ssn = "999999983";
		String searchUri = patientSearchURI + "?searchString=" + ssn;
		Document result = client.getAsXml(searchUri);
		NodeList patientsNode = XpathHelper.getNodeList(result, "//m:patients/m:patient");
		assertEquals(1, patientsNode.getLength());
	}

	@Test
	public void validateBasicPatientSearchUsingSearchStringSsnWithDashes() {
		String ssn = "999-99-9983";
		String searchUri = patientSearchURI + "?searchString=" + ssn;
		Document result = client.getAsXml(searchUri);
		NodeList patientsNode = XpathHelper.getNodeList(result, "//m:patients/m:patient");
		assertEquals(1, patientsNode.getLength());
	}

	@Test
	public void validateBasicPatientSearchUsingSsn() {
		String ssn = "999999983";
		String searchUri = patientSearchURI + "?ssn=" + ssn;
		Document result = client.getAsXml(searchUri);
		NodeList patientsNode = XpathHelper.getNodeList(result, "//m:patients/m:patient");
		assertEquals(1, patientsNode.getLength());
	}

	@Test
	public void validateBasicPatientSearchUsingSsnWithDashes() {
		String ssn = "999-99-9983";
		String searchUri = patientSearchURI + "?ssn=" + ssn;
		Document result = client.getAsXml(searchUri);
		NodeList patientsNode = XpathHelper.getNodeList(result, "//m:patients/m:patient");
		assertEquals(1, patientsNode.getLength());
	}

	@Test
	public void validateBasicPatientSearchUsingSearchStringWithLastNameInitialAndLastFourOfSsn() {
		String searchString = "p9983";
		String searchUri = patientSearchURI + "?searchstring=" + searchString;
		Document result = client.getAsXml(searchUri);
		NodeList patientsNode = XpathHelper.getNodeList(result, "//m:patients/m:patient");
		assertEquals(1, patientsNode.getLength());
	}

	@Test
	public void validateBasicPatientSearchUsingSearchStringWithLastFourOfSsn() {
		String searchString = "9983";
		String searchUri = patientSearchURI + "?searchstring=" + searchString;
		Document result = client.getAsXml(searchUri);
		NodeList patientsNode = XpathHelper.getNodeList(result, "//m:patients/m:patient");
		assertEquals(1, patientsNode.getLength());
	}

	@Test
	public void validateBasicPatientSearchUsingSearchStringWithDashLastNameOnly() {
		String searchUri = patientSearchURI + "?searchString=patient-71";
		Document result = client.getAsXml(searchUri);
		NodeList patientsNode = XpathHelper.getNodeList(result, "//m:patients/m:patient");
		assertEquals(1, patientsNode.getLength());
	}

	@Test
	public void validateBasicPatientSearchUsingSearchStringWithDash() {
		String searchUri = patientSearchURI + "?searchString=patient-71,zztest";
		Document result = client.getAsXml(searchUri);
		NodeList patientsNode = XpathHelper.getNodeList(result, "//m:patients/m:patient");
		assertEquals(1, patientsNode.getLength());
	}

	@Test
	public void validateBasicPatientSearchUsingEscapedSpace() {
		Patient patientWithSpace = IntegrationTestConfiguration.getPatient("patientWithSpaceInName");
		// search string in format of last,first, with first name containing an
		// escaped space
		String searchUri = String.format("%s?searchstring=%s,%s", patientSearchURI, patientWithSpace.getLastName(), patientWithSpace.getFirstName()).replace(" ", escaped_space);
		assertTrue(searchUri.contains(escaped_space));
		System.out.println("searching by: " + searchUri);
		Document result = client.getAsXml(searchUri);
		NodeList patientsNode = XpathHelper.getNodeList(result, "//m:patients/m:patient");
		assertEquals(1, patientsNode.getLength());
	}

}
