package com.agilex.healthcare.mobilehealthplatform.restservice.patient.calendar;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;

import java.net.URI;

import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.Ignore;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.client.AuthenticationInfo;
import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.Appointments;
import com.agilex.healthcare.mobilehealthplatform.domain.CalendarDirectory;
import com.agilex.healthcare.mobilehealthplatform.domain.LinkTitles;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.testutility.PatientLoader;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.ModeHelper;
import com.sun.jersey.api.client.Client;

public class PatientCalendarResourceTest {
	
	private static MobileHealthClientTestVersion client;
	
	@BeforeClass
	public static void checkWhetherTestsAreRunningInCIHMode() {
		Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev", "dev"));
		AuthenticationInfo authenticationInfo = new AuthenticationInfo("zztest.patient75", "pass", "oauth");
		client = TestHelper.createMobileHealthClient(authenticationInfo);

	}

	@AfterClass
	public static void destroy() {
		if (client != null) {
			client.getJerseyClient().destroy();
			client = null;
		}
	}
	
	@Test
	public void patientHasLinkToPublicKey() {
		URI publicKeyUri = determinePatientCalendarPublicKeyUri();
		assertNotNull(publicKeyUri);
	}
 
	@Test
	public void ensureThatPatientWithoutKeyCreatesNewKey() {
		URI publicKeyUri = determinePatientCalendarPublicKeyUri();
		CalendarDirectory links = client.getJerseyClient().resource(publicKeyUri).get(CalendarDirectory.class);
		assertNotNull(links);
		URI publicCalendarUri = links.getLink().getUriByTitle("PublicCalendarFeedByKey");
		assertNotNull(publicCalendarUri);
	}

	@Test
	public void createPatientPublicCalendarFeedKey() {
		URI publicKeyUri = determinePatientCalendarPublicKeyUri();
		CalendarDirectory links = client.getJerseyClient().resource(publicKeyUri).post(CalendarDirectory.class, null);
		assertNotNull(links);
		URI publicCalendarUri = links.getLink().getUriByTitle("PublicCalendarFeedByKey");
		assertNotNull(publicCalendarUri);
	}

	@Test
	public void ableToRetrieveAKeyAfterCreation() {
		URI publicKeyUri = determinePatientCalendarPublicKeyUri();
		client.getJerseyClient().resource(publicKeyUri).post(CalendarDirectory.class, null);

		CalendarDirectory links = client.getJerseyClient().resource(publicKeyUri).get(CalendarDirectory.class);
		URI publicCalendarUri = links.getLink().getUriByTitle("PublicCalendarFeedByKey");
		assertNotNull(publicCalendarUri);
	}

	
	@Test
	public void canFetchPublicCalendar() {
		URI publicKeyUri = determinePatientCalendarPublicKeyUri();
		client.getJerseyClient().resource(publicKeyUri).post(CalendarDirectory.class, null);

		CalendarDirectory links = client.getJerseyClient().resource(publicKeyUri).get(CalendarDirectory.class);
		URI publicCalendarUri = links.getLink().getUriByTitle("PublicCalendarEventsByKey");

		Client unauthenticatedJerseyClient = new Client();
		Appointments appointments = unauthenticatedJerseyClient.resource(publicCalendarUri).get(Appointments.class);
		assertEquals(0, appointments.size());
	}

	@Test
	public void ableToPostMultipleTimes() {
		URI publicKeyUri = determinePatientCalendarPublicKeyUri();

		client.getJerseyClient().resource(publicKeyUri).post(CalendarDirectory.class, null);
		CalendarDirectory links1 = client.getJerseyClient().resource(publicKeyUri).get(CalendarDirectory.class);
		URI publicCalendarUri1 = links1.getLink().getUriByTitle("CalendarEventsByKey");

		client.getJerseyClient().resource(publicKeyUri).post(CalendarDirectory.class, null);
		CalendarDirectory links2 = client.getJerseyClient().resource(publicKeyUri).get(CalendarDirectory.class);
		URI publicCalendarUri2 = links2.getLink().getUriByTitle("CalendarEventsByKey");

		assertEquals(publicCalendarUri1, publicCalendarUri2);
	}

	private URI determinePatientCalendarPublicKeyUri() {
		PatientIdentifier patientIdentifier = new PatientIdentifier("EDIPI", "INT-TEST-MOCK");
		Patient patient = PatientLoader.loadPatient(patientIdentifier, "zztest.patient75");
		URI publicKeyUri = patient.getLink().getUriByTitle(LinkTitles.PatientCalendarPublicKey);
		return publicKeyUri;
	}
}
