package com.agilex.healthcare.mobilehealthplatform.restservice.patient.allergy;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import org.junit.Assume;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Ignore;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.client.AuthenticationInfo;
import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.clientapi.PatientResourceClient;
import com.agilex.healthcare.mobilehealthplatform.domain.Allergies;
import com.agilex.healthcare.mobilehealthplatform.domain.Allergy;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.testutility.IntegrationTestConfiguration;
import com.agilex.healthcare.testutility.NullCheckerAssert;
import com.agilex.healthcare.testutility.PatientLoader;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.ModeHelper;
import com.agilex.healthcare.utility.NullChecker;

public class PatientAllergiesViaApi {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(PatientAllergiesViaApi.class);

	private Allergy testAllergy;

    @BeforeClass
    public static void isVeteranMode(){
        Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev"));
    }
    
	@Before
	public void setup() {
		testAllergy = IntegrationTestConfiguration.getTestAllergy();
		assertNotNull(testAllergy);
	}

	private Allergies getTestAllergies() {
		MobileHealthClientTestVersion client = TestHelper.createMobileHealthClient("zztest.patient01", "pass", "oauth");
		Patient patient = PatientLoader.loadPatient(new PatientIdentifier("EDIPI", testAllergy.getPatientId()),  "zztest.patient01");
		Allergies allergies = client.getPatientResourceClient().getPatientAllergies(patient);
		client.getJerseyClient().destroy();
		return allergies;
	}

	private Allergy retrieveTestAllergy() {
		Allergies allergies = getTestAllergies();
		Allergy allergy = null;
		for (Allergy potentialMatch : allergies)
			if (testAllergy.getSubstance().contentEquals(potentialMatch.getSubstance())) {
				allergy = potentialMatch;
				break;
			}
		if (allergy == null)
			logger.warn(String.format("Unable to load test allergy (%s)", testAllergy.getSubstance()));
		return allergy;
	}

	@Test
	public void retrievePatientAllergies() {
		Allergies allergies = getTestAllergies();
		NullCheckerAssert.assertNotNullish(allergies);
	}

	@Ignore
	public void patientAllergiesHasSelfLink() {
		Allergies allergies = getTestAllergies();
		NullCheckerAssert.assertNotNullish(allergies.getSelfUri());
	}

	@Ignore
	public void eachAllergyHasSelfLink() {
		Allergies allergies = getTestAllergies();
		for (Allergy allergy : allergies) {
			NullCheckerAssert.assertNotNullish(allergy.getSelfUri());
		}
	}

	@Test
	public void allergiesContainsSubstance() {
		Allergies allergies = getTestAllergies();
		for (Allergy allergy : allergies) {
			assertTrue(NullChecker.isNotNullish(allergy.getSubstance()));
		}
	}

	@Test
	public void allergyContainsProperSubstance() {
		Allergy retrievedAllergy = retrieveTestAllergy();
		assertEquals(testAllergy.getSubstance(), retrievedAllergy.getSubstance());
	}

	@Test
	public void allergyContainsProperReaction() {
		Allergy allergy = retrieveTestAllergy();
		assertEquals(testAllergy.getReaction().toLowerCase(), allergy.getReaction().toLowerCase());
	}

	@Test
	public void allergyContainsProperSeverity() {
		Allergy allergy = retrieveTestAllergy();
		assertEquals(testAllergy.getSeverity().toLowerCase(), allergy.getSeverity().toLowerCase());
	}

	@Test
	public void validateAllergyNoDataFoundWhenThereIsNoDataInTheDATABASE() throws Exception {

		String userName = "zztest.patient76";
		AuthenticationInfo authenticationInfo = new AuthenticationInfo(userName, "pass", "oauth");
		MobileHealthClientTestVersion noDataClient = TestHelper.createMobileHealthClient(authenticationInfo);
		
		PatientIdentifier patientIdentifier = new PatientIdentifier("EDIPI", "NO-DATA-MOCK");
		
		PatientResourceClient patientClient = noDataClient.getPatientResourceClient();
		Patient patient = PatientLoader.loadPatient(patientIdentifier, userName);
		Allergies allergies = patientClient.getPatientAllergies(patient);
		assertEquals(0, allergies.size());
		
		noDataClient.getJerseyClient().destroy();
	}
	
	@Test
	public void verifySortOrder()
	{
		Allergies allergies = getTestAllergies();
		
		String allergy1;
		String allergy2;
		
		for(int i = 0; i < allergies.getSize() - 1; i++)
		{
			allergy1 = allergies.get(i).getSubstance().toLowerCase();
			allergy2 = allergies.get(i + 1).getSubstance().toLowerCase();
			assertTrue(allergy1.compareTo(allergy2) <= 0);
		}
		
	}

}
