package com.agilex.healthcare.mobilehealthplatform.restservice.pain;

import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.net.URI;

import javax.ws.rs.core.UriBuilder;

import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.client.AuthenticationInfo;
import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.PainGoal;
import com.agilex.healthcare.mobilehealthplatform.domain.PainGoalEntries;
import com.agilex.healthcare.mobilehealthplatform.domain.PainGoalEntry;
import com.agilex.healthcare.mobilehealthplatform.domain.PainGoals;
import com.agilex.healthcare.testutility.IntegrationTestConfiguration;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.ModeHelper;
import com.agilex.healthcare.utility.PainGoalEntryStatus;
import com.agilex.healthcare.utility.PainGoalStatus;

public class PainResourceTest {
	private String patientEDIPI = "INT-TEST-MOCK";
	URI painGoalsURI = null;
	private static MobileHealthClientTestVersion client;
	
	@BeforeClass
	public static void validateMode() {
		Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev"));
		AuthenticationInfo authenticationInfo = new AuthenticationInfo("zztest.patient75", "pass", "oauth");
		client = TestHelper.createMobileHealthClient(authenticationInfo);
	}

	@AfterClass
	public static void destroy() {
		if (client != null) {
			client.getJerseyClient().destroy();
			client = null;
		}
	}

	@Before 
	public void before() {
		painGoalsURI = generatePainGoalsURI();
	}

	
	@Test
	public void getPainGoals() {
	
		PainGoals painGoals = client.getJerseyClient().resource(painGoalsURI).get(PainGoals.class);
		
		assertNotNull(painGoals);
	}
	
	@Test 
	public void postPainGoals() {
		PainGoals initialPainGoals = client.getJerseyClient().resource(painGoalsURI).get(PainGoals.class);
		assertNotNull(initialPainGoals);

		int initialNumberOfPainGoals = initialPainGoals.size();
		
		
		PainGoal painGoalA = new PainGoal();
		painGoalA.setDateStarted(DateHelper.getToday());
		painGoalA.setDetails("Details 1");
		painGoalA.setGoalType("Veteran");
		painGoalA.setName("Goal 1");
		painGoalA.setPatientId(patientEDIPI);
		painGoalA.setPercentComplete("20");
		painGoalA.setStatus(PainGoalStatus.PainGoalStatusActive);
		painGoalA.setTargetDate(DateHelper.get3MonthsFromNow());
		
		PainGoalEntries entries = new PainGoalEntries();
		PainGoalEntry entry = new PainGoalEntry();
		entry.setComments("Entry 1A");
		entry.setEntryDate(DateHelper.getToday());
		entry.setStatus(PainGoalEntryStatus.PainGoalEntryStatusActive);
		entries.add(entry);
		
		painGoalA.setPainGoalEntries(entries);
				
		PainGoal returnedPainGoal = client.getJerseyClient().resource(painGoalsURI).post(PainGoal.class, painGoalA);
		
		assertTrue("Goal 1".equalsIgnoreCase(returnedPainGoal.getName()));
		assertTrue("Details 1".equalsIgnoreCase(returnedPainGoal.getDetails()));
		assertTrue(patientEDIPI.equalsIgnoreCase(returnedPainGoal.getPatientId()));
		assertNotNull(returnedPainGoal.getDataIdentifier().getUniqueId());
		assertTrue(returnedPainGoal.getPainGoalEntries().size() == 1);
		assertTrue("Entry 1A".equalsIgnoreCase(returnedPainGoal.getNextActionStep()));

		
		PainGoals returnedPainGoals = client.getJerseyClient().resource(painGoalsURI).get(PainGoals.class);

		assertTrue(returnedPainGoals.size() == initialNumberOfPainGoals + 1);
		
		URI selfGoalURI = returnedPainGoal.getSelfUri();
		client.getJerseyClient().resource(selfGoalURI).delete();	
	}
	
	@Test 
	public void postPainGoalsWithoutEntries() {
		PainGoals initialPainGoals = client.getJerseyClient().resource(painGoalsURI).get(PainGoals.class);
		assertNotNull(initialPainGoals);

		int initialNumberOfPainGoals = initialPainGoals.size();
		
		
		PainGoal painGoalA = new PainGoal();
		painGoalA.setDateStarted(DateHelper.getToday());
		painGoalA.setDetails("Details 2");
		painGoalA.setGoalType("Veteran");
		painGoalA.setName("Goal 2");
		painGoalA.setPatientId(patientEDIPI);
		painGoalA.setPercentComplete("20");
		painGoalA.setStatus(PainGoalStatus.PainGoalStatusActive);
		painGoalA.setTargetDate(DateHelper.get3MonthsFromNow());
		

		PainGoal returnedPainGoal = client.getJerseyClient().resource(painGoalsURI).post(PainGoal.class, painGoalA);
		
		assertTrue("Goal 2".equalsIgnoreCase(returnedPainGoal.getName()));
		assertTrue("Details 2".equalsIgnoreCase(returnedPainGoal.getDetails()));
		assertTrue(patientEDIPI.equalsIgnoreCase(returnedPainGoal.getPatientId()));
		assertNotNull(returnedPainGoal.getDataIdentifier().getUniqueId());
		assertTrue("".equalsIgnoreCase(returnedPainGoal.getNextActionStep()));

		
		PainGoals returnedPainGoals = client.getJerseyClient().resource(painGoalsURI).get(PainGoals.class);

		assertTrue(returnedPainGoals.size() == initialNumberOfPainGoals + 1);
	
		URI selfGoalURI = returnedPainGoal.getSelfUri();
		client.getJerseyClient().resource(selfGoalURI).delete();	
	}
	
	
	@Test
	public void putGoal() {
		PainGoals initialPainGoals = client.getJerseyClient().resource(painGoalsURI).get(PainGoals.class);
		assertNotNull(initialPainGoals);

		int initialNumberOfPainGoals = initialPainGoals.size();
		
		
		PainGoal painGoalA = new PainGoal();
		painGoalA.setDateStarted(DateHelper.getToday());
		painGoalA.setDetails("Details 3");
		painGoalA.setGoalType("Veteran");
		painGoalA.setName("Goal 3");
		painGoalA.setPatientId(patientEDIPI);
		painGoalA.setPercentComplete("40");
		painGoalA.setStatus(PainGoalStatus.PainGoalStatusActive);
		painGoalA.setTargetDate(DateHelper.get3MonthsFromNow());
		
		PainGoalEntries entries = new PainGoalEntries();
		PainGoalEntry entry = new PainGoalEntry();
		entry.setComments("Entry 3A");
		entry.setEntryDate(DateHelper.getToday());
		entry.setStatus(PainGoalEntryStatus.PainGoalEntryStatusActive);
		entries.add(entry);
		
		painGoalA.setPainGoalEntries(entries);
		
				
		PainGoal postedGoal = client.getJerseyClient().resource(painGoalsURI).post(PainGoal.class, painGoalA);
		
		assertTrue("Goal 3".equalsIgnoreCase(postedGoal.getName()));
		assertTrue("Details 3".equalsIgnoreCase(postedGoal.getDetails()));
		assertTrue(patientEDIPI.equalsIgnoreCase(postedGoal.getPatientId()));
		assertNotNull(postedGoal.getDataIdentifier().getUniqueId());
		assertTrue(postedGoal.getPainGoalEntries().size() == 1);
		assertTrue("Entry 3A".equalsIgnoreCase(postedGoal.getNextActionStep()));

		
		PainGoals returnedPainGoals = client.getJerseyClient().resource(painGoalsURI).get(PainGoals.class);

		assertTrue(returnedPainGoals.size() == initialNumberOfPainGoals + 1);
		
		URI selfGoalURI = postedGoal.getSelfUri();
	
		postedGoal.setDetails("Details Modified");
	
		PainGoal putGoal = client.getJerseyClient().resource(selfGoalURI).put(PainGoal.class, postedGoal);
		
		assertTrue("Goal 3".equalsIgnoreCase(putGoal.getName()));
		assertTrue("Details Modified".equalsIgnoreCase(putGoal.getDetails()));
		assertTrue(patientEDIPI.equalsIgnoreCase(putGoal.getPatientId()));
		assertTrue(postedGoal.getUniqueId().equalsIgnoreCase(putGoal.getUniqueId()));
		
		assertTrue(putGoal.getPainGoalEntries().size() == 1);
		assertTrue("Entry 3A".equalsIgnoreCase(putGoal.getNextActionStep()));

		
		PainGoals returnedPainGoalsB = client.getJerseyClient().resource(painGoalsURI).get(PainGoals.class);

		assertTrue(returnedPainGoalsB.size() == initialNumberOfPainGoals + 1);		

		client.getJerseyClient().resource(selfGoalURI).delete();
	}
	
	@Test
	public void putGoalWithoutEntries() {
		PainGoals initialPainGoals = client.getJerseyClient().resource(painGoalsURI).get(PainGoals.class);
		assertNotNull(initialPainGoals);

		int initialNumberOfPainGoals = initialPainGoals.size();
		
		
		PainGoal painGoalA = new PainGoal();
		painGoalA.setDateStarted(DateHelper.getToday());
		painGoalA.setDetails("Details 4");
		painGoalA.setGoalType("Veteran");
		painGoalA.setName("Goal 4");
		painGoalA.setPatientId(patientEDIPI);
		painGoalA.setPercentComplete("40");
		painGoalA.setStatus(PainGoalStatus.PainGoalStatusActive);
		painGoalA.setTargetDate(DateHelper.get3MonthsFromNow());
		
				
		PainGoal postedGoal = client.getJerseyClient().resource(painGoalsURI).post(PainGoal.class, painGoalA);
		
		assertTrue("Goal 4".equalsIgnoreCase(postedGoal.getName()));
		assertTrue("Details 4".equalsIgnoreCase(postedGoal.getDetails()));
		assertTrue(patientEDIPI.equalsIgnoreCase(postedGoal.getPatientId()));
		assertNotNull(postedGoal.getDataIdentifier().getUniqueId());
		assertTrue("".equalsIgnoreCase(postedGoal.getNextActionStep()));

		
		PainGoals returnedPainGoals = client.getJerseyClient().resource(painGoalsURI).get(PainGoals.class);

		assertTrue(returnedPainGoals.size() == initialNumberOfPainGoals + 1);
		
		URI selfGoalURI = postedGoal.getSelfUri();
	
		postedGoal.setDetails("Details Modified");
	
		PainGoal putGoal = client.getJerseyClient().resource(selfGoalURI).put(PainGoal.class, postedGoal);
		
		assertTrue("Goal 4".equalsIgnoreCase(putGoal.getName()));
		assertTrue("Details Modified".equalsIgnoreCase(putGoal.getDetails()));
		assertTrue(patientEDIPI.equalsIgnoreCase(putGoal.getPatientId()));
		assertTrue(postedGoal.getUniqueId().equalsIgnoreCase(putGoal.getUniqueId()));

		assertTrue("".equalsIgnoreCase(putGoal.getNextActionStep()));
		
		PainGoals returnedPainGoalsB = client.getJerseyClient().resource(painGoalsURI).get(PainGoals.class);

		assertTrue(returnedPainGoalsB.size() == initialNumberOfPainGoals + 1);		
		
		client.getJerseyClient().resource(selfGoalURI).delete();
	}
	
	@Test
	public void testGoalEntries() {

		PainGoal painGoalA = new PainGoal();
		painGoalA.setDateStarted(DateHelper.getToday());
		painGoalA.setDetails("Details 5");
		painGoalA.setGoalType("Veteran");
		painGoalA.setName("Goal 5");
		painGoalA.setPatientId(patientEDIPI);
		painGoalA.setPercentComplete("40");
		painGoalA.setStatus(PainGoalStatus.PainGoalStatusActive);
		painGoalA.setTargetDate(DateHelper.get3MonthsFromNow());

		PainGoal postedGoal = client.getJerseyClient().resource(painGoalsURI).post(PainGoal.class, painGoalA);

		URI selfGoalEntriesURI = generatePainGoalEntriesURI(postedGoal.getUniqueId());

		PainGoalEntries gottenPainGoalEntries = client.getJerseyClient().resource(selfGoalEntriesURI).get(PainGoalEntries.class);
		
		assertNotNull(gottenPainGoalEntries);
		assertTrue(gottenPainGoalEntries.size() == 0);
		
		
		PainGoalEntries entries = new PainGoalEntries();
		PainGoalEntry entry = new PainGoalEntry();
		entry.setComments("Entry 5A");
		entry.setEntryDate(DateHelper.getToday());
		entry.setStatus(PainGoalEntryStatus.PainGoalEntryStatusActive);
		entries.add(entry);
		
		PainGoalEntry entry2 = new PainGoalEntry();
		entry2.setComments("Entry 5B");
		entry2.setEntryDate(DateHelper.getToday());
		entry2.setStatus(PainGoalEntryStatus.PainGoalEntryStatusActive);
		entries.add(entry2);
		
		PainGoalEntry entry3 = new PainGoalEntry();
		entry3.setComments("Entry 5C");
		entry3.setEntryDate(DateHelper.getToday());
		entry3.setStatus(PainGoalEntryStatus.PainGoalEntryStatusActive);
		entries.add(entry3);
		
		PainGoalEntry postedEntry = client.getJerseyClient().resource(selfGoalEntriesURI).post(PainGoalEntry.class, entry);
		PainGoalEntry postedEntry2 = client.getJerseyClient().resource(selfGoalEntriesURI).post(PainGoalEntry.class, entry2);
		PainGoalEntry postedEntry3 = client.getJerseyClient().resource(selfGoalEntriesURI).post(PainGoalEntry.class, entry3);
		
		assertNotNull(postedEntry.getUniqueId());
		assertNotNull(postedEntry2.getUniqueId());
		assertNotNull(postedEntry3.getUniqueId());
		
		PainGoalEntries gottenPainGoalEntriesAfterPost = client.getJerseyClient().resource(selfGoalEntriesURI).get(PainGoalEntries.class);
		assertNotNull(gottenPainGoalEntriesAfterPost);
		assertTrue(gottenPainGoalEntriesAfterPost.size() == 3);
		assertTrue(gottenPainGoalEntriesAfterPost.get(0).getComments().equalsIgnoreCase("Entry 5C"));
		assertTrue(gottenPainGoalEntriesAfterPost.get(1).getComments().equalsIgnoreCase("Entry 5B"));
		assertTrue(gottenPainGoalEntriesAfterPost.get(2).getComments().equalsIgnoreCase("Entry 5A"));
		
		entry.setComments("Entry 5A modified");
		entry.setStatus(PainGoalEntryStatus.PainGoalEntryStatusComplete);
		URI selfGoalEntryURI = generatePainGoalEntriesSelfURI(postedGoal.getUniqueId(), postedEntry.getUniqueId());
		PainGoalEntry postedModifiedEntry = client.getJerseyClient().resource(selfGoalEntryURI).put(PainGoalEntry.class, entry);
		assertTrue(postedModifiedEntry.getUniqueId().equalsIgnoreCase(postedEntry.getUniqueId()));
		
		PainGoalEntries gottenPainGoalEntriesAfterPut = client.getJerseyClient().resource(selfGoalEntriesURI).get(PainGoalEntries.class);
		assertTrue(gottenPainGoalEntriesAfterPut.size() == 3);
		assertTrue(gottenPainGoalEntriesAfterPut.get(2).getComments().equalsIgnoreCase("Entry 5A modified"));
		assertTrue(gottenPainGoalEntriesAfterPut.get(2).getStatus().equals(PainGoalEntryStatus.PainGoalEntryStatusComplete));
		
		URI selfGoalURI = postedGoal.getSelfUri();
		PainGoal returnedPainGoal = client.getJerseyClient().resource(selfGoalURI).get(PainGoal.class);
		assertTrue(returnedPainGoal.getNextActionStep() + " should be " + "Entry 5C", returnedPainGoal.getNextActionStep().equalsIgnoreCase("Entry 5C"));
		
		URI selfGoalEntryToDeleteURI = generatePainGoalEntriesSelfURI(postedGoal.getUniqueId(), postedEntry3.getUniqueId());
		client.getJerseyClient().resource(selfGoalEntryToDeleteURI).delete();
		PainGoalEntries gottenPainGoalEntriesAfterDelete = client.getJerseyClient().resource(selfGoalEntriesURI).get(PainGoalEntries.class);
		assertTrue(gottenPainGoalEntriesAfterDelete.size() == 2);


		PainGoal returnedPainGoalAfterEntryUpdate = client.getJerseyClient().resource(selfGoalURI).get(PainGoal.class);
		assertTrue(returnedPainGoalAfterEntryUpdate.getNextActionStep().equalsIgnoreCase("Entry 5B"));
		
		
		client.getJerseyClient().resource(selfGoalURI).delete();		
	}
	
	private URI generatePainGoalsURI() {
		String goalsURI = String.format("%s/patient/EDIPI/%s/pain-goals", IntegrationTestConfiguration.getMobileHealthServerBaseUri(), patientEDIPI);
		return UriBuilder.fromUri(goalsURI).build();		
	}
	
	private URI generatePainGoalEntriesURI(String goalId) {
		String goalsURI = String.format("%s/patient/EDIPI/%s/pain-goals/system/sed/id/%s/pain-goal-entries", IntegrationTestConfiguration.getMobileHealthServerBaseUri(), patientEDIPI, goalId);
		return UriBuilder.fromUri(goalsURI).build();		
	}
	
	private URI generatePainGoalEntriesSelfURI(String goalId, String entryId) {
		String goalsURI = String.format("%s/patient/EDIPI/%s/pain-goals/system/sed/id/%s/pain-goal-entries/id/%s", IntegrationTestConfiguration.getMobileHealthServerBaseUri(), patientEDIPI, goalId, entryId);
		return UriBuilder.fromUri(goalsURI).build();		
	}
}
