package com.agilex.healthcare.mobilehealthplatform.restservice.notification;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.net.URI;

import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriBuilder;

import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.Test;
import org.springframework.http.HttpStatus;

import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.AtomLink;
import com.agilex.healthcare.mobilehealthplatform.domain.AtomLinks;
import com.agilex.healthcare.mobilehealthplatform.domain.DeviceRegistration;
import com.agilex.healthcare.mobilehealthplatform.domain.NotificationUsers;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.testutility.IntegrationTestConfiguration;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.ModeHelper;
import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.WebResource;

public class NotificationUserResourceTest {

	private static MobileHealthClientTestVersion restClientForProvider;

	private static MobileHealthClientTestVersion restClientForVeteran;

	private static final String resourceUri = IntegrationTestConfiguration.getMobileHealthServerBaseUri() + "/notification-service/users";
	private static final String auditBaseUri = IntegrationTestConfiguration.getMobileHealthServerBaseUri() + "/notification-service/audits/auditLogs";
	private static final String defaultDeviceToken = "41b37c7b9b6c0a6567a0c0439c662a14aeb3ae4b0072dc6985f279dd8a3dab83";

	@BeforeClass
	public static void setUp() {
		Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev"));
		restClientForProvider = TestHelper.createMobileHealthClient("zztest.staff01", "pass", "basic");
		restClientForVeteran = TestHelper.createMobileHealthClient("zztest.patient26", "pass", "oauth");
		
		associateDeviceTokenWithUser("PATID26");
	}

	@AfterClass
	public static void destroy() {
		if (restClientForProvider != null) {
			restClientForProvider.getJerseyClient().destroy();
		}
		if (restClientForVeteran != null) {
			restClientForVeteran.getJerseyClient().destroy();
		}

	}

	private static void associateDeviceTokenWithUser(String userId) {
		DeviceRegistration registration = new DeviceRegistration(userId, defaultDeviceToken, userId + "first", userId + "last");
		URI uri = createDeviceUri(userId, NotificationUserResourceTest.defaultDeviceToken);
		restClientForVeteran.getJerseyClient().resource(uri).type(MediaType.APPLICATION_XML).accept("application/xml").post(DeviceRegistration.class, registration);
	}

	@Test
	public void searchByLastName() {
		String lastname = IntegrationTestConfiguration.getPatient     WithSSN().getLastName();

		NotificationUsers notificationUsers = searchNotificationUsers("patient26");

		assertEquals(1, notificationUsers.size());
		assertEquals(lastname.toLowerCase(), notificationUsers.get(0).getLastName().toLowerCase());
	}

	@Test
	public void notificationUriIsReturned() {
		String lastname = IntegrationTestConfiguration.getPatient     WithSSN().getLastName();

		NotificationUsers notificationUsers = searchNotificationUsers("patient26");

		AtomLinks link = notificationUsers.get(0).getLink();

		boolean notificationLinkFound = false;
		for (AtomLink atomLink : link) {
			if (atomLink.getTitle().equalsIgnoreCase("notifications")) {
				assertEquals(IntegrationTestConfiguration.getMobileHealthServerBaseUri() + "/notification-service/user" + "/id/EDIPI/PATID26/notifications", atomLink.getHref().toString());
				notificationLinkFound = true;
			}
		}
		assertTrue(notificationLinkFound);
	}

	@Test
	public void validateNotificationAuditLogUriIsReturned() {
		String lastname = IntegrationTestConfiguration.getPatient     WithSSN().getLastName();

		NotificationUsers notificationUsers = searchNotificationUsers("patient26");

		AtomLinks link = notificationUsers.get(0).getLink();

		boolean notificationLinkFound = false;
		for (AtomLink atomLink : link) {
			if (atomLink.getTitle().equalsIgnoreCase("notifications-auditLogs")) {
				assertEquals(auditBaseUri + "/user/id/EDIPI/PATID26", atomLink.getHref().toString());
				notificationLinkFound = true;
			}
		}
		assertTrue(notificationLinkFound);
	}

	@Test
	public void searchByFirstName() {
		String firstname = IntegrationTestConfiguration.getPatient     WithSSN().getFirstName();

		NotificationUsers notificationUsers = searchNotificationUsers("," + firstname);

		assertEquals(1, notificationUsers.size());
		assertEquals(firstname.toLowerCase(), notificationUsers.get(0).getFirstName().toLowerCase());
	}

	@Test
	public void searchByLastInit() {
		String lastname = IntegrationTestConfiguration.getPatient     WithSSN().getLastName();
		String expectedLastInit = lastname.substring(0, 1);

		NotificationUsers notificationUsers = searchNotificationUsers(lastname);
		String actualLastInit = notificationUsers.get(0).getLastName().substring(0, 1);

		assertEquals(1, notificationUsers.size());
		assertEquals(expectedLastInit.toLowerCase(), actualLastInit.toLowerCase());
	}

	@Test
	public void searchBySSN() {
		Patient testPatient = IntegrationTestConfiguration.getPatient     WithSSN();
		String ssn = testPatient.getSsn();

		String last4ssn = ssn.substring(ssn.length() - 4, ssn.length());
		NotificationUsers notificationUsers = searchNotificationUsers(last4ssn);
		String actualSSN = notificationUsers.get(0).getSsn().replaceAll("-", "");

		assertEquals(1, notificationUsers.size());
		assertEquals(ssn, actualSSN);
	}

	@Test
	public void validateSearchReturnsZeroResultsWhenUserDoesNotExist() {
		NotificationUsers notificationUsers = searchNotificationUsers("Invalid");
		assertEquals(0, notificationUsers.size());
	}

	private NotificationUsers searchNotificationUsers(String searchString) {
		WebResource notificationUserResource = restClientForProvider.getJerseyClient().resource(resourceUri);
		notificationUserResource = notificationUserResource.queryParam("searchstring", searchString).queryParam("siteCode", "688");
		NotificationUsers notificationUsers = notificationUserResource.get(NotificationUsers.class);
		return notificationUsers;
	}

	@Test
	public void searchByNull() {
		ClientResponse response = restClientForProvider.getJerseyClient().resource(resourceUri).get(ClientResponse.class);
		assertEquals(HttpStatus.BAD_REQUEST.value(), response.getStatus());
	}

	private static URI createDeviceUri(String userId, String deviceToken) {
		String deviceRegistrationUriAsString = String.format("%s/notification-service/user/id/EDIPI/%s/device-registrations/", IntegrationTestConfiguration.getMobileHealthServerBaseUri(), userId);
		return UriBuilder.fromUri(deviceRegistrationUriAsString).build();
	}
}
