package com.agilex.healthcare.mobilehealthplatform.restservice.notification;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.net.URI;
import java.util.Date;

import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriBuilder;

import org.junit.After;
import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.springframework.http.HttpStatus;

import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.DeviceRegistration;
import com.agilex.healthcare.mobilehealthplatform.domain.Notification;
import com.agilex.healthcare.mobilehealthplatform.domain.Notifications;
import com.agilex.healthcare.testutility.IntegrationTestConfiguration;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.ModeHelper;
import com.sun.jersey.api.client.ClientResponse;

public class NotificationResourceTest {

	private static final String TEST_USER_ID = "D123401";
	private static final String TEST_DEVICE_TOKEN = "0a701d904f7a06b52904cefe3ccc3df8f0a2e88bea9ab62939b5d3b2dd52710b";
	private Notification notification1;
	private Notification notification2;
	private Notification notification3;
	private static MobileHealthClientTestVersion client;

	@BeforeClass
	public static void checkWhetherTestsAreRunningInCIHMode() {
		Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev"));
		client = TestHelper.createMobileHealthClient("zztest.patient01", "pass", "oauth");
	}
	
	@AfterClass
	public static void destroy() {
		if (client != null) {
			client.getJerseyClient().destroy();
			client = null;
		}
	}

	@Before
	public void setUp() {
		notification1 = createAndSaveNotification(TEST_USER_ID, "Notification from yesterday", DateHelper.getYesterday());
		notification2 = createAndSaveNotification(TEST_USER_ID, "Notification from today", new Date());
		notification3 = createAndSaveNotification(TEST_USER_ID, "Notification from tomorrow", DateHelper.getTomorrow());		
	}
	
	@After
	public void tearDown() {
		URI deleteUri1 = createSpecificNotificationUri(TEST_USER_ID, notification1.getNotificationId());
		client.getJerseyClient().resource(deleteUri1).delete();

		URI deleteUri2 = createSpecificNotificationUri(TEST_USER_ID, notification2.getNotificationId());
		client.getJerseyClient().resource(deleteUri2).delete();

		URI deleteUri3 = createSpecificNotificationUri(TEST_USER_ID, notification3.getNotificationId());
		client.getJerseyClient().resource(deleteUri3).delete();		
	}
	
	@Test
	public void verifyPostToSpecificUser() {
		Notification notification = new Notification();
		notification.setBody("verify post unit test for notification resource");
		associateDeviceTokenWithUser(TEST_USER_ID);
		URI uri = createSpecificNotificationsUri(TEST_USER_ID);
		Notification savedNotification = client.getJerseyClient().resource(uri).type(MediaType.APPLICATION_XML).accept("application/xml").post(Notification.class, notification);
		assertNotNull(savedNotification);

		URI deleteUri = createSpecificNotificationUri(TEST_USER_ID, savedNotification.getNotificationId());
		client.getJerseyClient().resource(deleteUri).delete();
	}
	
	@Test
	public void verifyLongPostToSpecificUser() {
		Notification notification = new Notification();
		notification.setBody("this is a long message containing 387 characters.  it should be truncated. 1234567890123456789012345678901234567890123456789012345678901234567890" +
				"1234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890" +
				"1234567890123456789012345678901234567890123456789012345678901234567890endofmessage");
		notification.setUserId(TEST_USER_ID);
		associateDeviceTokenWithUser(TEST_USER_ID);
		URI uri = createSpecificNotificationsUri(TEST_USER_ID);
		Notification savedNotification = client.getJerseyClient().resource(uri).type(MediaType.APPLICATION_XML).accept("application/xml").post(Notification.class, notification);
		assertNotNull(savedNotification);

		URI deleteUri = createSpecificNotificationUri(TEST_USER_ID, savedNotification.getNotificationId());
		client.getJerseyClient().resource(deleteUri).delete();
	}

	@Test
	public void verifyGetAllForSpecificUser() {
		URI uri = createSpecificNotificationsUri(TEST_USER_ID);
		Notifications retrieved = client.getJerseyClient().resource(uri).get(Notifications.class);
		assertTrue(retrieved.size() > 0);
	}

	@Test
	public void verifyGetTodaysNotifications() {
		URI uri = createTodayNotificationsUri(TEST_USER_ID);
		Notifications retrieved = client.getJerseyClient().resource(uri).get(Notifications.class);
		assertTrue(retrieved.size()>0);
	}
	

	@Test
	public void verifyGetSpecificNotification() {
		URI uri = createSpecificNotificationUri(TEST_USER_ID, notification1.getNotificationId());
		Notification retrieved = client.getJerseyClient().resource(uri).get(Notification.class);

		assertEquals("Notification from yesterday", retrieved.getBody());
		assertEquals(TEST_USER_ID, retrieved.getUserId());
	}

	@Test
	public void verifyDeleteSpecificNotification() {
		associateDeviceTokenWithUser(TEST_USER_ID);
		Notification savedNotification = createAndSaveNotification(TEST_USER_ID, "Specific Notification");

		URI uri = createSpecificNotificationUri(TEST_USER_ID, savedNotification.getNotificationId());
		Notification retrieved = client.getJerseyClient().resource(uri).get(Notification.class);
		assertEquals(savedNotification.getNotificationId(), retrieved.getNotificationId());

		client.getJerseyClient().resource(uri).delete();

		ClientResponse response = client.getJerseyClient().resource(uri).get(ClientResponse.class);
		assertEquals(HttpStatus.NOT_FOUND.value(), response.getStatus());
	}

	private Notification createAndSaveNotification(String userId, String body) {
		Notification notification = createNotification(userId, body, new Date());
		URI uri = createSpecificNotificationsUri(userId);
		Notification savedNotification = client.getJerseyClient().resource(uri).type(MediaType.APPLICATION_XML).accept("application/xml").post(Notification.class, notification);

		return savedNotification;
	}

	private Notification createAndSaveNotification(String userId, String body, Date date) {
		associateDeviceTokenWithUser(TEST_USER_ID);
		Notification notification = createNotification(userId, body, date);
		URI uri = createSpecificNotificationsUri(userId);
		Notification savedNotification = client.getJerseyClient().resource(uri).type(MediaType.APPLICATION_XML).accept("application/xml").post(Notification.class, notification);

		return savedNotification;
	}
	
	private Notification createNotification(String userId, String body, Date date) {
		Notification notification = new Notification();
		notification.setBody(body);
		notification.setUserId(userId);
		notification.setActiveFlag(true);
		notification.setDate(date);
		return notification;
	}

	private void associateDeviceTokenWithUser(String userId) {
		DeviceRegistration registration = new DeviceRegistration(userId, TEST_DEVICE_TOKEN, userId+"first", userId+"last");

		URI uri = createDeviceUri(userId, TEST_DEVICE_TOKEN);
		client.getJerseyClient().resource(uri).type(MediaType.APPLICATION_XML).accept("application/xml").post(DeviceRegistration.class, registration);
	}

	private URI createDeviceUri(String userId, String deviceToken) {
		String deviceRegistrationUriAsString = String.format("%s/notification-service/user/id/EDIPI/%s/device-registrations/", IntegrationTestConfiguration.getMobileHealthServerBaseUri(), userId);
		return UriBuilder.fromUri(deviceRegistrationUriAsString).build();
	}

	private URI createSpecificNotificationsUri(String userId) {
		String notificationsUriAsString = String.format("%s/notification-service/user/id/EDIPI/%s/notifications", IntegrationTestConfiguration.getMobileHealthServerBaseUri(), userId);
		return UriBuilder.fromUri(notificationsUriAsString).build();
	}

	private URI createSpecificNotificationUri(String userId, String notificationId) {
		String notificationsUriAsString = String.format("%s/notification-service/user/id/EDIPI/%s/notifications/id/%s", IntegrationTestConfiguration.getMobileHealthServerBaseUri(), userId, notificationId);
		return UriBuilder.fromUri(notificationsUriAsString).build();
	}

	private URI createTodayNotificationsUri(String userId) {
		String notificationsUriAsString = String.format("%s/notification-service/user/id/EDIPI/%s/notifications/today", IntegrationTestConfiguration.getMobileHealthServerBaseUri(), userId);
		return UriBuilder.fromUri(notificationsUriAsString).build();
	}
}
