package com.agilex.healthcare.mobilehealthplatform.restservice.notification;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;

import java.net.URI;
import java.util.Date;

import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriBuilder;

import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.Ignore;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.AuditLog;
import com.agilex.healthcare.mobilehealthplatform.domain.AuditLogs;
import com.agilex.healthcare.mobilehealthplatform.domain.DeviceRegistration;
import com.agilex.healthcare.mobilehealthplatform.domain.Notification;
import com.agilex.healthcare.mobilehealthplatform.domain.Notifications;
import com.agilex.healthcare.mobilehealthplatform.domain.code.AuditLogCode;
import com.agilex.healthcare.testutility.IntegrationTestConfiguration;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.ModeHelper;

@Ignore
public class NotificationAuditLogResourceTest {
	private static final String VALID_DEVICE_TOKEN = "0a701d904f7a06b52904cefe3ccc3df8f0a2e88bea9ab62939b5d3b2dd52710b";

	@BeforeClass
	public static void checkWhetherTestsAreRunningInCIHMode() {
		Assume.assumeTrue(ModeHelper.isMode("dev"));
	}
		
	@Test
	public void verifyNotificationGeneratesAuditSavedWithValidToken() {
		int expectedAudits = 1;
		String userId = "D123401";
		MobileHealthClientTestVersion client = TestHelper.createMobileHealthClient("zztest.patient01", "pass", "oauth");
		createAndSaveNotification(client, userId, "Audit Log Test with valid token", new Date(), VALID_DEVICE_TOKEN);
		
		AuditLogs auditLogs = fetchAuditLogs(client, userId);
		assertEquals(expectedAudits, auditLogs.size());
	}
	
	@Test
	public void verifyRetrieveSingleAuditLog() {
		int expectedAudits = 1;
		String userId = "PATID26";
		MobileHealthClientTestVersion client = TestHelper.createMobileHealthClient("zztest.patient26", "pass", "oauth");
		createAndSaveNotification(client, userId, "Audit Log Test with valid token", new Date(), VALID_DEVICE_TOKEN);
		
		AuditLogs auditLogs = fetchAuditLogs(client, userId);
		assertEquals(expectedAudits, auditLogs.size());
		URI auditSelfUri = auditLogs.get(0).getSelfUri();
		assertEquals(createSpecificAuditLogUri(userId, auditLogs.get(0).getId()).toString(), auditSelfUri.toString());
	}

	@Test
	public void verifyNotificationGeneratesAuditSaveAndRejectedWithInvalidToken() {
		int expectedAudits = 2; // audit for save & rejected
		String userId = "PATID6";
		MobileHealthClientTestVersion client = TestHelper.createMobileHealthClient("tom.smith", "pass", "oauth");
		String invalidTokenWithValidLength = "fffffd904f7a06b52904cefe3ccc3df8f0a2e88bea9ab62939b5d3b2dd52710b";
		createAndSaveNotification(client, userId, "Audit Log Test with invalid token", new Date(), invalidTokenWithValidLength);
		
		AuditLogs auditLogs = fetchAuditLogs(client, userId);
		assertEquals(expectedAudits, auditLogs.size());
	}
	
	@Test
	public void verifyNotificationSaveAndDeleteGeneratesAuditLogs() {
		int expectedAudits = 3; // audit for save, retrieve, and delete
		String userId = "PATID7";
		MobileHealthClientTestVersion client = TestHelper.createMobileHealthClient("ashley.simpson", "pass", "oauth");
		Notification savedNotification = createAndSaveNotification(client, userId, "Audit Log Notification To Delete", new Date(), VALID_DEVICE_TOKEN);
		
		URI uri = createSpecificNotificationUri(userId, savedNotification.getNotificationId());
		client.getJerseyClient().resource(uri).delete();
		
		AuditLogs auditLogs = fetchAuditLogs(client, userId);
		assertEquals(expectedAudits, auditLogs.size());
		
	}
	
	@Test
	public void verifyRetrieveOfAuditLogFromSelfLink() {
		String userId = "D123401";

		MobileHealthClientTestVersion client = TestHelper.createMobileHealthClient("zztest.patient01", "pass", "oauth");

		createAndSaveNotification(client, userId, "Verify Audit Log Self Uri", new Date(), VALID_DEVICE_TOKEN);
		
		AuditLogs auditLogs = fetchAuditLogs(client, userId);
		AuditLog firstLog = auditLogs.get(0);
		URI firstLogUri = firstLog.getSelfUri();
		AuditLog retrievedAuditLog = client.getJerseyClient().resource(firstLogUri).get(AuditLog.class);
		assertNotNull(retrievedAuditLog);
		assertEquals(AuditLogCode.NOTIFICATION, retrievedAuditLog.getType());
	}
	
	@Test
	public void verifyNotificationSaveAndRetrieveAuditsForSingleNotification() {
		int expectedAudits = 2; // audit for save and retrieve
		String userId = "D123401";
		
		MobileHealthClientTestVersion client = TestHelper.createMobileHealthClient("zztest.patient01", "pass", "oauth");
		Notification savedNotification = createAndSaveNotification(client, userId, "Audit Log Notification For Single Retrieve", new Date(), VALID_DEVICE_TOKEN);
		
		URI uri = createSpecificNotificationUri(userId, savedNotification.getNotificationId());
		client.getJerseyClient().resource(uri).get(Notification.class);
		
		AuditLogs auditLogs = fetchAuditLogs(client, userId);
		assertEquals(expectedAudits, auditLogs.size());
	}
	
	@Test
	public void verifyNotificationSaveAndRetrieveAuditsForMultipleNotifications() {
		
		String userId = "D123401";
		
		MobileHealthClientTestVersion client = TestHelper.createMobileHealthClient("zztest.patient01", "pass", "oauth");
		
		createAndSaveNotification(client, userId, "Audit Log Notification For Multi Retrieve", new Date(), VALID_DEVICE_TOKEN);
		createAndSaveNotification(client, userId, "Audit Log Notification For Multi Retrieve", new Date(), VALID_DEVICE_TOKEN);
		URI notificationsUri = createSpecificNotificationsUri(userId);
		client.getJerseyClient().resource(notificationsUri).get(Notifications.class);
		
		int expectedAudits = 4; // 2 saves and 2 retrieves
		AuditLogs auditLogs = fetchAuditLogs(client, userId);
		assertEquals(expectedAudits, auditLogs.size());
	}
	
	private void associateDeviceTokenWithUser(MobileHealthClientTestVersion client, String userId, String token) {
		DeviceRegistration registration = new DeviceRegistration(userId, token, userId + "first", userId + "last");
		String deviceRegistrationUriAsString = String.format("%s/notification-service/user/id/EDIPI/%s/device-registrations/",
				IntegrationTestConfiguration.getMobileHealthServerBaseUri(), userId);
		URI uri = UriBuilder.fromUri(deviceRegistrationUriAsString).build();
		client.getJerseyClient().resource(uri).type(MediaType.APPLICATION_XML).accept("application/xml").post(DeviceRegistration.class, registration);
	}
	
	private Notification createAndSaveNotification(MobileHealthClientTestVersion client, String userId, String body, Date date, String token) {
		associateDeviceTokenWithUser(client, userId, token);
		Notification notification = createNotification(userId, body, date);

		String notificationsUriAsString = String.format("%s/notification-service/user/id/EDIPI/%s/notifications",
				IntegrationTestConfiguration.getMobileHealthServerBaseUri(), userId);
		URI uri = UriBuilder.fromUri(notificationsUriAsString).build();

		return client.getJerseyClient().resource(uri).type(MediaType.APPLICATION_XML).accept("application/xml").post(Notification.class, notification);
	}

	private Notification createNotification(String userId, String body, Date date) {
		Notification notification = new Notification();
		notification.setBody(body);
		notification.setUserId(userId);
		notification.setActiveFlag(true);
		notification.setDate(date);
		return notification;
	}
	
	private AuditLogs fetchAuditLogs(MobileHealthClientTestVersion client, String userId) {
		URI auditLogUri = createAuditLogUri(userId);
		AuditLogs auditLogs = client.getJerseyClient().resource(auditLogUri).get(AuditLogs.class);
		return auditLogs;
	}
	
	private URI createAuditLogUri(String userId) {
		String auditLogUriAsString = String.format("%s/notification-service/audits/auditLogs/user/id/EDIPI/%s", IntegrationTestConfiguration.getMobileHealthServerBaseUri(), userId);
		URI auditLogUri = UriBuilder.fromUri(auditLogUriAsString).build();
		return auditLogUri;
	}
	
	private URI createSpecificAuditLogUri(String userId, String auditLogId) {
		String auditLogUriAsString = String.format("%s/notification-service/audits/auditLogs/user/id/EDIPI/%s/auditLog/%s", IntegrationTestConfiguration.getMobileHealthServerBaseUri(), userId, auditLogId);
		URI auditLogUri = UriBuilder.fromUri(auditLogUriAsString).build();
		return auditLogUri;
	}
	
	private URI createSpecificNotificationUri(String userId, String notificationId) {
		String notificationsUriAsString = String.format("%s/notification-service/user/id/EDIPI/%s/notifications/id/%s", IntegrationTestConfiguration.getMobileHealthServerBaseUri(), userId, notificationId);
		return UriBuilder.fromUri(notificationsUriAsString).build();
	}
	
	private URI createSpecificNotificationsUri(String userId) {
		String notificationsUriAsString = String.format("%s/notification-service/user/id/EDIPI/%s/notifications", IntegrationTestConfiguration.getMobileHealthServerBaseUri(), userId);
		return UriBuilder.fromUri(notificationsUriAsString).build();
	}
}