package com.agilex.healthcare.mobilehealthplatform.restservice.notification;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;

import java.net.URI;
import java.util.UUID;

import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriBuilder;

import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Ignore;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.DeviceRegistration;
import com.agilex.healthcare.mobilehealthplatform.domain.DeviceRegistrations;
import com.agilex.healthcare.testutility.IntegrationTestConfiguration;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.ModeHelper;

public class DeviceRegistrationResourceTest {
	private static MobileHealthClientTestVersion restClientForGallowYounger;
	private static MobileHealthClientTestVersion restClientForAldie     ;

	private String userId1;
	private String userId2;
	String deviceToken1;
	String deviceToken2;
	private DeviceRegistration deviceRegistration1;
	private DeviceRegistration deviceRegistration2;

	@BeforeClass
	public static void checkWhetherTestsAreRunningInCIHMode() {
		Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev"));
		restClientForGallowYounger = TestHelper.createMobileHealthClient("zztest.patient01", "pass", "oauth");
		restClientForAldie      = TestHelper.createMobileHealthClient("zztest.patient26", "pass", "oauth");
	}
	
	@AfterClass
	public static void classTearDown() {
		if (restClientForGallowYounger != null) {
			restClientForGallowYounger.getJerseyClient().destroy();
		}
		if (restClientForAldie      != null) {
			restClientForAldie     .getJerseyClient().destroy();
		}
	}
	
	@Before
	public void setUp() {
		userId1 = "D123401";
		userId2 = "PATID26";

		deviceToken1 = UUID.randomUUID().toString();
		deviceToken2 = UUID.randomUUID().toString();
		
		deviceRegistration1 = createAndSaveDeviceRegistration(restClientForGallowYounger, userId1, deviceToken1);
		deviceRegistration2 = createAndSaveDeviceRegistration(restClientForAldie     , userId2, deviceToken2);
	}
	
	@Test
	public void verifyCreateNewToken() {
		assertNotNull(deviceRegistration1);
		assertEquals(userId1, deviceRegistration1.getUserId());
		assertEquals(deviceToken1, deviceRegistration1.getDeviceToken());
	}
	
	@Test
	public void verifyGetSingle() {
		DeviceRegistration retrieved = restClientForGallowYounger.getJerseyClient().resource(deviceRegistration1.getSelfUri()).get(DeviceRegistration.class);
		assertNotNull(retrieved);
		assertEquals(deviceRegistration1.getDeviceRegistrationId(), retrieved.getDeviceRegistrationId());
		assertEquals(deviceRegistration1.getDeviceToken(), retrieved.getDeviceToken());
	}
	
	@Test
	public void verifyGetDeviceRegistration() {
		DeviceRegistration retrieved = restClientForAldie     .getJerseyClient().resource(createDeviceTokenUri(userId2, deviceToken2)).get(DeviceRegistration.class);
		assertNotNull(retrieved);
		assertEquals(deviceRegistration2.getDeviceRegistrationId(), retrieved.getDeviceRegistrationId());
		assertEquals(deviceRegistration2.getDeviceToken(), retrieved.getDeviceToken());
	}

	//FIXME: Remove this ignore
	@Ignore
	public void verifyDoesNotDuplicate() {
		URI uri = createDeviceUri(userId2);
		DeviceRegistrations retrieved = restClientForAldie     .getJerseyClient().resource(uri).get(DeviceRegistrations.class);
		assertEquals(1, retrieved.size());
		
		createAndSaveDeviceRegistration(restClientForAldie     , userId2, deviceToken2);
		DeviceRegistrations retrievedLater = restClientForAldie     .getJerseyClient().resource(uri).get(DeviceRegistrations.class);
		assertEquals(1, retrievedLater.size());
	}
	
	@Test
	public void verifyPut() {
		URI uri = createDeviceUri(userId1);
		
		deviceRegistration1.setOptedIn(false);
		
		DeviceRegistration updated = restClientForGallowYounger.getJerseyClient().resource(uri).type(MediaType.APPLICATION_XML).accept("application/xml").put(DeviceRegistration.class, deviceRegistration1);
		assertNotNull(updated);
		assertEquals(false, updated.isOptedIn());
		assertEquals(deviceRegistration1.getDeviceToken(), updated.getDeviceToken());
		assertEquals(deviceRegistration1.getDeviceRegistrationId(), updated.getDeviceRegistrationId());
		
		DeviceRegistration retrieved = restClientForGallowYounger.getJerseyClient().resource(deviceRegistration1.getSelfUri()).get(DeviceRegistration.class);
		assertNotNull(retrieved);
		assertEquals(false, retrieved.isOptedIn());
		assertEquals(deviceRegistration1.getDeviceToken(), retrieved.getDeviceToken());
	}
	
	@Test
	public void verifyDeleteToken() {
		String userId = "D123401";
		createAndSaveDeviceRegistration(restClientForGallowYounger, userId, deviceToken1);
		URI uri = createDeviceUri(userId);
		
		DeviceRegistrations deviceRegistrations = restClientForGallowYounger.getJerseyClient().resource(uri).get(DeviceRegistrations.class);
		assertEquals(1, deviceRegistrations.size());
		
		deleteDeviceRegistration(restClientForGallowYounger, userId, deviceToken1);

		deviceRegistrations = restClientForGallowYounger.getJerseyClient().resource(uri).get(DeviceRegistrations.class);
		assertEquals(0, deviceRegistrations.size());
	}
	
	@Test
	public void verifyOptBackInAfterOptOut() {
		URI uri = createDeviceUri(userId1);
		
		deviceRegistration1.setOptedIn(false);
		
		DeviceRegistration optedOutRegistration = restClientForGallowYounger.getJerseyClient().resource(uri).type(MediaType.APPLICATION_XML).accept("application/xml").put(DeviceRegistration.class, deviceRegistration1);
		assertEquals(false, optedOutRegistration.isOptedIn());
		
		DeviceRegistration optedInRegistration = optedOutRegistration;
		optedInRegistration.setOptedIn(true);
		
		optedInRegistration = restClientForGallowYounger.getJerseyClient().resource(uri).type(MediaType.APPLICATION_XML).accept("application/xml").put(DeviceRegistration.class, optedInRegistration);
		assertEquals(true, optedInRegistration.isOptedIn());
	}
	
	@Test
	public void verifyAllTokensOptedOut() {
		String userId = "D123401";
		createAndSaveDeviceRegistration(restClientForGallowYounger, userId, "dont care");
		createAndSaveDeviceRegistration(restClientForGallowYounger, userId, "dont care 2");
		createAndSaveDeviceRegistration(restClientForGallowYounger, userId, "dont care 3");
		
		DeviceRegistration registration = new DeviceRegistration(userId, "only the user id matters");
		registration.setOptedIn(false);
		URI uri = createDeviceUri(userId); 
		
		restClientForGallowYounger.getJerseyClient().resource(uri).type(MediaType.APPLICATION_XML).accept("application/xml").put(DeviceRegistration.class, registration);
		
		DeviceRegistrations registrations = restClientForGallowYounger.getJerseyClient().resource(uri).get(DeviceRegistrations.class);
		assertNotNull(registrations);
		assertEquals(0, registrations.size());
	}
	
	private URI createDeviceUri(String userId) {
		String deviceRegistrationUriAsString = String.format("%s/notification-service/user/id/EDIPI/%s/device-registrations", IntegrationTestConfiguration.getMobileHealthServerBaseUri(), userId);
		return UriBuilder.fromUri(deviceRegistrationUriAsString).build();
	}
	
	private URI createDeviceTokenUri(String userId, String deviceToken) {
		String deviceRegistrationUriAsString = String.format("%s/notification-service/user/id/EDIPI/%s/device-registrations/token/%s", IntegrationTestConfiguration.getMobileHealthServerBaseUri(), userId, deviceToken);
		return UriBuilder.fromUri(deviceRegistrationUriAsString).build();
	}
		
	private DeviceRegistration createAndSaveDeviceRegistration(MobileHealthClientTestVersion client, String userId, String deviceToken) {
		DeviceRegistration deviceRegistration = new DeviceRegistration(userId, deviceToken);
		URI uri = createDeviceUri(userId);
		return client.getJerseyClient().resource(uri).type(MediaType.APPLICATION_XML).accept("application/xml").post(DeviceRegistration.class, deviceRegistration);
	}
	
	private void deleteDeviceRegistration(MobileHealthClientTestVersion client, String userId, String deviceToken) {
		URI uri = createDeviceTokenUri(userId, deviceToken);
		client.getJerseyClient().resource(uri).delete();
	}
}
