package com.agilex.healthcare.mobilehealthplatform.restservice.mygoals;

import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.net.URI;
import java.util.Date;

import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriBuilder;

import org.joda.time.DateTime;
import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.LinkTitles;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.Patients;
import com.agilex.healthcare.testutility.IntegrationTestConfiguration;
import com.agilex.healthcare.testutility.PatientLoader;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.ModeHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.WebResource;


public class ProviderPatientMygoalsResourceTest {
	private static PatientIdentifier automatedTestUserId = new PatientIdentifier("EDIPI", "PATID66");
	Patient automatedTestUser = PatientLoader.loadPatientAsProvider(automatedTestUserId, "zztest.staff01");
	DateTime today = new DateTime();
	Date twoWeeksAgo = today.minusDays(14).toDate();
	
	private boolean defaultDebug = true;
	private WebResource cachedResource = null;
	
	private Client jerseyClient;

	protected WebResource getResourceHandle(URI patientMygoalsUri) {
		if (cachedResource == null) {
			if (defaultDebug)
				System.out.println("initializing resource with URI " + patientMygoalsUri);
			cachedResource = jerseyClient.resource(patientMygoalsUri);
		}
		return cachedResource;
	}


	@BeforeClass
	public static void checkWhetherTestsAreRunningInCIHMode() {
		Assume.assumeTrue(ModeHelper.isMode("va-provider-dev"));
	}
	
	@AfterClass
	public static void destroyClient() {
		
	}

	@Test
	public void getPatients() {
		Patients patientResults = retrievePatients("patient26", "zztest");
		assertNotNull(patientResults);
		assertTrue(patientResults.size() > 0);
	}
	
	@Test
	public void getMygoalsReportForTestPatient() {
		String result = retrieveMygoalsReport();
		assertNotNull(result);
		assertTrue(result.startsWith("%PDF"));
	}

	private Patients retrievePatients(String lastName, String firstName) {
		String patientsUri = getPatientResourcePath();
		URI patientSearchUri = UriBuilder.fromUri(patientsUri).build();
	
		Patient patientSearchCriteria = getPatientSearchCriteria(lastName, firstName);
		jerseyClient = TestHelper.createMobileHealthClient().getJerseyClient();
		
		Patients patients = null;
		if (patientSearchCriteria != null) {
			com.sun.jersey.api.client.WebResource resourceHandle = getResourceHandle(patientSearchUri);
			resourceHandle = addPatientSearchQueryParams(resourceHandle, patientSearchCriteria);
			patients = resourceHandle.type(MediaType.APPLICATION_JSON).get(Patients.class);
		} else {
			patients = new Patients();
		}
		jerseyClient.destroy();
		jerseyClient = null;
	
		return patients;
	}

	private Patient getPatientSearchCriteria(String lastname, String firstname) {
		Patient prototype = new Patient();
		prototype.setLastName(lastname);
		prototype.setFirstName(firstname);	
		return prototype;
	}

	private WebResource addPatientSearchQueryParams(WebResource resource, Patient prototype) {
		resource = addPatientSearchQueryParam(resource, "lastname", prototype.getLastName());
		resource = addPatientSearchQueryParam(resource, "firstname", prototype.getFirstName());
		resource = addPatientSearchQueryParam(resource, "patientid", prototype.getId());
		return resource;
	}
	
	private WebResource addPatientSearchQueryParam(WebResource resource, String queryParamKey, String queryParamValue) {
		if (NullChecker.isNotNullish(queryParamValue)) {
			resource = resource.queryParam(queryParamKey, queryParamValue);
		}
		return resource;
	}
	
	private String retrieveMygoalsReport() {
		URI baseMygoalsUri = getBaseMygoalsURI(automatedTestUser);
		URI mygoalsReportUri = UriBuilder.fromUri(baseMygoalsUri).path("report").build();
				
		jerseyClient = TestHelper.createMobileHealthClient().getJerseyClient();
		String result = getResourceHandle(mygoalsReportUri).accept("application/pdf").get(String.class);
		
		jerseyClient.destroy();
		jerseyClient = null;
		
		return result;
	}
	
	private URI getBaseMygoalsURI(Patient patient) {
		URI patientMygoalsUri = patient.getLink().getUriByTitle(LinkTitles.MyGoalsInventory);
		URI mygoalsUri = UriBuilder.fromUri(patientMygoalsUri).build();
		return mygoalsUri;

	}	
	
	protected String getPatientResourcePath() {
		return IntegrationTestConfiguration.getMobileHealthServerBaseUri() + "/patients";
	}	
	
}