package com.agilex.healthcare.mobilehealthplatform.restservice.mood;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;

import java.net.URI;

import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriBuilder;

import org.junit.After;
import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.client.AuthenticationInfo;
import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.MoodEvent;
import com.agilex.healthcare.mobilehealthplatform.domain.MoodEvents;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.testutility.IntegrationTestConfiguration;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.ModeHelper;
import com.sun.jersey.api.client.UniformInterfaceException;

public class MoodResourceTest {
	private static MobileHealthClientTestVersion client;
	PatientIdentifier patientIdentifier = new PatientIdentifier("EDIPI", "INT-TEST-MOCK");

	@BeforeClass
	public static void checkWhetherTestsAreRunningInCIHMode() {
		Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev", "dev"));
		AuthenticationInfo authenticationInfo = new AuthenticationInfo("zztest.patient75", "pass", "oauth");
		client = TestHelper.createMobileHealthClient(authenticationInfo);
	}
	
	@AfterClass
	public static void tearDown() {
		if (client != null) {
			client.getJerseyClient().destroy();
			client = null;
		}
	}
	
	@After
	public void cleanup() {
		MoodResourceTest test = new MoodResourceTest();
		URI uri = test.getContactLogsUri(patientIdentifier);
		MoodEvents retrievedEvents = test.client.getJerseyClient().resource(uri).get(MoodEvents.class);
		
		for(MoodEvent mood : retrievedEvents) {
			URI selfUri = mood.getSelfUri();
			test.client.getJerseyClient().resource(selfUri).delete();
		}
	}

	@Test
	public void getResultsWithoutFilter() {
		MoodEvent mood = new MoodEvent();
		PatientIdentifier patientIdentifier = generateUniqueId();
		mood.setPatientIdentifier(patientIdentifier);
		mood.setDate(DateHelper.parseDate("1/1/2012"));
		mood.setValue("01");
		MoodEvent savedMoodInRange = this.saveMood(mood);
		assertNotNull(savedMoodInRange);

		mood = new MoodEvent();
		mood.setPatientIdentifier(patientIdentifier);
		mood.setDate(DateHelper.parseDate("1/1/2015"));
		mood.setValue("02");
		MoodEvent savedMoodNotInRange = this.saveMood(mood);
		assertNotNull(savedMoodNotInRange);

		URI uri = getContactLogsUri(patientIdentifier);
		MoodEvents retrievedEvents = client.getJerseyClient().resource(uri).get(MoodEvents.class);
		assertNotNull(retrievedEvents);
		assertEquals(2, retrievedEvents.size());
	}

	@Test
	public void getResultsWithBeginEndDateFilter() {
		MoodEvent mood = new MoodEvent();
		PatientIdentifier patientIdentifier = generateUniqueId();
		mood.setPatientIdentifier(patientIdentifier);
		mood.setDate(DateHelper.parseDate("1/1/2012"));
		mood.setValue("01");
		MoodEvent savedMoodInRange = this.saveMood(mood);
		assertNotNull(savedMoodInRange);

		mood = new MoodEvent();
		mood.setPatientIdentifier(patientIdentifier);
		mood.setDate(DateHelper.parseDate("1/1/2015"));
		mood.setValue("02");
		MoodEvent savedMoodNotInRange = this.saveMood(mood);
		assertNotNull(savedMoodNotInRange);

		URI uri = getContactLogsUri(patientIdentifier);
		uri = UriBuilder.fromUri(uri.toString()).queryParam("startDate", "12/1/2011").queryParam("endDate", "12/1/2012").build();
		MoodEvents retrievedEvents = client.getJerseyClient().resource(uri).get(MoodEvents.class);
		assertNotNull(retrievedEvents);
		assertEquals(1, retrievedEvents.size());
	}

	@Test
	public void getResultsWithEntryDateFilter() {
		MoodEvent mood = new MoodEvent();
		PatientIdentifier patientIdentifier = generateUniqueId();
		mood.setPatientIdentifier(patientIdentifier);
		mood.setDate(DateHelper.parseDate("1/1/2012"));
		mood.setValue("01");
		MoodEvent savedMoodInRange = this.saveMood(mood);
		assertNotNull(savedMoodInRange);

		mood = new MoodEvent();
		mood.setPatientIdentifier(patientIdentifier);
		mood.setDate(DateHelper.parseDate("1/1/2015"));
		mood.setValue("02");
		MoodEvent savedMoodNotInRange = this.saveMood(mood);
		assertNotNull(savedMoodNotInRange);

		URI uri = getContactLogsUri(patientIdentifier);
		uri = UriBuilder.fromUri(uri.toString()).queryParam("entryDate", "1/1/2012").build();
		MoodEvents retrievedEvents = client.getJerseyClient().resource(uri).get(MoodEvents.class);
		assertNotNull(retrievedEvents);
		assertEquals(1, retrievedEvents.size());
	}

	@Test
	public void getSingleResultUsingSelfLink() {
		MoodEvent mood = new MoodEvent();
		PatientIdentifier patientIdentifier = generateUniqueId();
		mood.setPatientIdentifier(patientIdentifier);
		mood.setDate(DateHelper.parseDate("1/1/2012"));
		mood.setValue("01");
		mood.setNotes("Self Link Test");
		MoodEvent savedMood = this.saveMood(mood);
		assertNotNull(savedMood);
		assertNotNull(savedMood.getSelfLink());

		URI selfLink = savedMood.getSelfUri();
		MoodEvent moodFromSelfLink = client.getJerseyClient().resource(selfLink).get(MoodEvent.class);
		assertNotNull(moodFromSelfLink);
		assertEquals(mood.getNotes(), moodFromSelfLink.getNotes());
	}

	@Test(expected = UniformInterfaceException.class)
	public void saveWithInvalidData() {
		MoodEvent mood = new MoodEvent();
		PatientIdentifier patientIdentifier = generateUniqueId();
		mood.setPatientIdentifier(patientIdentifier);
		mood.setDate(DateHelper.parseDate("1/1/2012"));
		mood.setValue("20");

		URI uri = getContactLogsUri(mood.getPatientIdentifier());
		MoodEvent savedMood = client.getJerseyClient().resource(uri).type(MediaType.APPLICATION_XML).accept("application/xml").post(MoodEvent.class, mood);
		assertNull(savedMood);
	}

	@Test
	public void saveLargeNote() {
		MoodEvent mood = new MoodEvent();
		PatientIdentifier patientIdentifier = generateUniqueId();
		mood.setPatientIdentifier(patientIdentifier);
		mood.setNotes("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa" + "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
				+ "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa" + "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
				+ "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa");
		mood.setValue("1");
		mood.setDate(DateHelper.parseDate("1/1/2012"));
		MoodEvent savedMoodEvent = saveMood(mood);
		assertNotNull(savedMoodEvent);
		assertEquals(mood.getNotes(), savedMoodEvent.getNotes());
	}

	@Test
	public void testUpdateUsingSelfLink() {
		MoodEvent mood = new MoodEvent();
		PatientIdentifier patientIdentifier = generateUniqueId();
		mood.setPatientIdentifier(patientIdentifier);
		mood.setDate(DateHelper.parseDate("1/1/2012"));
		mood.setValue("01");
		MoodEvent savedMood = this.saveMood(mood);
		assertNotNull(savedMood);
		assertNotNull(savedMood.getSelfLink());
		assertEquals("01", savedMood.getValue());

		savedMood.setValue("10");
		URI selfLink = savedMood.getSelfUri();
		MoodEvent updatedMood = client.getJerseyClient().resource(selfLink).type(MediaType.APPLICATION_XML).accept("application/xml").put(MoodEvent.class, savedMood);
		assertNotNull(updatedMood);
		assertEquals("10", updatedMood.getValue());
	}

	@Test(expected = UniformInterfaceException.class)
	public void testDelete() {
		MoodEvent mood = new MoodEvent();
		PatientIdentifier patientIdentifier = generateUniqueId();
		mood.setPatientIdentifier(patientIdentifier);
		mood.setDate(DateHelper.parseDate("1/1/2012"));
		mood.setValue("01");
		MoodEvent savedMood = this.saveMood(mood);

		client.getJerseyClient().resource(savedMood.getSelfUri()).delete();

		MoodEvent shouldFail = client.getJerseyClient().resource(savedMood.getSelfUri()).get(MoodEvent.class);
		assertNull(shouldFail);
	}

	private PatientIdentifier generateUniqueId() {
		return new PatientIdentifier("EDIPI", "INT-TEST-MOCK");
	}

	private MoodEvent saveMood(MoodEvent moodEvent) {
		URI uri = getContactLogsUri(moodEvent.getPatientIdentifier());
		return client.getJerseyClient().resource(uri).type(MediaType.APPLICATION_XML).accept("application/xml").post(MoodEvent.class, moodEvent);
	}
	
	

	private URI getContactLogsUri(PatientIdentifier patientIdentifier) {
		String vitalsUriAsString = String.format("%s/patient/%s/%s/mood", IntegrationTestConfiguration.getMobileHealthServerBaseUri(), patientIdentifier.getAssigningAuthority(), patientIdentifier.getUniqueId());
		URI vitalsUri = UriBuilder.fromUri(vitalsUriAsString).build();
		return vitalsUri;
	}
}
