package com.agilex.healthcare.mobilehealthplatform.restservice.medications;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.net.URI;

import javax.ws.rs.core.UriBuilder;

import org.joda.time.DateTime;
import org.joda.time.Days;
import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.Ignore;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.Medication;
import com.agilex.healthcare.mobilehealthplatform.domain.Medications;
import com.agilex.healthcare.testutility.IntegrationTestConfiguration;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.ModeHelper;

public class MedicationResourceTest {

	private String patientId = "D123401";

	private static MobileHealthClientTestVersion client;
	@BeforeClass
	public static void startup() {
		Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev", "dev"));
		client = TestHelper.createMobileHealthClient();
	}
	
	@AfterClass
	public static void destroy() {
		if (client != null) {
			client.getJerseyClient().destroy();
			client = null;
		}
	}
	@Test
	public void verifyGetByStatus() {
		URI uri = createMedicationsUri(this.patientId);
		uri = UriBuilder.fromUri(uri.toString()).queryParam("status", "active").build();
		Medications medications = client.getJerseyClient().resource(uri).get(Medications.class);

		assertNotNull(medications);
		assertTrue(medications.size() >= 2);
	}

	@Ignore
	public void verifySortOrder() {
		URI uri = createMedicationsUri(this.patientId);
		
		Medications medications = client.getJerseyClient().resource(uri).get(Medications.class);

		assertNotNull(medications);
		assertTrue(medications.size() > 0);

		Medication previous = medications.get(0);
		for(Medication medication : medications) {
			if(previous.getStatus().equalsIgnoreCase(medication.getStatus()))
			{
				assertTrue(previous.getDrugName().toLowerCase().compareTo(medication.getDrugName().toLowerCase()) <= 0);
			}
			previous = medication;
		}
	}

	@Test
	public void verifyExpiredSortedAfterActive() {
		URI uri = createMedicationsUri(this.patientId);
		
		Medications medications = client.getJerseyClient().resource(uri).get(Medications.class);

		assertNotNull(medications);
		assertTrue(medications.size() > 0);

		Medication previous = medications.get(0);
		for(Medication medication : medications) {
			if(medication.getStatus().equalsIgnoreCase("active"))
				assertEquals(previous.getStatus().toLowerCase(), "active");
			
			if(medication.getStatus().equalsIgnoreCase("expired") && previous.getStatus().equalsIgnoreCase("expired"))
				assertEquals(previous.getStatus().toLowerCase(), "expired");

			previous = medication;
		}
	}
	@Test
	public void verifyAgeOfExpiredMeds() {
		URI uri = createMedicationsUri(this.patientId);
		uri = UriBuilder.fromUri(uri.toString()).queryParam("status", "expired").build();
		Medications medications = client.getJerseyClient().resource(uri).get(Medications.class);

		Medication med;
		DateTime today = new DateTime();
		DateTime drugExpiration = null;

		for (int i = 0; i < medications.getSize(); i++) {
			med = medications.get(i);
			if (med.getEndDate() != null && med.getStatus().equalsIgnoreCase("expired")) {
				drugExpiration = new DateTime(med.getDate());
				assertTrue(Days.daysBetween(today, drugExpiration).getDays() <= 120);
			}
		}
	}

	@Test
	public void verifyGetBySourceSystem() {
		URI uri = createMedicationsUri(this.patientId);
		uri = UriBuilder.fromUri(uri.toString()).queryParam("sourceSystem", "VA").build();
		Medications medications = client.getJerseyClient().resource(uri).get(Medications.class);

		assertNotNull(medications);
		assertTrue(medications.size() >= 1);
	}

	@Test
	public void verifyGetBySources() {
		URI uri = createMedicationsUri(this.patientId);
		uri = UriBuilder.fromUri(uri.toString()).queryParam("sourceSystem", "other").queryParam("medicationSource", "other").build();
		Medications medications = client.getJerseyClient().resource(uri).get(Medications.class);

		assertNotNull(medications);
		assertTrue(medications.size() >= 1);
	}

	@Test
	public void verifyGetByDate() {
		URI uri = createMedicationsUri(this.patientId);
		uri = UriBuilder.fromUri(uri.toString()).queryParam("startDate", "1/1/1990").queryParam("endDate", "1/2/1990").build();
		Medications medications = client.getJerseyClient().resource(uri).get(Medications.class);

		assertNotNull(medications);
		assertTrue(medications.size() == 1);
	}

	@Test
	public void verifyGetByMultipleStatuses() {
		URI uri = createMedicationsUri(this.patientId);
		uri = UriBuilder.fromUri(uri.toString()).queryParam("status", "SUSPENDED,DISCONTINUED,ACTIVE").build();
		Medications medications = client.getJerseyClient().resource(uri).get(Medications.class);

		assertNotNull(medications);
		assertTrue(medications.size() >= 4);
	}

	@Test
	public void verifyGetByMedicationSource() {
		URI uri = createMedicationsUri(this.patientId);
		uri = UriBuilder.fromUri(uri.toString()).queryParam("medicationSource", "inpatient").build();
		Medications medications = client.getJerseyClient().resource(uri).get(Medications.class);

		assertNotNull(medications);
		assertTrue(medications.size() >= 4);
	}

	@Test
	public void verifyPagination() {
		URI uri = createMedicationsUri(this.patientId);
		uri = UriBuilder.fromUri(uri.toString()).queryParam("page", "1").queryParam("resultCount", "5").build();
		Medications medications = client.getJerseyClient().resource(uri).get(Medications.class);

		assertNotNull(medications);
		assertEquals(5, medications.size());
	}

	@Test
	public void verifyPaginationWhenCountIsLargerThanData() {
		URI uri = createMedicationsUri(this.patientId);
		uri = UriBuilder.fromUri(uri.toString()).queryParam("page", "1").queryParam("resultCount", "100").build();
		Medications medications = client.getJerseyClient().resource(uri).get(Medications.class);

		assertNotNull(medications);
		assertTrue(medications.size() >= 12);
	}

	public URI createMedicationsUri(String patientId) {
		String medicationsUriAsString = String.format("%s/patient/EDIPI/%s/medications", IntegrationTestConfiguration.getMobileHealthServerBaseUri(), patientId);
		URI medicationUri = UriBuilder.fromUri(medicationsUriAsString).build();
		return medicationUri;
	}

}
