package com.agilex.healthcare.mobilehealthplatform.restservice.medications;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assume.assumeTrue;

import java.net.URI;

import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Ignore;
import org.junit.Test;
import org.w3c.dom.Document;
import org.w3c.dom.Node;

import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.LinkTitles;
import com.agilex.healthcare.mobilehealthplatform.domain.Medication;
import com.agilex.healthcare.testutility.IntegrationTestConfiguration;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.ModeHelper;
import com.agilex.healthcare.utility.XpathHelper;

public abstract class MedicationDataAssertion {
	private Medication testMedication = IntegrationTestConfiguration.getOutpatientMedication();
	private static final String MEDICATION_DATE_FORMAT = "MM/dd/yyyy HH:mm:ss";
	private static MobileHealthClientTestVersion client;
	private static URI medLink = null;
	private Node medicationNode;

	public abstract URI getMedicationUri();

	public abstract Medication getTestMedication();

	@BeforeClass
	public static void startup() {
		Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev", "dev"));
		client = TestHelper.createMobileHealthClient();
	}
	
	@AfterClass
	public static void destroy() {
		if (client != null) {
			client.getJerseyClient().destroy();
			client = null;
		}
	}
	
	@Before
	public void setup() {
		testMedication = getTestMedication();
		medLink = getMedicationUri();
		medicationNode = getMedicationNode();
	}

	@Test
	public void validateDrugName() {
		String expectedValue = testMedication.getDrugName();
		runTest("drugName", expectedValue);
	}

	@Test
	public void validateDaysSupply() {
		assumeTrue(!"INPATIENT".equalsIgnoreCase(testMedication.getMedicationSource()));
		String expectedValue = testMedication.getDaysSupply();
		runTest("daysSupply", expectedValue);
	}

	@Test
	public void validateQuantity() {
		assumeTrue(!"INPATIENT".equalsIgnoreCase(testMedication.getMedicationSource()));
		String expectedValue = testMedication.getQuantity();
		runTest("quantity", expectedValue);
	}

	@Test
	public void validateSig() {
		String expectedValue = testMedication.getSig();
		runTest("sig", expectedValue);
	}

	@Test
	public void validateLastFilledDate() {
		assumeTrue(!"INPATIENT".equalsIgnoreCase(testMedication.getMedicationSource()));
		String expectedValue = DateHelper.format(testMedication.getLastFilledDate(), "MMM dd, yyyy");
		runTest("lastFilledDate", expectedValue);
	}

	@Test
	public void validateRefills() {
		assumeTrue(!"INPATIENT".equalsIgnoreCase(testMedication.getMedicationSource()));
		String expectedValue = testMedication.getRefills().toString();
		runTest("refills", expectedValue);
	}

	@Test
	public void validateStartDate() {
		String expectedValue;
		if (testMedication.getStartDate() == null)
			expectedValue = "";
		else
			expectedValue = DateHelper.format(testMedication.getStartDate(), MEDICATION_DATE_FORMAT);
		runTest("startDate", expectedValue);
	}

	@Test
	public void validateEndDate() {
		String expectedValue;
		if (testMedication.getEndDate() == null)
			expectedValue = "";
		else
			expectedValue = DateHelper.format(testMedication.getEndDate(), MEDICATION_DATE_FORMAT);
		runTest("endDate", expectedValue);
	}

	@Test
	public void validateMedicationSource() {
		String expectedValue = testMedication.getMedicationSource();
		runTest("medicationSource", expectedValue);
	}

	@Test
	public void validateOrderNumber() {
		String expectedValue = testMedication.getOrderNumber();
		runTest("orderNumber", expectedValue);
	}

	@Test
	public void validateActive() {
		String expectedValue = Boolean.toString(testMedication.isActive());
		runTest("active", expectedValue);
	}

	@Ignore
	public void validateStatus() {
		String expectedValue = testMedication.getStatus();
		runTest("status", expectedValue);
	}

	@Test
	public void verifyMedicationDetailAtomLinkExists() throws Exception {
		Document meds = client.getAsXml(medLink);
		String orderNumber = testMedication.getOrderNumber();
		Node medicationDetailAtomLinkNode = XpathHelper
				.getNode(meds, "//m:medications/m:medication[orderNumber='" + orderNumber + "']/atom:link[@title='" + LinkTitles.PatientMedicationDetail + "']");
		Node namedItem = medicationDetailAtomLinkNode.getAttributes().getNamedItem("href");
		String href = namedItem.getNodeValue();
		assertNotNull(href);
	}

	private void runTest(String fieldName, String expectedValue) {
		String actualValue = XpathHelper.getString(medicationNode, fieldName);
		assertEquals(expectedValue, actualValue);
	}

	private Node getMedicationNode() {
		Document meds = client.getAsXml(medLink);
		String orderNumber = testMedication.getOrderNumber();
		Node medicationNode = XpathHelper.getNode(meds, "//m:medications/m:medication[orderNumber='" + orderNumber + "']");
		if (medicationNode == null)
			throw new RuntimeException("Unable to retrieve test medication " + orderNumber);

		return medicationNode;
	}
}
