package com.agilex.healthcare.mobilehealthplatform.restservice.learn;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.net.URI;

import javax.ws.rs.core.UriBuilder;

import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.LearnArticle;
import com.agilex.healthcare.mobilehealthplatform.domain.LearnCategories;
import com.agilex.healthcare.mobilehealthplatform.domain.LearnCategory;
import com.agilex.healthcare.mobilehealthplatform.domain.LearnCondition;
import com.agilex.healthcare.mobilehealthplatform.domain.LearnConditions;
import com.agilex.healthcare.mobilehealthplatform.domain.LearnSubCategory;
import com.agilex.healthcare.testutility.IntegrationTestConfiguration;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.ModeHelper;
import com.sun.jersey.api.client.UniformInterfaceException;

public class LearnResourceTest {
	private static MobileHealthClientTestVersion client;

	@BeforeClass
	public static void checkWhetherTestsAreRunningInCIHMode() {
		Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev", "dev"));
		client = TestHelper.createMobileHealthClient();
	}
	
	@AfterClass
	public static void destroy() {
		if (client != null) {
			client.getJerseyClient().destroy();
			client = null;
		}
	}

	@Test
	public void fetchLearnConditions() {
		String baseUri = IntegrationTestConfiguration.getMobileHealthServerBaseUri();
		URI uri = UriBuilder.fromUri(baseUri).path("public/learn/conditions").build();
		LearnConditions conditions = client.getJerseyClient().resource(uri).get(LearnConditions.class);

		assertNotNull(conditions.getLink());
		assertTrue(conditions.getLink().size() > 0);
	}

	@Test
	public void fetchLearnCategoriesByCondition() {
		LearnCondition validCondition = new LearnCondition();
		validCondition.setId("condition1");
		LearnCategories categories = getCategoriesForCondition(validCondition);

		assertEquals(3, categories.size());		
		assertEquals("Learn about Category A", categories.getCategoryById("categoryA").getTitle());
		assertEquals("Learn about Category B", categories.getCategoryById("categoryB").getTitle());
		assertEquals("Learn about Category C", categories.getCategoryById("categoryC").getTitle());
	}
	
	@Test (expected=UniformInterfaceException.class)
	public void fetchLearnCategoriesByInvalidCondition() {
		LearnCondition invalidCondition = new LearnCondition();
		invalidCondition.setId("invalid");
		getCategoriesForCondition(invalidCondition);
	}

	private LearnCategories getCategoriesForCondition(LearnCondition condition) {
		String baseUri = IntegrationTestConfiguration.getMobileHealthServerBaseUri();
		URI categoriesUri = UriBuilder.fromUri(baseUri).path("public/learn/conditions/id").path(condition.getId()).path("categories").build();
		LearnCategories retrievedCategories = client.getJerseyClient().resource(categoriesUri).get(LearnCategories.class);
		return retrievedCategories;
	}
	
	@Test
	public void fetchLearnCategoryById() {
		String categoryId = "categoryA";
		String baseUri = IntegrationTestConfiguration.getMobileHealthServerBaseUri();
		URI specificCategoryUri = UriBuilder.fromUri(baseUri).path("public/learn/categories/id/").path(categoryId).build();

		LearnCategory category = client.getJerseyClient().resource(specificCategoryUri).get(LearnCategory.class);
		LearnSubCategory subcategory1 = category.getSubCategories().getSubCategoryById("subcat1");
		LearnSubCategory subcategory2 = category.getSubCategories().getSubCategoryById("subcat2");
		
		assertEquals("Learn about Category A", category.getTitle());
		assertEquals(2, category.getNumberOfSubCategories());
		assertEquals("SubCategory 1", subcategory1.getTitle());
		assertEquals(2, subcategory1.getNumberOfArticles());
		assertEquals("What is Article A about?", subcategory1.getArticles().getArticleById("articleA").getTitle());
		assertEquals("What is Article B about?", subcategory1.getArticles().getArticleById("articleB").getTitle());
		assertEquals("SubCategory 2", subcategory2.getTitle());
		assertEquals(2, subcategory2.getNumberOfArticles());
		assertEquals("What is Article C about?", subcategory2.getArticles().getArticleById("articleC").getTitle());
		assertEquals("What is Article D about?", subcategory2.getArticles().getArticleById("articleD").getTitle());
	}

	@Test (expected=UniformInterfaceException.class)
	public void fetchLearnCategoryByInvalidId() {
		String categoryId = "invalid";
		String baseUri = IntegrationTestConfiguration.getMobileHealthServerBaseUri();
		URI specificCategoryUri = UriBuilder.fromUri(baseUri).path("public/learn/categories/id/").path(categoryId).build();
		client.getJerseyClient().resource(specificCategoryUri).get(LearnCategory.class);
	}
	
	@Test
	public void fetchLearnArticleById() {
		String articleId = "articleA";
		
		String baseUri = IntegrationTestConfiguration.getMobileHealthServerBaseUri();
		URI specificArticleUri = UriBuilder.fromUri(baseUri).path("public/learn/articles/id/").path(articleId).build();
		LearnArticle article = client.getJerseyClient().resource(specificArticleUri).get(LearnArticle.class);

		assertNotNull(article);
		assertEquals("What is Article A about?", article.getTitle());
		assertEquals("Article A provides information about the letter A.", article.getContent());
	}
	
	@Test (expected=UniformInterfaceException.class)
	public void fetchLearnArticleByInvalidId() {
		String articleId = "invalid";
		
		String baseUri = IntegrationTestConfiguration.getMobileHealthServerBaseUri();
		URI specificArticleUri = UriBuilder.fromUri(baseUri).path("public/learn/articles/id/").path(articleId).build();
		client.getJerseyClient().resource(specificArticleUri).get(LearnArticle.class);
	}
}