package com.agilex.healthcare.mobilehealthplatform.restservice.facility;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.net.URI;

import javax.ws.rs.core.UriBuilder;

import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.Facilities;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppType;
import com.agilex.healthcare.testutility.IntegrationTestConfiguration;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.ModeHelper;
import com.agilex.healthcare.utility.NullChecker;

public class FacilityResourceTest {
	private static MobileHealthClientTestVersion client;
	
	@BeforeClass
	public static void checkWhetherTestsAreRunningInProperModes() {
		Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev", "dev"));
		client = TestHelper.createMobileHealthClient();
	}
	
	@AfterClass
	public static void destroy() {
		if (client != null) {
			client.getJerseyClient().destroy();
			client = null;
		}
	}
	
	@Test
	public void verifyRetrieve() {
		URI facilityUri = createFacilitiesUri();
		Facilities facilities = client.getJerseyClient().resource(facilityUri).get(Facilities.class);
		
		assertNotNull(facilities);
		assertTrue(facilities.size() > 0);
	}
	
	@Test
	public void verifyRetrieveWithTermParameter() {
		URI facilityUri = createFacilitiesUri();
		Facilities facilities = client.getJerseyClient().resource(facilityUri).get(Facilities.class);
		
		URI facilityUriWithTerm = createFacilitiesUri(AppType.MBB.getName(), "va", false);
		Facilities filteredFacilities = client.getJerseyClient().resource(facilityUriWithTerm).get(Facilities.class);
		
		assertNotNull(facilities);
		assertTrue(facilities.size() > 0);
		assertNotNull(filteredFacilities);
		assertTrue(filteredFacilities.size() > 0);
		assertTrue("filtered list is not smaller that complete list", filteredFacilities.size() < facilities.size());
	}
	
	@Test
	public void verifyRetrieveWithParentFacilitiesOnlyParameter() {
		URI facilityUri = createFacilitiesUri();
		Facilities facilities = client.getJerseyClient().resource(facilityUri).get(Facilities.class);
		
		URI facilityUriWithTerm = createFacilitiesUri(AppType.MBB.getName(), "", true);
		Facilities filteredFacilities = client.getJerseyClient().resource(facilityUriWithTerm).get(Facilities.class);
		
		assertNotNull(facilities);
		assertTrue(facilities.size() > 0);
		assertNotNull(filteredFacilities);
		assertTrue(filteredFacilities.size() > 0);
		assertTrue("filtered list is not smaller that complete list", filteredFacilities.size() < facilities.size());
	}
	
	@Test
	public void verifyRetrieveWithTermParameterAndParentFacilitiesOnlyParameter() {
		URI facilityUriWithTerm = createFacilitiesUri("", "va", true);
		Facilities filteredFacilities = client.getJerseyClient().resource(facilityUriWithTerm).get(Facilities.class);
		assertNotNull(filteredFacilities);
		//assertEquals("Currently this list should contain 14 entries", 14, filteredFacilities.size());
	}
	
	@Test
	public void verifyMostRecentFacilities() {
		URI patientMostRecentFacilitiesUri = createPatientMostRecentFacilitiesUri("EDIPI", "D123401");
		
		Facilities facilities = client.getJerseyClient().resource(patientMostRecentFacilitiesUri).get(Facilities.class);
		
		assertNotNull(facilities);
		assertTrue(facilities.size() >= 0);
		assertTrue(facilities.size() <= 3);
		
		for (int i = 0; i < facilities.size(); i++) {
			for (int j = 0; j < facilities.size(); j++) {
				if (i != j && facilities.get(i).equals(facilities.get(j))) {
					fail("The most recent facilities list has a duplicate entry but shouldn't");
				}
			}
		}
	}
	
	private URI createFacilitiesUri() {
		return createFacilitiesUri(AppType.MBB.getName(), null, false);
	}
	
	private URI createFacilitiesUri(String app, String term, boolean parentFacilitiesOnly) {
		StringBuilder sb = new StringBuilder(IntegrationTestConfiguration.getResourceUri("facility-service").toString());
		
		sb.append("?parentFacilitiesOnly=");
		sb.append(parentFacilitiesOnly);
		
		if (NullChecker.isNotNullish(term)) {
			sb.append("&term=");
			sb.append(term);
		}

		if (NullChecker.isNotNullish(app)) {
			sb.append("&app=");
			sb.append(app);
		}
		
		return UriBuilder.fromUri(sb.toString()).build();
	}
	
	private URI createPatientMostRecentFacilitiesUri(String assigningAuthority, String userId) {
		String notificationsUriAsString = String.format("%s/facility-service/patient/%s/%s/facilities", IntegrationTestConfiguration.getMobileHealthServerBaseUri(), assigningAuthority, userId);
		return UriBuilder.fromUri(notificationsUriAsString).build();
	}
}
