package com.agilex.healthcare.mobilehealthplatform.restservice.exercise;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.net.URI;

import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriBuilder;

import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.client.AuthenticationInfo;
import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.datalayer.exercise.ExerciseFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.Exercise;
import com.agilex.healthcare.mobilehealthplatform.domain.Exercises;
import com.agilex.healthcare.mobilehealthplatform.domain.LinkTitles;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.code.ExerciseCode;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.ModeHelper;
import com.sun.jersey.api.client.UniformInterfaceException;

public class ExerciseResourceTest {

	private static MobileHealthClientTestVersion client;
	
	@BeforeClass
	public static void checkWhetherTestsAreRunningInCIHMode() {
		Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev", "dev"));
		AuthenticationInfo authenticationInfo = new AuthenticationInfo("zztest.patient75", "pass", "oauth");
		client = TestHelper.createMobileHealthClient(authenticationInfo);
	}
	
	@AfterClass
	public static void tearDown() {
		if (client != null) {
			client.getJerseyClient().destroy();
			client = null;
		}
	}
	
	public void cleanup(Exercises exercises) {
		for(Exercise exercise : exercises) {
			cleanup(exercise);
		}
	}
	
	public void cleanup(Exercise exercise) {
		client.getJerseyClient().resource(exercise.getSelfUri()).delete();
	}

	@Test
	public void getExercisesWithoutFilter() {
		PatientIdentifier patientIdentifier = getIntegrationIdentifier();
		generateExercises(patientIdentifier);

		URI uri = getExerciseUri(patientIdentifier);
		Exercises exercises = client.getJerseyClient().resource(uri).get(Exercises.class);
		assertNotNull(exercises);
		assertTrue(exercises.size() >= 4);
		
		cleanup(exercises);
	}

	@Test
	public void getExercisesWithFilterStartEnd() {
		PatientIdentifier patientIdentifier = getIntegrationIdentifier();
		generateExercises(patientIdentifier);

		ExerciseFilter filter = new ExerciseFilter();
		filter.setStartDate(DateHelper.parseDate("1/4/2012"));
		filter.setEndDate(DateHelper.parseDate("1/10/2012"));

		URI uri = getExerciseUri(patientIdentifier);
		uri = UriBuilder.fromUri(uri.toString()).queryParam("startDate", "1/4/2012").queryParam("endDate", "1/10/2012").build();
		Exercises exercises = client.getJerseyClient().resource(uri).get(Exercises.class);
		assertNotNull(exercises);
		assertEquals(1, exercises.size());

		uri = getExerciseUri(patientIdentifier);
		uri = UriBuilder.fromUri(uri.toString()).queryParam("startDate", "1/1/2012").queryParam("endDate", "1/4/2012").build();
		exercises = client.getJerseyClient().resource(uri).get(Exercises.class);
		assertNotNull(exercises);
		assertEquals(4, exercises.size());
		
		cleanup(exercises);
	}

	@Test
	public void getExercisesWithFilterSingleDate() {
		PatientIdentifier patientIdentifier = getIntegrationIdentifier();
		generateExercises(patientIdentifier);

		URI uri = getExerciseUri(patientIdentifier);
		uri = UriBuilder.fromUri(uri.toString()).queryParam("date", "1/2/2012").build();
		Exercises exercises = client.getJerseyClient().resource(uri).get(Exercises.class);
		assertNotNull(exercises);
		assertEquals(1, exercises.size());
		
		cleanup(exercises);
		
		uri = getExerciseUri(patientIdentifier);
		uri = UriBuilder.fromUri(uri).build();
		exercises = client.getJerseyClient().resource(uri).get(Exercises.class);
		
		cleanup(exercises);
	}

	@Test
	public void verifyUpdate() {
		PatientIdentifier patientIdentifier = getIntegrationIdentifier();

		Exercise exercise = new Exercise();
		exercise.setPatientIdentifier(patientIdentifier);
		exercise.setActivity(ExerciseCode.ACTIVITY_TYPE_BICYCLING);
		exercise.setDistance(1);
		exercise.setDuration(1);
		exercise.setDate(DateHelper.parseDate("1/1/2012"));
		Exercise savedExercise = saveExercise(exercise);
		assertNotNull(savedExercise);
		assertEquals(ExerciseCode.ACTIVITY_TYPE_BICYCLING, savedExercise.getActivity());
		assertNotNull(savedExercise.getSelfUri());

		savedExercise.setActivity(ExerciseCode.ACTIVITY_TYPE_RUNNING);
		Exercise updatedExercise = client.getJerseyClient().resource(savedExercise.getSelfUri()).type(MediaType.APPLICATION_XML).accept("application/xml").put(Exercise.class, savedExercise);
		assertNotNull(updatedExercise);
		assertEquals(ExerciseCode.ACTIVITY_TYPE_RUNNING, updatedExercise.getActivity());

		URI uri = getExerciseUri(patientIdentifier);
		Exercises exercises = client.getJerseyClient().resource(uri).get(Exercises.class);
		assertNotNull(exercises);
		assertEquals(1, exercises.size());
		
		cleanup(exercises);
	}

	@Test
	public void verifySortOrder() {
		PatientIdentifier patientIdentifier = getIntegrationIdentifier();
		generateExercises(patientIdentifier);
		URI uri = getExerciseUri(patientIdentifier);
		Exercises exercises = client.getJerseyClient().resource(uri).get(Exercises.class);

		assertNotNull(exercises);
		assertEquals(4, exercises.size());
		assertEquals(4, exercises.get(0).getDuration());
		assertEquals(3, exercises.get(1).getDuration());
		assertEquals(2, exercises.get(2).getDuration());
		assertEquals(1, exercises.get(3).getDuration());
		
		cleanup(exercises);
	}

	@Test(expected = Exception.class)
	public void verifyValidationForDistance() {
		PatientIdentifier patientIdentifier = getIntegrationIdentifier();

		Exercise exercise = new Exercise();
		exercise.setPatientIdentifier(patientIdentifier);
		exercise.setActivity(ExerciseCode.ACTIVITY_TYPE_BICYCLING);
		exercise.setDistance(-1);
		exercise.setDuration(1);
		exercise.setDate(DateHelper.parseDate("1/1/2012"));
		saveExercise(exercise);
	}

	@Test(expected = Exception.class)
	public void verifyValidationForDuration() {
		PatientIdentifier patientIdentifier = getIntegrationIdentifier();

		Exercise exercise = new Exercise();
		exercise.setPatientIdentifier(patientIdentifier);
		exercise.setActivity(ExerciseCode.ACTIVITY_TYPE_BICYCLING);
		exercise.setDistance(1);
		exercise.setDuration(-1);
		exercise.setDate(DateHelper.parseDate("1/1/2012"));
		saveExercise(exercise);

		exercise = new Exercise();
		exercise.setPatientIdentifier(patientIdentifier);
		exercise.setActivity(ExerciseCode.ACTIVITY_TYPE_BICYCLING);
		exercise.setDistance(1);
		exercise.setDate(DateHelper.parseDate("1/1/2012"));
		try {
			saveExercise(exercise);
			assertFalse(true);
		} catch (UniformInterfaceException e) {
			assertTrue(true);
		}
	}

	@Test(expected = Exception.class)
	public void verifyValidationForDate() {
		PatientIdentifier patientIdentifier = getIntegrationIdentifier();

		Exercise exercise = new Exercise();
		exercise.setPatientIdentifier(patientIdentifier);
		exercise.setActivity(ExerciseCode.ACTIVITY_TYPE_BICYCLING);
		exercise.setDistance(1);
		exercise.setDuration(-1);
		saveExercise(exercise);
	}

	@Test
	public void verifyValidationForNotes() {
		PatientIdentifier patientIdentifier = getIntegrationIdentifier();
		String largeNote = "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa" + "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
				+ "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa" + "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
				+ "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa" + "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa";

		Exercise exercise = new Exercise();
		exercise.setPatientIdentifier(patientIdentifier);
		exercise.setActivity(ExerciseCode.ACTIVITY_TYPE_BICYCLING);
		exercise.setDistance(1);
		exercise.setDuration(1);
		exercise.setNotes(largeNote);
		exercise.setDate(DateHelper.parseDate("1/1/2012"));
		Exercise savedExercise = saveExercise(exercise);

		assertEquals(exercise.getNotes(), savedExercise.getNotes());
		
		cleanup(savedExercise);

	}

	@Test(expected = UniformInterfaceException.class)
	public void verifyDelete() {
		PatientIdentifier patientIdentifier = getIntegrationIdentifier();

		Exercise exercise = new Exercise();
		exercise.setPatientIdentifier(patientIdentifier);
		exercise.setActivity(ExerciseCode.ACTIVITY_TYPE_BICYCLING);
		exercise.setDistance(1);
		exercise.setDuration(1);
		exercise.setDate(DateHelper.parseDate("1/1/2012"));
		Exercise savedExercise = saveExercise(exercise);

		client.getJerseyClient().resource(savedExercise.getSelfUri()).delete();

		Exercise shouldThrowException = client.getJerseyClient().resource(savedExercise.getSelfUri()).get(Exercise.class);
		assertNull(shouldThrowException);
	}

	public void generateExercises(PatientIdentifier patientIdentifier) {

		Exercise exercise = new Exercise();
		exercise.setPatientIdentifier(patientIdentifier);
		exercise.setActivity(ExerciseCode.ACTIVITY_TYPE_BICYCLING);
		exercise.setDistance(1);
		exercise.setDuration(1);
		exercise.setDate(DateHelper.parseDate("1/1/2012"));
		saveExercise(exercise);

		exercise = new Exercise();
		exercise.setPatientIdentifier(patientIdentifier);
		exercise.setActivity(ExerciseCode.ACTIVITY_TYPE_RUNNING);
		exercise.setDistance(2);
		exercise.setDuration(2);
		exercise.setDate(DateHelper.parseDate("1/2/2012"));
		saveExercise(exercise);

		exercise = new Exercise();
		exercise.setPatientIdentifier(patientIdentifier);
		exercise.setActivity(ExerciseCode.ACTIVITY_TYPE_WALKING);
		exercise.setDistance(3);
		exercise.setDuration(3);
		exercise.setDate(DateHelper.parseDate("1/3/2012"));
		saveExercise(exercise);

		exercise = new Exercise();
		exercise.setPatientIdentifier(patientIdentifier);
		exercise.setActivity(ExerciseCode.ACTIVITY_TYPE_RUNNING);
		exercise.setDistance(4);
		exercise.setDuration(4);
		exercise.setDate(DateHelper.parseDate("1/4/2012"));
		saveExercise(exercise);
	}

	private PatientIdentifier getIntegrationIdentifier() {
		return new PatientIdentifier("EDIPI", "INT-TEST-MOCK");
	}

	public Exercise saveExercise(Exercise exercise) {
		URI uri = getExerciseUri(exercise.getPatientIdentifier());
		return client.getJerseyClient().resource(uri).type(MediaType.APPLICATION_XML).accept("application/xml").post(Exercise.class, exercise);
	}

	private URI getExerciseUri(PatientIdentifier patientIdentifier) {
		return TestHelper.getPatientDataUri(patientIdentifier, LinkTitles.PatientExercise, "zztest.patient75");
	}

}
