package com.agilex.healthcare.mobilehealthplatform.restservice.documents;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;

import java.io.InputStream;
import java.net.URI;

import javax.ws.rs.core.MediaType;

import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.Test;
import org.springframework.http.HttpStatus;

import com.agilex.healthcare.mobilehealthplatform.client.AuthenticationInfo;
import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.DietEntry;
import com.agilex.healthcare.mobilehealthplatform.domain.Document;
import com.agilex.healthcare.mobilehealthplatform.domain.Documents;
import com.agilex.healthcare.mobilehealthplatform.domain.LinkTitles;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.testutility.PatientLoader;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.MealType;
import com.agilex.healthcare.utility.ModeHelper;
import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.ClientResponse;

public class DocumentResourceTest {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(DocumentResourceTest.class);
	private PatientIdentifier patientIdentifier = new PatientIdentifier("EDIPI", "INT-TEST-MOCK");
	private String userName = "zztest.patient75";

	
	private static MobileHealthClientTestVersion client;

	@BeforeClass
	public static void checkWhetherTestsAreRunningInCIHMode() {
		Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev", "dev"));
		AuthenticationInfo authenticationInfo = new AuthenticationInfo("zztest.patient75", "pass", "oauth");
		client = TestHelper.createMobileHealthClient(authenticationInfo);
	}

	@AfterClass
	public static void destroy() {
		if (client != null) {
			client.getJerseyClient().destroy();
			client = null;
		}
	}
	
	@Test
	public void createDocument() {
		Document createdDocument = saveDocument(client.getJerseyClient(), patientIdentifier, "Test Document Title");
		logger.debug("!!! saved document metadata, document uri: " + createdDocument.getSelfUri());
	}

	@Test
	public void createDocumentAndRetrieve() {
		Document createdDocument = saveDocument(client.getJerseyClient(), patientIdentifier, "title");
		logger.debug("saved document metadata, document uri: " + createdDocument.getSelfUri());

		Document retrievedDocument = client.getJerseyClient().resource(createdDocument.getSelfUri()).get(Document.class);
		assertNotNull(retrievedDocument);
		assertEquals("title", retrievedDocument.getDocumentTitle());
	}

	@Test
	public void deleteDocument() {
		Document createdDocument = saveDocument(client.getJerseyClient(), patientIdentifier, "title-of-document-to-delete");
		logger.debug("saved document metadata, document uri: " + createdDocument.getSelfUri());

		client.getJerseyClient().resource(createdDocument.getSelfUri()).delete();

		ClientResponse response = client.getJerseyClient().resource(createdDocument.getSelfUri()).get(ClientResponse.class);
		assertEquals(HttpStatus.NOT_FOUND.value(), response.getStatus());
	}

	@Test
	public void createdDocumentContainsLinkToContent() {
		Document createdDocument = saveDocument(client.getJerseyClient(), patientIdentifier, "title");
		logger.debug("saved document metadata, document uri: " + createdDocument.getSelfUri());

		URI bodyUri = createdDocument.getLink().getUriByTitle(LinkTitles.DocumentContent);
		assertNotNull(bodyUri);
	}

	@Test
	public void retrievedDocumentContainsLinkToContent() {
		Document createdDocument = saveDocument(client.getJerseyClient(), patientIdentifier, "title");
		logger.debug("saved document metadata, document uri: " + createdDocument.getSelfUri());

		Document retrievedDocument = client.getJerseyClient().resource(createdDocument.getSelfUri()).get(Document.class);
		URI bodyUri = retrievedDocument.getLink().getUriByTitle(LinkTitles.DocumentContent);
		assertNotNull(bodyUri);
	}

	@Test
	public void createDocumentWithTextContent() {
		Document createdDocument = saveDocument(client.getJerseyClient(), patientIdentifier, "document-with-text-content");
		logger.debug("saved document metadata, document uri: " + createdDocument.getSelfUri());
		URI contentUri = createdDocument.getLink().getUriByTitle(LinkTitles.DocumentContent);

		String documentContent = client.getJerseyClient().resource(contentUri).type(MediaType.TEXT_PLAIN).get(String.class);

		client.getJerseyClient().resource(contentUri).type(MediaType.TEXT_PLAIN).put("hello-world");

		documentContent = client.getJerseyClient().resource(contentUri).type(MediaType.TEXT_PLAIN).get(String.class);
		assertEquals("hello-world", documentContent);
	}

	@Test
	public void deleteDocumentWithBody() {
		Document createdDocument = saveDocument(client.getJerseyClient(), patientIdentifier, "title-of-document-with-text-to-delete");
		logger.debug("saved document metadata, document uri: " + createdDocument.getSelfUri());
		URI contentUri = createdDocument.getLink().getUriByTitle(LinkTitles.DocumentContent);

		client.getJerseyClient().resource(contentUri).type(MediaType.TEXT_PLAIN).put("hello-world");

		client.getJerseyClient().resource(createdDocument.getSelfUri()).delete();

		ClientResponse response = client.getJerseyClient().resource(createdDocument.getSelfUri()).get(ClientResponse.class);
		assertEquals(HttpStatus.NOT_FOUND.value(), response.getStatus());
	}

	@Test
	public void createDocumentWithImageContent() {
		Document createdDocument = saveDocument(client.getJerseyClient(), patientIdentifier, "document-with-image-content");
		logger.debug("saved document metadata, document uri: " + createdDocument.getSelfUri());
		URI contentUri = createdDocument.getLink().getUriByTitle(LinkTitles.DocumentContent);

		InputStream image = this.createInputStreamForImage();
		client.getJerseyClient().resource(contentUri).type("image/jpg").put(image);

		ClientResponse response = client.getJerseyClient().resource(contentUri).get(ClientResponse.class);
		assertNotNull(response.getEntityInputStream());
	}

	@Test
	public void retrieveDocumentsAssociatedWithDataBringsBackOnlyItemsAssociatedWithData() {
		Patient patient = PatientLoader.loadPatient(patientIdentifier, userName);
		saveDocument(client.getJerseyClient(), patient.getPatientIdentifier(), "document-not-associated-with-data");

		DietEntry savedDiet = createDietEntry(patient);

//		Documents documents = getDocumentsAssociatedWithItem(savedDiet);
//		assertEquals(0, documents.size());

		Document savedDocument = saveDocument(client.getJerseyClient(), patientIdentifier, savedDiet);
		logger.debug("saved document metadata, document uri: " + savedDocument.getSelfUri());
		Documents documents = getDocumentsAssociatedWithItem(client.getJerseyClient(), savedDiet);
		assertEquals(1, documents.size());

		savedDocument = saveDocument(client.getJerseyClient(), patientIdentifier, savedDiet);
		logger.debug("saved document metadata, document uri: " + savedDocument.getSelfUri());
		documents = getDocumentsAssociatedWithItem(client.getJerseyClient(), savedDiet);
		assertEquals(2, documents.size());
		
		client.getJerseyClient().resource(savedDiet.getSelfUri()).delete();
	}

	@Test
	public void createDocumentAssociatedWithDiet() {
		Patient patient = PatientLoader.loadPatient(patientIdentifier, userName);
		DietEntry savedDiet = createDietEntry(patient);

		Document savedDocument = saveDocument(client.getJerseyClient(), patientIdentifier, savedDiet);
		logger.debug("saved document metadata, document uri: " + savedDocument.getSelfUri());

		Documents documents = getDocumentsAssociatedWithItem(client.getJerseyClient(), savedDiet);
		assertEquals("document-associated-with-data", documents.get(0).getDocumentTitle());
		
		client.getJerseyClient().resource(savedDiet.getSelfUri()).delete();
	}

	private DietEntry createDietEntry(Patient patient) {
		URI dietsUri = patient.getLink().getUriByTitle(LinkTitles.PatientDiet);
		DietEntry diet = new DietEntry();
		diet.setEntryDate(DateHelper.parseDate("1/1/2000"));
		diet.setCalories("100");
		diet.setCarbs("10");
		diet.setFat("1");
		diet.setProtein("0");
		diet.setMealType(MealType.SNACK);
		DietEntry savedDiet = client.getJerseyClient().resource(dietsUri).type(MediaType.APPLICATION_XML).accept("application/xml").post(DietEntry.class, diet);
		return savedDiet;
	}

	private InputStream createInputStreamForImage() {
		return this.getClass().getClassLoader().getResourceAsStream("Rad-image-1-small.jpg");
	}

	private Document saveDocument(Client jerseyClient, PatientIdentifier patientIdentifier, String documentTitle) {
		Patient patient = PatientLoader.loadPatient(patientIdentifier, userName);
		URI documentsUri = patient.getLink().getUriByTitle(LinkTitles.PatientDocuments);
		return saveDocument(jerseyClient, patientIdentifier, documentTitle, documentsUri);
	}

	private Document saveDocument(Client jerseyClient, PatientIdentifier patientIdentifier, String documentTitle, URI documentsUri) {
		Document document = new Document();
		document.setPatientIdentifier(patientIdentifier);
		document.setDocumentTitle(documentTitle);
		Document savedDocument = jerseyClient.resource(documentsUri).type(MediaType.APPLICATION_XML).accept("application/xml").post(Document.class, document);
		return savedDocument;
	}

	private Document saveDocument(Client jerseyClient, PatientIdentifier patientIdentifier, DietEntry savedDiet) {
		URI relatedDocumentUri = savedDiet.getLink().getUriByTitle(LinkTitles.RelatedDocuments);
		String title = "document-associated-with-data";
		return saveDocument(jerseyClient, patientIdentifier, title, relatedDocumentUri);
	}

	private Documents getDocumentsAssociatedWithItem(Client jerseyClient, DietEntry savedDiet) {
		URI relatedDocumentUri = savedDiet.getLink().getUriByTitle(LinkTitles.RelatedDocuments);
		logger.debug("retrieve documents: " + relatedDocumentUri);
		Documents documents = jerseyClient.resource(relatedDocumentUri).get(Documents.class);
		return documents;
	}
}
