package com.agilex.healthcare.mobilehealthplatform.restservice.contactlog;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.net.URI;
import java.util.Date;

import javax.ws.rs.core.MediaType;

import org.junit.After;
import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.client.AuthenticationInfo;
import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.ContactLog;
import com.agilex.healthcare.mobilehealthplatform.domain.ContactLogs;
import com.agilex.healthcare.mobilehealthplatform.domain.LinkTitles;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.testutility.PatientLoader;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.ModeHelper;
import com.sun.jersey.api.client.UniformInterfaceException;

public class ContactLogResourceTest {
	private static final String INTEGRATIONTEST_USER = "zztest.patient75";
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(ContactLogResourceTest.class);
	private static MobileHealthClientTestVersion client = null;
	
	@BeforeClass
	public static void checkWhetherTestsAreRunningInCIHMode() {
		Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev", "dev"));
		AuthenticationInfo authenticationInfo = new AuthenticationInfo(INTEGRATIONTEST_USER, "pass", "oauth");
		client = TestHelper.createMobileHealthClient(authenticationInfo);
	}
	

	@AfterClass
	public static void destroy() {
		if (client != null) {
			client.getJerseyClient().destroy();
			client = null;
		}
	}
	
	@After
	public void cleanup() {
		URI contactLogUri = getContactLogsUri(getIntegrationTestPatient(), INTEGRATIONTEST_USER);
		ContactLogs contactLogs = client.getJerseyClient().resource(contactLogUri).get(ContactLogs.class);
		
		for(ContactLog log : contactLogs) {
			URI selfUri = log.getSelfUri();
			client.getJerseyClient().resource(selfUri).delete();
		}
	}

	private ContactLog init(PatientIdentifier patientIdentifier, String userName) {
		ContactLog log = new ContactLog();
		log.setActivity("act-01");
		log.setContact("con-01");
		log.setNote("not-01");
		log.setTask("tas-01");
		log.setLogTime(DateHelper.parseDateTime("1/1/2000 11:00:00"));
		log.setPatientIdentifier(patientIdentifier);
		saveContactLog(log, userName);

		log = new ContactLog();
		log.setActivity("act-02");
		log.setContact("con-02");
		log.setNote("not-02");
		log.setTask("tas-02");
		log.setLogTime(DateHelper.parseDateTime("1/2/2000 11:00:00"));
		log.setPatientIdentifier(patientIdentifier);
		saveContactLog(log, userName);

		log = new ContactLog();
		log.setActivity("act-03");
		log.setContact("con-03");
		log.setNote("not-03");
		log.setTask("tas-03");
		log.setLogTime(DateHelper.parseDateTime("1/3/2000 11:00:00"));
		log.setPatientIdentifier(patientIdentifier);
		saveContactLog(log, userName);

		log = new ContactLog();
		log.setActivity("act-04");
		log.setContact("con-04");
		log.setNote("not-04");
		log.setTask("tas-04");
		log.setLogTime(DateHelper.parseDateTime("1/4/2000 11:00:00"));
		log.setPatientIdentifier(patientIdentifier);
		saveContactLog(log, userName);

		log = new ContactLog();
		log.setActivity("act-05");
		log.setContact("con-05");
		log.setNote("not-05");
		log.setTask("tas-05");
		log.setLogTime(DateHelper.parseDateTime("1/5/2000 11:00:00"));
		log.setPatientIdentifier(patientIdentifier);
		ContactLog contact = saveContactLog(log, userName);

		return contact;
	}

	@Test
	public void getZeroResultsForContactLog() {
		String userName = "zztest.patient76";
		AuthenticationInfo authenticationInfo = new AuthenticationInfo(userName, "pass", "oauth");
		MobileHealthClientTestVersion noDataClient = TestHelper.createMobileHealthClient(authenticationInfo);
		
		PatientIdentifier patientIdentifer = new PatientIdentifier("EDIPI", "NO-DATA-MOCK");
		URI contactLogUri = getContactLogsUri(patientIdentifer, userName);
		System.out.println("Contact URI" + contactLogUri.toString());
		ContactLogs zeroResults = noDataClient.getJerseyClient().resource(contactLogUri).get(ContactLogs.class);
		
		assertEquals(0, zeroResults.size());
		
		noDataClient.getJerseyClient().destroy();
	}

	@Test
	public void testCreateContactLog() {
		PatientIdentifier patientIdentifer = getIntegrationTestPatient();
		ContactLog newLog = new ContactLog();
		newLog.setActivity("Phone Call");
		newLog.setPatientIdentifier(patientIdentifer);
		newLog.setNote("New Log To Test");
		newLog.setContact("George      ");
		newLog.setLogTime(new Date());
		newLog.setTask("Some Task");

		URI postUri = getContactLogsUri(patientIdentifer, INTEGRATIONTEST_USER);
		ContactLog savedContactLog = client.getJerseyClient().resource(postUri).type(MediaType.APPLICATION_XML).accept("application/xml").post(ContactLog.class, newLog);
		assertNotNull(savedContactLog);
		assertEquals("Phone Call", savedContactLog.getActivity());

		URI contactLogUri = getContactLogsUri(patientIdentifer, INTEGRATIONTEST_USER);
		ContactLogs retrievedContactLogs = client.getJerseyClient().resource(contactLogUri).get(ContactLogs.class);
		assertTrue(retrievedContactLogs.size() > 0);
	}

	@Test
	public void testGetContactLogs() {
		PatientIdentifier patientIdentifer = getIntegrationTestPatient();
		init(patientIdentifer, INTEGRATIONTEST_USER);
		URI contactLogUri = getContactLogsUri(patientIdentifer, INTEGRATIONTEST_USER);
		ContactLogs manyResults = client.getJerseyClient().resource(contactLogUri).get(ContactLogs.class);
		logger.info("getContactLogs has a result count of " + manyResults.size());
		assertTrue(manyResults.size() >= 5);
	}

	@Test
	public void testUpdateContactLog() {
		PatientIdentifier patientIdentifer = getIntegrationTestPatient();
		ContactLog contactLog = init(patientIdentifer, INTEGRATIONTEST_USER);

		contactLog.setContact("cont-mod");
		ContactLog modifiedContactLog = client.getJerseyClient().resource(contactLog.getSelfUri()).type(MediaType.APPLICATION_XML).accept("application/xml").put(ContactLog.class, contactLog);
		assertNotNull(modifiedContactLog);
		assertEquals("cont-mod", modifiedContactLog.getContact());
	}

	@Test(expected = UniformInterfaceException.class)
	public void testDeleteContactLog() {
		PatientIdentifier patientIdentifer = getIntegrationTestPatient();

		ContactLog contactLog = new ContactLog();
		contactLog.setPatientIdentifier(patientIdentifer);
		contactLog.setActivity("act-06");
		contactLog.setContact("con-06");
		contactLog.setNote("not-06");
		contactLog.setTask("tas-06");

		contactLog = saveContactLog(contactLog, INTEGRATIONTEST_USER);
		assertNotNull(contactLog);
		assertNotNull(contactLog.getId());

		client.getJerseyClient().resource(contactLog.getSelfUri()).delete();

		ContactLog retrievedLog = client.getJerseyClient().resource(contactLog.getSelfUri()).get(ContactLog.class);
		assertNull(retrievedLog);
	}

	private ContactLog saveContactLog(ContactLog log, String userName) {
		URI uri = getContactLogsUri(log.getPatientIdentifier(), userName);
		return client.getJerseyClient().resource(uri).type(MediaType.APPLICATION_XML).accept("application/xml").post(ContactLog.class, log);
	}

	private URI getContactLogsUri(PatientIdentifier patientIdentifier, String userName) {
		Patient patient = PatientLoader.loadPatient(patientIdentifier, userName);
		return patient.getLink().getUriByTitle(LinkTitles.PatientContactLogs);
	}

	@Test
	public void getByEventDate() {
		PatientIdentifier patientIdentifer = getIntegrationTestPatient();
		init(patientIdentifer, INTEGRATIONTEST_USER);
		URI contactEventsUri = getContactLogsUri(patientIdentifer, INTEGRATIONTEST_USER);
		ContactLogs contactLogs = client.getJerseyClient().resource(contactEventsUri).queryParam("date", "1/2/2000").get(ContactLogs.class);
		assertEquals(1, contactLogs.size());
	}

	@Test
	public void getByDateRange() {
		PatientIdentifier patientIdentifer = getIntegrationTestPatient();
		init(patientIdentifer, INTEGRATIONTEST_USER);
		URI contactEventsUri = getContactLogsUri(patientIdentifer, INTEGRATIONTEST_USER);
		ContactLogs contactLogs = client.getJerseyClient().resource(contactEventsUri).queryParam("startdate", "1/2/2000").queryParam("enddate", "1/4/2000").get(ContactLogs.class);
		assertEquals(3, contactLogs.size()); 
	}

	private PatientIdentifier getIntegrationTestPatient() {
		return new PatientIdentifier("EDIPI", "INT-TEST-MOCK");
	}

}
