package com.agilex.healthcare.mobilehealthplatform.restservice.assessmentresults;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;

import java.net.URI;
import java.util.Date;
import java.util.UUID;

import javax.ws.rs.core.MediaType;

import org.junit.After;
import org.junit.Assume;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentQuestion;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentQuestionChoices;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResponse;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResponses;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResult;
import com.agilex.healthcare.mobilehealthplatform.domain.LinkTitles;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.code.AssessmentCode;
import com.agilex.healthcare.testutility.IntegrationTestConfiguration;
import com.agilex.healthcare.utility.ModeHelper;
import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.ClientResponse;

public class PublicAssessmentResultsResourceTest {
	Client jerseyClient;
	private static final String ANONYMOUS = AssessmentCode.ANONYMOUS_RESULT_PATIENT_ID.getUniqueId();
	private static final int UNAUTHORIZED = 401;
	private PatientIdentifier gallowId = new PatientIdentifier("EDIPI", "PATID26");

	@BeforeClass
	public static void checkWhetherTestsAreRunningInCIHMode() {
		Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev", "dev"));
	}
	
	@Before
	public void setup() {
		jerseyClient = new Client();
	}
	
	@After
	public void tearDown() {
		jerseyClient.destroy();
		jerseyClient = null;
	}

	@Test
	public void postAnonymousResultForNull() {
		AssessmentResult anonymousResult = createAnonymousResult(null);
		AssessmentResult savedResult = postAndRetrieveResult(anonymousResult);
		assertNotNull(savedResult);
		assertEquals(ANONYMOUS, savedResult.getPatientId());
	}

	@Test
	public void postAnonymousResultForTestPatient() {
		AssessmentResult anonymousResult = createAnonymousResult(gallowId);
		AssessmentResult savedResult = postAndRetrieveResult(anonymousResult);
		assertNotNull(savedResult);
		assertEquals(ANONYMOUS, savedResult.getPatientId());
	}

	private ClientResponse postResult(AssessmentResult resultToPost) {
		URI publicAssessmentResultsUri = IntegrationTestConfiguration.getResourceUri(LinkTitles.AssessmentResults);
		ClientResponse response = jerseyClient.resource(publicAssessmentResultsUri).type(MediaType.APPLICATION_JSON).post(ClientResponse.class, resultToPost);
		return response;
	}

	private AssessmentResult postAndRetrieveResult(AssessmentResult anonymousResult) {
		URI publicAssessmentResultsUri = IntegrationTestConfiguration.getResourceUri(LinkTitles.AssessmentResults);
		AssessmentResult response = jerseyClient.resource(publicAssessmentResultsUri).type(MediaType.APPLICATION_JSON).post(AssessmentResult.class, anonymousResult);
		return response;
	}

	private AssessmentResult createAnonymousResult(PatientIdentifier patientIdentifier) {
		AssessmentResult assessmentResult = createAndPopulateSampleResult(patientIdentifier);
		assessmentResult.setAuthenticationStrategy(AssessmentCode.AUTHENTICATION_STRATEGY_ANONYMOUS);
		assessmentResult.setAssessmentId("testAnonymous");
		return assessmentResult;
	}

	private AssessmentResult createNonAnonymousResult(PatientIdentifier patientIdentifier) {
		AssessmentResult assessmentResult = createAndPopulateSampleResult(patientIdentifier);
		assessmentResult.setAuthenticationStrategy(AssessmentCode.AUTHENTICATION_STRATEGY_BASIC);
		assessmentResult.setAssessmentId("strain");
		return assessmentResult;
	}

	private AssessmentResult createAndPopulateSampleResult(PatientIdentifier patientIdentifier) {
		AssessmentResult assessmentResult = new AssessmentResult();

		assessmentResult.setUniqueTitle(UUID.randomUUID().toString());
		assessmentResult.setVersion("1.0");
		assessmentResult.setAuthenticationStrategy(AssessmentCode.AUTHENTICATION_STRATEGY_ANONYMOUS);
		assessmentResult.setNotes("This assessment had these notes.");
		assessmentResult.setAssessmentId("testAssessmentId");
		assessmentResult.setPatientIdentifier(patientIdentifier);
		assessmentResult.setDateTaken(new Date());
		assessmentResult.setInProgress(false);
		assessmentResult.setWaitingPeriod(7);

		AssessmentResponses responses = new AssessmentResponses();
		populateAssessmentResponses(responses);
		assessmentResult.setResponses(responses);

		return assessmentResult;
	}

	private void populateAssessmentResponses(AssessmentResponses responses) {
		AssessmentResponse response1 = new AssessmentResponse();
		response1.setQuestion(new AssessmentQuestion());
		response1.setSelectedChoices(new AssessmentQuestionChoices());
		responses.add(response1);

		AssessmentResponse response2 = new AssessmentResponse();
		response2.setQuestion(new AssessmentQuestion());
		response2.setSelectedChoices(new AssessmentQuestionChoices());
		responses.add(response2);
	}
}