package com.agilex.healthcare.mobilehealthplatform.restservice.assessmentresults;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.net.URI;
import java.util.Date;
import java.util.UUID;

import javax.ws.rs.core.MediaType;

import org.junit.After;
import org.junit.Assume;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.client.AuthenticationInfo;
import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.Assessment;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentEligibility;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentProperties;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentProperty;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentQuestion;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentQuestionChoice;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentQuestionChoices;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResponse;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResponses;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResult;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResultReport;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResults;
import com.agilex.healthcare.mobilehealthplatform.domain.Assessments;
import com.agilex.healthcare.mobilehealthplatform.domain.GraphData;
import com.agilex.healthcare.mobilehealthplatform.domain.GraphDataSeries;
import com.agilex.healthcare.mobilehealthplatform.domain.LinkTitles;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.code.AssessmentCode;
import com.agilex.healthcare.testutility.PatientLoader;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.ModeHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.ClientResponse;

public class PatientAssessmentsResourceTest {
	private static final String ANONYMOUS = AssessmentCode.ANONYMOUS_RESULT_PATIENT_ID.getUniqueId();
	MobileHealthClientTestVersion client = null;
	private static PatientIdentifier aldieId = new PatientIdentifier("EDIPI", "PATID26");
	private static PatientIdentifier aldieEnterpriseId = new PatientIdentifier("EDIPI", "PATID26");
	Patient aldiePatient = PatientLoader.loadPatient(aldieId, "zztest.patient26");
	private String haAssessmentId = "A1";
	private String strainAssessmentId = "strain";
	private String ptsdAssessmentId = "ptsd";
	private String anonymousAssessmentId = "testAnonymous";

	@BeforeClass
	public static void checkWhetherTestsAreRunningInCIHMode() {
		Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev", "dev"));
	}
	
	@Before
	public void createClient() {
		AuthenticationInfo authenticationInfo = new AuthenticationInfo("zztest.patient26", "pass", "oauth");
		client = TestHelper.createMobileHealthClient(authenticationInfo);
	}
	
	@After
	public void destroyClient() {
		client.getJerseyClient().destroy();
		client = null;
	}

	@Test
	public void postAnonymousResultForNull() {
		AssessmentResult anonymousResult = generateAnonymousResult(null);
		AssessmentResult savedResult = postAndRetrieveResult(anonymousResult);
		assertNotNull(savedResult);
		assertEquals(ANONYMOUS, savedResult.getPatientId());
	}

	@Test
	public void postAnonymousResultForTestPatient() {
		AssessmentResult anonymousResult = generateAnonymousResult(aldieId);
		AssessmentResult savedResult = postAndRetrieveResult(anonymousResult);
		assertNotNull(savedResult);
		assertEquals(ANONYMOUS, savedResult.getPatientId());
	}

	@Test
	public void postAuthenticatedResultForTestPatient() {
		AssessmentResult authenticatedResult = generateAuthenticatedResult(aldieId, strainAssessmentId);
		AssessmentResult result = postAndRetrieveResult(authenticatedResult);
		assertNotNull(result);
		assertEquals(aldieEnterpriseId, result.getPatientIdentifier());
	}

	@Test
	public void postDraftForTestPatient() {
		AssessmentResult draft = generateDraft(aldieId);
		AssessmentResult result = postAndRetrieveResult(draft);
		assertNotNull(result);
		assertEquals(aldieEnterpriseId, result.getPatientIdentifier());
	}

	@Test
	public void getDraftForTestPatient() {
		AssessmentResult draft = generateDraft(aldieId);
		postResult(draft);
		AssessmentResult retrievedDraft = getDraft(aldiePatient, draft.getAssessmentId());
		assertNotNull(retrievedDraft);
	}

	@Test
	public void attemptToFetchDraftsForPatientWithoutDrafts() {
		String userName = "zztest.patient76";
		AuthenticationInfo authenticationInfo = new AuthenticationInfo(userName, "pass", "oauth");
		MobileHealthClientTestVersion noDataClient = TestHelper.createMobileHealthClient(authenticationInfo);
		
		Patient patient = PatientLoader.loadPatient(new PatientIdentifier("EDIPI", "NO-DATA-MOCK"), userName);

		Client jerseyClient = noDataClient.getJerseyClient();
		Assessment assessment = getAssessment(patient, haAssessmentId, jerseyClient);
		URI patientAssessmentDraftsUri = assessment.getLink().getUriByTitle(LinkTitles.PatientAssessmentDraft);
		
		AssessmentResult response = jerseyClient.resource(patientAssessmentDraftsUri).type(MediaType.APPLICATION_JSON).get(AssessmentResult.class);
		assertEquals(response.getUniqueId(), "placeHolderAssessmentResult");
		
		jerseyClient.destroy();
	}
	
	@Test
	public void getPastResultsDoesNotError() {
		String userName = "zztest.patient76";
		AuthenticationInfo authenticationInfo = new AuthenticationInfo(userName, "pass", "oauth");
		MobileHealthClientTestVersion noDataClient = TestHelper.createMobileHealthClient(authenticationInfo);
		
		Patient patient = PatientLoader.loadPatient(new PatientIdentifier("EDIPI", "NO-DATA-MOCK"), userName);

		String[] assessmentIdsToTest = { haAssessmentId, ptsdAssessmentId, strainAssessmentId };
		for (String assessmentId : assessmentIdsToTest) {
			
			Client jerseyClient = noDataClient.getJerseyClient();
			Assessment assessment = getAssessment(patient, assessmentId, jerseyClient);
			URI patientAssessmentResultsUri = assessment.getLink().getUriByTitle(LinkTitles.PatientAssessmentResults);
			
			AssessmentResults retrievedResults = jerseyClient.resource(patientAssessmentResultsUri).type(MediaType.APPLICATION_JSON)
					.get(AssessmentResults.class);
			
			if (retrievedResults.size() > 0) {
				URI uriForResult = retrievedResults.get(0).getSelfLink().getHref();
				jerseyClient.resource(uriForResult).type(MediaType.APPLICATION_JSON).get(AssessmentResult.class);
			}
		}
		noDataClient.getJerseyClient().destroy();
	}

	@Test
	public void getReportForAuthenticatedResult() {
		AssessmentResult nonAnonymousResult = generateAuthenticatedResult(aldieId, strainAssessmentId);
		AssessmentResult retrievedResult = postAndRetrieveResult(nonAnonymousResult);

		AssessmentResultReport report = retrievedResult.getAssessmentResultReport();
		assertNotNull(report);
		assertNotNull(report.getProperties());
	}

	@Test
	public void getGraphDataForSimpleResults() {
		for (int i = 0; i < 3; i++) {
			AssessmentResult assessmentResult = generateTestAuthenticatedResult(aldieId);
			postResult(assessmentResult);
		}
		GraphData graphData = getGraphData(haAssessmentId);

		assertNotNull(graphData);
		assertEquals(1, graphData.getDataSeriesCollection().size());
		GraphDataSeries scoreDataSeries = graphData.getDataSeriesCollection().get(0);
		assertEquals("Assessment Result Scores", scoreDataSeries.getTitle());
		assertTrue("Expected 3 data items, but got " + scoreDataSeries.size(), scoreDataSeries.size() >= 3);
	}

	@Test
	public void testSortingGraphData() {
		for (int i = 0; i < 5; i++) {
			AssessmentResult assessmentResult = generateTestAuthenticatedResult(aldieId);
			postResult(assessmentResult);
		}
		GraphData graphData = getGraphData(haAssessmentId);

		assertNotNull(graphData);
		assertEquals(1, graphData.getDataSeriesCollection().size());
		GraphDataSeries scoreDataSeries = graphData.getDataSeriesCollection().get(0);
		assertEquals("Assessment Result Scores", scoreDataSeries.getTitle());
		assertTrue(scoreDataSeries.size() >= 5);
		for (int i = 0; i < 4; i++) {
			assertTrue(scoreDataSeries.get(i).getX() >= scoreDataSeries.get(i + 1).getX());
		}
	}

	@Test
	public void alwaysEligibleForAssessmentWithNoWaitingPeriod() {
		AssessmentEligibility eligibility = getEligibilty(aldiePatient, haAssessmentId);

		assertTrue(eligibility.isEligibile());
		assertEquals(0, eligibility.getDaysLeft());
		assertNull(eligibility.getMessage());
	}

	@Test
	public void notEligibleForRecentAssessment() {
		AssessmentResult resultToPost = generateAuthenticatedResult(aldieId, strainAssessmentId);
		resultToPost.setAssessmentId(strainAssessmentId);
		postResult(resultToPost);
		AssessmentEligibility eligibility = getEligibilty(aldiePatient, strainAssessmentId);

		assertFalse(eligibility.isEligibile());
		assertEquals(7, eligibility.getDaysLeft());
		assertNotNull(eligibility.getMessage());
	}

	@Test
	public void verifyStrainReportGetsSaved() {
		AssessmentResult assessmentResult = generateAuthenticatedResult(aldieId, strainAssessmentId);
		assessmentResult = postAndRetrieveResult(assessmentResult);

		String savedAssessmentResultReportSummary = assessmentResult.getAssessmentResultReport().getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY);
		assertTrue(NullChecker.isNotNullish(savedAssessmentResultReportSummary));

		Client jerseyClient = client.getJerseyClient();
		AssessmentResult retrievedAssessmentResult = jerseyClient.resource(assessmentResult.getSelfUri()).get(AssessmentResult.class);
		String retrievedAssessmentResultReportSummary = retrievedAssessmentResult.getAssessmentResultReport().getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY);
		assertTrue(NullChecker.isNotNullish(retrievedAssessmentResultReportSummary));

		assertEquals(savedAssessmentResultReportSummary, retrievedAssessmentResultReportSummary);
	}

	@Test
	public void verifyPTSDReportGetsSaved() {
		AssessmentResult assessmentResult = generateAuthenticatedResult(aldieId, ptsdAssessmentId);
		assessmentResult = postAndRetrieveResult(assessmentResult);

		String expectedAssessmentResult = assessmentResult.getAssessmentResultReport().getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY);
		assertTrue(NullChecker.isNotNullish(expectedAssessmentResult));

		Client jerseyClient = client.getJerseyClient();
		assessmentResult = jerseyClient.resource(assessmentResult.getSelfUri()).get(AssessmentResult.class);
		String actualAssessmentResult = assessmentResult.getAssessmentResultReport().getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY);
		assertTrue(NullChecker.isNotNullish(actualAssessmentResult));

		assertEquals(expectedAssessmentResult, actualAssessmentResult);
	}

	private ClientResponse postResult(AssessmentResult resultToPost) {
		Client jerseyClient = client.getJerseyClient();
		Assessment assessment = getAssessment(aldiePatient, resultToPost.getAssessmentId(), jerseyClient);
		URI patientAssessmentResultsUri = assessment.getLink().getUriByTitle(LinkTitles.PatientAssessmentResults);
		
		ClientResponse response = jerseyClient.resource(patientAssessmentResultsUri).type(MediaType.APPLICATION_JSON).post(ClientResponse.class, resultToPost);
		return response;
	}

	private AssessmentResult postAndRetrieveResult(AssessmentResult resultToPost) {
		Client jerseyClient = client.getJerseyClient();
		Assessment assessment = getAssessment(aldiePatient, resultToPost.getAssessmentId(), jerseyClient);
		URI patientAssessmentResultsUri = assessment.getLink().getUriByTitle(LinkTitles.PatientAssessmentResults);
		
		AssessmentResult result = jerseyClient.resource(patientAssessmentResultsUri).type(MediaType.APPLICATION_JSON).post(AssessmentResult.class, resultToPost);
		return result;
	}

	private AssessmentEligibility getEligibilty(Patient patient, String assessmentId) {
		Client jerseyClient = client.getJerseyClient();
		Assessment assessment = getAssessment(patient, assessmentId, jerseyClient);
		URI eligibilityUri = assessment.getLink().getUriByTitle(LinkTitles.PatientAssessmentEligibility);
		
		AssessmentEligibility eligibility = jerseyClient.resource(eligibilityUri).type(MediaType.APPLICATION_JSON).get(AssessmentEligibility.class);
		return eligibility;
	}

	private GraphData getGraphData(String assessmentId) {
		Client jerseyClient = client.getJerseyClient();
		Assessment assessment = getAssessment(aldiePatient, assessmentId, jerseyClient);
		URI assessmentResultsUri = assessment.getLink().getUriByTitle(LinkTitles.PatientAssessmentGraphData);
		
		GraphData graphData = jerseyClient.resource(assessmentResultsUri).type(MediaType.APPLICATION_JSON).get(GraphData.class);
		return graphData;
	}

	private Assessment getAssessment(Patient patient, String assessmentId, Client jerseyClient) {
		URI patientAssessmentsUri = patient.getLink().getUriByTitle(LinkTitles.Assessments);
		Assessments assessments = jerseyClient.resource(patientAssessmentsUri).get(Assessments.class);
		Assessment assessment = assessments.getAssessmentById(assessmentId);

		if (assessment == null) {
			throw new RuntimeException("Unable to find assessment " + assessmentId);
		}
		return assessment;
	}

	private AssessmentResult getDraft(Patient patient, String assessmentId) {
		Client jerseyClient = client.getJerseyClient();
		Assessment assessment = getAssessment(patient, assessmentId, jerseyClient);
		URI patientAssessmentDraftsUri = assessment.getLink().getUriByTitle(LinkTitles.PatientAssessmentDraft);
		
		AssessmentResult assessmentResult = jerseyClient.resource(patientAssessmentDraftsUri).type(MediaType.APPLICATION_JSON).get(AssessmentResult.class);
		return assessmentResult;
	}

	private AssessmentResult generateAnonymousResult(PatientIdentifier userId) {
		AssessmentResult assessmentResult = generateMockResult(anonymousAssessmentId);
		assessmentResult.setAuthenticationStrategy(AssessmentCode.AUTHENTICATION_STRATEGY_ANONYMOUS);
		return assessmentResult;
	}

	private AssessmentResult generateAuthenticatedResult(PatientIdentifier patientIdentifier, String assessmentId) {
		AssessmentResult assessmentResult = generateMockResult(assessmentId);
		assessmentResult.setAuthenticationStrategy(AssessmentCode.AUTHENTICATION_STRATEGY_BASIC);
		return assessmentResult;
	}

	private AssessmentResult generateTestAuthenticatedResult(PatientIdentifier patientIdentifier) {
		AssessmentResult assessmentResult = generateMockResult(haAssessmentId);
		assessmentResult.setAuthenticationStrategy(AssessmentCode.AUTHENTICATION_STRATEGY_BASIC);
		return assessmentResult;
	}

	private AssessmentResult generateDraft(PatientIdentifier patientIdentifier) {
		AssessmentResult assessmentResult = generateMockResult(strainAssessmentId);
		assessmentResult.setAuthenticationStrategy(AssessmentCode.AUTHENTICATION_STRATEGY_BASIC);
		assessmentResult.setInProgress(true);
		return assessmentResult;
	}

	private AssessmentResult generateMockResult(String assessmentId) {
		AssessmentResult assessmentResult = new AssessmentResult();
		assessmentResult.setUniqueTitle(UUID.randomUUID().toString());
		assessmentResult.setUniqueId(UUID.randomUUID().toString());
		assessmentResult.setVersion("1.0");
		assessmentResult.setAuthenticationStrategy(AssessmentCode.AUTHENTICATION_STRATEGY_ANONYMOUS);
		assessmentResult.setNotes("This assessment had these notes.");
		assessmentResult.setAssessmentId(assessmentId);

		assessmentResult.setDateTaken(new Date());
		assessmentResult.setInProgress(false);
		assessmentResult.setScoringAlgorithm(AssessmentCode.SCORING_ALGORITHM_ZARIT);

		int[] responsePointValues = { 0, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
		AssessmentResponses assessmentResponses = new AssessmentResponses();
		int questionNumber = 1;
		for (int responsePointValue : responsePointValues) {
			AssessmentResponse assessmentResponse = new AssessmentResponse();
			AssessmentQuestionChoices selectedChoices = new AssessmentQuestionChoices();
			AssessmentQuestionChoice selectedChoice = new AssessmentQuestionChoice();
			AssessmentProperties properties = new AssessmentProperties();
			AssessmentQuestion assessmentQuestion = new AssessmentQuestion();

			properties.add(new AssessmentProperty(AssessmentCode.RESULT_PROPERTY_POINT_VALUE, "" + responsePointValue));
			selectedChoice.setProperties(properties);
			selectedChoices.add(selectedChoice);
			assessmentResponse.setSelectedChoices(selectedChoices);

			assessmentQuestion.setQuestionNumber(questionNumber);
			assessmentResponse.setQuestion(assessmentQuestion);

			assessmentResponses.add(assessmentResponse);
			questionNumber++;
		}

		assessmentResult.setResponses(assessmentResponses);
		return assessmentResult;
	}
}