package com.agilex.healthcare.mobilehealthplatform.restservice.assessment;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.net.URI;

import javax.ws.rs.core.UriBuilder;

import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.Assessment;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentQuestion;
import com.agilex.healthcare.mobilehealthplatform.domain.Assessments;
import com.agilex.healthcare.mobilehealthplatform.domain.AtomLink;
import com.agilex.healthcare.mobilehealthplatform.domain.AtomLinks;
import com.agilex.healthcare.mobilehealthplatform.domain.LinkTitles;
import com.agilex.healthcare.mobilehealthplatform.domain.atom.rel;
import com.agilex.healthcare.testutility.IntegrationTestConfiguration;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.ModeHelper;

public class PublicAssessmentResourceTest {
	private static MobileHealthClientTestVersion client;

	@BeforeClass
	public static void checkWhetherTestsAreRunningInCIHMode() {
		Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev", "dev"));
		client = TestHelper.createMobileHealthClient();
	}
	
	@AfterClass
	public static void tearDown() {
		if (client != null) {
			client.getJerseyClient().destroy();
			client = null;
		}
	}

	@Test
	public void testNumberOfAssessments() {
		Assessments assessments = getAllAssessments();
		assertNotNull(assessments);
		assertTrue(assessments.size() > 0);
	}

	@Test
	public void testFetchStrainAssessment() {
		Assessment assessment = getStrainAssessment();
		assertNotNull(assessment);
		assertEquals("Zarit Caregiver Assessment", assessment.getUniqueTitle());
	}

	@Test
	public void testFetchHealthAssessmentAssessments() {
		Assessments assessments = getHealthAssessmentAssessments();
		assertNotNull(assessments);
		assertEquals(4, assessments.size());
		assertTrue(assessmentsContainsAssessmentId(assessments, "A1"));
		assertTrue(assessmentsContainsAssessmentId(assessments, "A2"));
		assertTrue(assessmentsContainsAssessmentId(assessments, "A3"));
        assertTrue(assessmentsContainsAssessmentId(assessments, "A4"));
	}

	private boolean assessmentsContainsAssessmentId(Assessments assessmentsInHealthAssessment, String assessmentId) {
		for (Assessment assessment : assessmentsInHealthAssessment) {
			if (assessment.getId().equals(assessmentId)) {
				return true;
			}
		}
		return false;
	}

	@Test
	public void testEachAssessmentHasLinksToQuestions() {
		AtomLinks assessmentLinks = getRelatedLinks(getAllAssessments().getLink());
		for (AtomLink assessmentLink : assessmentLinks) {
			Assessment retrievedAssessment = client.getJerseyClient().resource(assessmentLink.getHref()).get(Assessment.class);
			assertNotNull(retrievedAssessment);
			AtomLinks linksToQuestions = getRelatedLinks(retrievedAssessment.getLink());
			assertNotNull(linksToQuestions);
		}
	}

	@Test
	public void testEachQuestionExists() {
		AtomLinks assessmentLinks = getRelatedLinks(getAllAssessments().getLink());
		for (AtomLink assessmentLink : assessmentLinks) {
			Assessment retrievedAssessment = client.getJerseyClient().resource(assessmentLink.getHref()).get(Assessment.class);
			AtomLinks questionLinks = getRelatedLinks(retrievedAssessment.getLink());
			for (AtomLink questionLink : questionLinks) {
				AssessmentQuestion retrievedQuestion = client.getJerseyClient().resource(questionLink.getHref()).get(AssessmentQuestion.class);
				assertNotNull(retrievedQuestion);
				assertNotNull(retrievedQuestion.getId());
				assertNotNull(retrievedQuestion.getType());
				assertNotNull(retrievedQuestion.getPrompt());
			}
		}
	}

	private AtomLinks getRelatedLinks(AtomLinks allLinks) {
		AtomLinks relatedLinks = new AtomLinks();
		for (AtomLink atomLink : allLinks) {
			if (atomLink.getRel().equals(rel.related)) {
				relatedLinks.add(atomLink);
			}
		}
		return relatedLinks;
	}

	private Assessments getAllAssessments() {
		URI assessmentsUri = IntegrationTestConfiguration.getResourceUri(LinkTitles.Assessments);
		return client.getJerseyClient().resource(assessmentsUri).get(Assessments.class);
	}

	private Assessment getStrainAssessment() {
		URI allAssessmentsUri = IntegrationTestConfiguration.getResourceUri(LinkTitles.Assessments);
		URI strainAssessmentsUri = UriBuilder.fromUri(allAssessmentsUri).queryParam("assessmentID", "strain").build();
		URI strainUri = getStrainUriFromAssessments(strainAssessmentsUri);
		Assessment assessment = client.getJerseyClient().resource(strainUri).get(Assessment.class);
		return assessment;
	}

	private Assessments getHealthAssessmentAssessments() {
		URI assessmentsUri = IntegrationTestConfiguration.getResourceUri(LinkTitles.HealthAssessmentAssessments);
		return client.getJerseyClient().resource(assessmentsUri).get(Assessments.class);
	}

	private URI getStrainUriFromAssessments(URI assessmentsUri) {
		Assessments assessments = client.getJerseyClient().resource(assessmentsUri).get(Assessments.class);
		Assessment strain = findAssessmentById(assessments, "strain");
		return strain.getSelfUri();
	}

	private Assessment findAssessmentById(Assessments assessments, String id) {
		Assessment match = null;
		for (Assessment assessment : assessments) {
			if (assessment.getId().contentEquals(id)) {
				match = assessment;
			}
		}
		return match;
	}
}