package com.agilex.healthcare.mobilehealthplatform.restservice.appointments;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.net.URI;
import java.util.Date;

import javax.ws.rs.core.UriBuilder;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.Test;
import org.w3c.dom.Document;
import org.w3c.dom.NodeList;

import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.Appointment;
import com.agilex.healthcare.mobilehealthplatform.domain.Appointments;
import com.agilex.healthcare.mobilehealthplatform.domain.LinkTitles;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.testutility.PatientLoader;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.ModeHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.XpathHelper;

public class PatientAppointmentsTest {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(PatientAppointmentsTest.class);

	private static MobileHealthClientTestVersion client;

	@BeforeClass
	public static void checkWhetherTestsAreRunningInCIHMode() {
		Assume.assumeTrue(ModeHelper.isMode("dev"));
		client = TestHelper.createMobileHealthClient();
		Logger.getRootLogger().setLevel(Level.INFO);
	}

	@AfterClass
	public static void destroy() {
		if (client != null) {
			client.getJerseyClient().destroy();
			client = null;
		}
	}
	
	@Test
	public void retrieveAppointments() {
		URI appointmentsUri = getAppointmentsUri();
		checkReturnCount(appointmentsUri, 111);
	}

	@Test
	public void retrievePastAppointments() {
		URI appointmentsUri = getPastAppointmentsUri();
		checkReturnCount(appointmentsUri, 5);
	}

	@Test
	public void retrieveAppointmentsInOrder() {
		URI appointmentsUri = getAppointmentsUri();
		ensureAppointmentOrder(appointmentsUri);
	}

	@Test
	public void retrievePastAppointmentsInOrder() {
		URI appointmentsUri = getPastAppointmentsUri();
		ensureAppointmentOrder(appointmentsUri);
	}

	@Test
	public void retrieveWithDateEntryFilter() {
		URI uri = getAppointmentsUri();
		String entryDate = "1/1/2012";
		String startDate = null;
		String endDate = null;
		int expectedCount = 1;
		String expectedEverythingAfterDate = "1/1/2012";
		String expectedEverythingBeforeDate = "1/2/2012";
		checkReturnCount(uri, entryDate, startDate, endDate, expectedCount, expectedEverythingAfterDate, expectedEverythingBeforeDate);
	}

	@Test
	public void retrieveWithDateEntryFilterInTheFuture() {
		URI uri = getAppointmentsUri();
		String entryDate = "1/4/2022";
		String startDate = null;
		String endDate = null;
		int expectedCount = 1;
		String expectedEverythingAfterDate = "1/4/2022";
		String expectedEverythingBeforeDate = "1/5/2022";
		checkReturnCount(uri, entryDate, startDate, endDate, expectedCount, expectedEverythingAfterDate, expectedEverythingBeforeDate);
	}

	@Test
	public void retrievePastWithDateEntryFilter() {
		URI uri = getAppointmentsUri();
		String entryDate = "1/1/2012";
		String startDate = null;
		String endDate = null;
		int expectedCount = 1;
		String expectedEverythingAfterDate = "1/1/2012";
		String expectedEverythingBeforeDate = "1/2/2012";
		checkReturnCount(uri, entryDate, startDate, endDate, expectedCount, expectedEverythingAfterDate, expectedEverythingBeforeDate);
	}

	@Test
	public void retrievePastWithDateEntryFilterInTheFuture() {
		URI uri = getAppointmentsUri();
		String entryDate = "1/4/2022";
		String startDate = null;
		String endDate = null;
		int expectedCount = 1;
		String expectedEverythingAfterDate = "1/4/2022";
		String expectedEverythingBeforeDate = "1/5/2022";
		checkReturnCount(uri, entryDate, startDate, endDate, expectedCount, expectedEverythingAfterDate, expectedEverythingBeforeDate);
	}

	@Test
	public void retrieveWithDateRangeFilter() {
		URI uri = getAppointmentsUri();
		String entryDate = null;
		String startDate = "1/1/2012";
		String endDate = "1/5/2012";
		int expectedCount = 3;
		String expectedEverythingAfterDate = "1/1/2012";
		String expectedEverythingBeforeDate = "1/5/2012";
		checkReturnCount(uri, entryDate, startDate, endDate, expectedCount, expectedEverythingAfterDate, expectedEverythingBeforeDate);
	}

	private Appointments checkReturnCount(URI appointmentsBaseUri, int expectedCount) {
		return checkReturnCount(appointmentsBaseUri, null, null, null, expectedCount, null, null);

	}

	private Appointments checkReturnCount(URI appointmentsBaseUri, String entryDate, String startDate, String endDate, int expectedCount, String expectedEverythingAfterDate, String expectedEverythingBeforeDate) {
		UriBuilder uriBuilder = UriBuilder.fromUri(appointmentsBaseUri);
		if (NullChecker.isNotNullish(entryDate))
			uriBuilder.queryParam("date", entryDate);
		if (NullChecker.isNotNullish(startDate))
			uriBuilder.queryParam("startDate", startDate);
		if (NullChecker.isNotNullish(endDate))
			uriBuilder.queryParam("endDate", endDate);

		URI uri = uriBuilder.build();
		logger.debug("get: " + uri);
		Appointments appointments = client.getJerseyClient().resource(uri).get(Appointments.class);

		assertEquals(expectedCount, appointments.size());
		for (Appointment appointment : appointments) {
			Date appointmentDate = appointment.getAppointmentStartDate();
			if (expectedEverythingAfterDate != null)
				assertTrue(appointmentDate.after(DateHelper.parseDate(expectedEverythingAfterDate)));
			if (expectedEverythingBeforeDate != null)
				assertTrue(appointmentDate.before(DateHelper.parseDate(expectedEverythingBeforeDate)));
		}
		return appointments;
	}

	private void ensureAppointmentOrder(URI appointmentsUri) {
		Appointments appointments = client.getJerseyClient().resource(appointmentsUri).get(Appointments.class);

		Date previousDate = null;
		for (Appointment appointment : appointments) {
			if (previousDate != null) {
				assertTrue(appointment.getAppointmentStartDate().before(previousDate) ||appointment.getAppointmentStartDate().equals(previousDate));
			}
			previousDate = appointment.getAppointmentStartDate();
		}
	}

	@Test
	public void retrieveAppointmentsWithNoResults() {
		MobileHealthClientTestVersion noDataClient = TestHelper.createMobileHealthClient("zztest.patient76","pass","oauth");
		URI appointmentsUri = getPastAppointmentsUriForPatientWithNoData();
		Document appointmentsXml = noDataClient.getAsXml(appointmentsUri);
		NodeList appointmentsList = XpathHelper.getNodeList(appointmentsXml, "//m:appointment");
		assertEquals(0, appointmentsList.getLength());
		noDataClient.getJerseyClient().destroy();
	}

	@Test
	public void retrieveAppointmentsAsJson() {
		URI appointmentsUri = getPastAppointmentsUri();
		String appointmentsJson = client.getJerseyClient().resource(appointmentsUri).accept("application/json").get(String.class);
		System.out.println("json: " + appointmentsJson);
	}

	private URI getPastAppointmentsUriForPatientWithNoData() {
		PatientIdentifier patientIdentifier = new PatientIdentifier("EDIPI", "NO-DATA-MOCK");
		Patient patient = PatientLoader.loadPatient(patientIdentifier, "zztest.patient76", "pass", "oauth");
		URI appointmentsUri = patient.getLink().getUriByTitle(LinkTitles.PatientPastAppointments);
		return appointmentsUri;
	}

	private URI getPastAppointmentsUri() {
		Patient patient = PatientLoader.loadTestPatient();
		URI appointmentsUri = patient.getLink().getUriByTitle(LinkTitles.PatientPastAppointments);
		return appointmentsUri;
	}

	private URI getAppointmentsUri() {

		Patient patient = PatientLoader.loadTestPatient();
		URI appointmentsUri = patient.getLink().getUriByTitle(LinkTitles.PatientAppointments);
		logger.debug("base uri: " + appointmentsUri);
		return appointmentsUri;
	}
}
