package com.agilex.healthcare.mobilehealthplatform.restservice.appointments;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.net.URI;
import java.util.List;

import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriBuilder;

import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.Ignore;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequest;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequestDetailCode;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequestInProcess;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequests;
import com.agilex.healthcare.mobilehealthplatform.domain.LinkTitles;
import com.agilex.healthcare.mobilehealthplatform.domain.ResourceDirectory;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestStatus;
import com.agilex.healthcare.testutility.ResourceDirectoryLoader;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.ModeHelper;
import com.agilex.healthcare.utility.NullChecker;

public class ClerkAppointmentRequestResourceTest {
	private static MobileHealthClientTestVersion client;

	@BeforeClass
	public static void checkWhetherTestsAreRunningInCIHMode() {
		Assume.assumeTrue(ModeHelper.isMode("dev"));
		client = TestHelper.createMobileHealthClient();
	}
	
	@AfterClass
	public static void destroy() {
		if (client != null) {
			client.getJerseyClient().destroy();
			client = null;
		}
	}
	
	@Test
	public void retrieveAppointmentsForDCVAMC() {
		URI appointmentRequestsUri = getAppointmentRequestsUri();

		UriBuilder uriBuilder = UriBuilder.fromUri(appointmentRequestsUri);
		uriBuilder.queryParam("facilityName", "DCVAMC");
		URI uri = uriBuilder.build();

		AppointmentRequests appointmentRequests = client.getJerseyClient().resource(uri).get(AppointmentRequests.class);

		assertTrue(appointmentRequests.size()>0);
		verifyCancelledAppointmentRequestsWithOutDetailCodeAreNotReturned(appointmentRequests);
	}
	

	private void verifyCancelledAppointmentRequestsWithOutDetailCodeAreNotReturned(AppointmentRequests appointmentRequests) {
		for (AppointmentRequest appointmentRequest : appointmentRequests) {
			if(isCancelledAppointmentRequest(appointmentRequest) && doesAppointmentRequestContainDetailCode(appointmentRequest) == false){
				fail("Appointment requests include cancelled appointments with out detail code");
			}
		}
	}

	private boolean doesAppointmentRequestContainDetailCode(AppointmentRequest appointmentRequest) {
		List<AppointmentRequestDetailCode> arDetailCodes = appointmentRequest.getAppointmentRequestDetailCode();
		if(NullChecker.isNotNullish(arDetailCodes)){
			return true;
		}
		return false;
	}

	private boolean isCancelledAppointmentRequest(AppointmentRequest appointmentRequest) {
		if(AppointmentRequestStatus.CANCELLED.getName().equalsIgnoreCase(appointmentRequest.getStatus())){
			return true;
		}
		return false;
	}


	@Test
	public void retrieveAppointmentsForChicago() {
		URI appointmentRequestsUri = getAppointmentRequestsUri();

		UriBuilder uriBuilder = UriBuilder.fromUri(appointmentRequestsUri);
		uriBuilder.queryParam("parentSiteCode", "537");
		uriBuilder.queryParam("startDate", "5/1/2012");
		uriBuilder.queryParam("endDate", "5/30/2012");
		
		URI uri = uriBuilder.build();

		AppointmentRequests appointmentRequests = client.getJerseyClient().resource(uri).get(AppointmentRequests.class);

		assertEquals("Chicago is not supported in pilot", 0, appointmentRequests.size());
	}

	@Test
	public void retrieveAppointments() {
		URI appointmentRequestsUri = getAppointmentRequestsUri();
		AppointmentRequests appointmentRequests = client.getJerseyClient().resource(appointmentRequestsUri).get(AppointmentRequests.class);
		assertTrue(appointmentRequests.size()>0);
	}
	
	@Test
	public void validateAppointmentInProcessWorkflow(){
		AppointmentRequest submittedAppointmentRequest = getSubmittedAppointmentRequestForDCVAMC();
		URI inProcessUri = UriBuilder.fromUri(submittedAppointmentRequest.getSelfUri()).path("/inprocess").build();
		
		//Set Inprocess lock
		AppointmentRequestInProcess appInProcess = client.getJerseyClient().resource(inProcessUri).type(MediaType.APPLICATION_XML).accept("application/xml").post(AppointmentRequestInProcess.class);
		assertEquals(submittedAppointmentRequest.getUniqueId(), appInProcess.getAppointmentRequestId());
		
		// Check the appointment request to make sure it has beingProcessedBy object
		AppointmentRequest retrieveAppointmentRequest = client.getJerseyClient().resource(submittedAppointmentRequest.getSelfUri()).get(AppointmentRequest.class);
		assertEquals(appInProcess.getFirstName(), retrieveAppointmentRequest.getBeingProcessedBy().getFirstName());
		assertEquals(appInProcess.getLastName(), retrieveAppointmentRequest.getBeingProcessedBy().getLastName());
		
		//Remove inprocess lock
		client.getJerseyClient().resource(inProcessUri).delete(String.class);
		retrieveAppointmentRequest = client.getJerseyClient().resource(submittedAppointmentRequest.getSelfUri()).get(AppointmentRequest.class);
		assertNull(retrieveAppointmentRequest.getBeingProcessedBy());
	}

	@Ignore
	public void validateAppointmentInProcessCreationReturns400IfItsBeenCreatedAlreadyByAnotherProcess(){
		// David: This is no longer a valid test until we send two requests as different users
		// TODO: Ask if something like this has been done before, and implement here
		
		AppointmentRequest submittedAppointmentRequest = getSubmittedAppointmentRequestForDCVAMC();
		URI inProcessUri = UriBuilder.fromUri(submittedAppointmentRequest.getSelfUri()).path("/inprocess").build();
		
		//Set Inprocess lock
		client.getJerseyClient().resource(inProcessUri).type(MediaType.APPLICATION_XML).accept(MediaType.APPLICATION_XML).post(AppointmentRequestInProcess.class);

		boolean expectionOccurred= false;
		try {
			AppointmentRequestInProcess appInProcess = client.getJerseyClient().resource(inProcessUri).type(MediaType.APPLICATION_XML).accept(MediaType.APPLICATION_XML).post(AppointmentRequestInProcess.class);
		} catch (Exception e) {
			expectionOccurred = true;
		}
		
		assertTrue(expectionOccurred);
		//Remove inprocess lock
		client.getJerseyClient().resource(inProcessUri).delete(String.class);

	}
	
	private AppointmentRequest getSubmittedAppointmentRequestForDCVAMC() {
		URI appointmentRequestsUri = getAppointmentRequestsUri();

		UriBuilder uriBuilder = UriBuilder.fromUri(appointmentRequestsUri);
		uriBuilder.queryParam("facilityName", "DCVAMC");
		URI uri = uriBuilder.build();

		AppointmentRequests appointmentRequests = client.getJerseyClient().resource(uri).get(AppointmentRequests.class);

		AppointmentRequest submittedAR = null;
		for (AppointmentRequest appointmentRequest : appointmentRequests) {
			if(appointmentRequest.getStatus().equalsIgnoreCase(AppointmentRequestStatus.SUBMITTED.getName())){
				submittedAR = appointmentRequest;
				break;
			}
		}
		
		return submittedAR;
	}

	private URI getAppointmentRequestsUri() {
		ResourceDirectory resourceDirectory = ResourceDirectoryLoader.loadResourceDirectory();
		URI appointmentRequestsUri = resourceDirectory.getLink().getUriByTitle(LinkTitles.AppointmentRequests);
		return appointmentRequestsUri;
	}
}
