package com.agilex.healthcare.mobilehealthplatform.restservice.appointments;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.net.URI;
import java.util.Calendar;
import java.util.Date;
import java.util.HashSet;
import java.util.List;

import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriBuilder;

import org.junit.*;

import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequest;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequestDetailCode;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequestMessage;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequestMessages;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequests;
import com.agilex.healthcare.mobilehealthplatform.domain.DetailCode;
import com.agilex.healthcare.mobilehealthplatform.domain.Facility;
import com.agilex.healthcare.mobilehealthplatform.domain.LinkTitles;
import com.agilex.healthcare.mobilehealthplatform.domain.MhpUser;
import com.agilex.healthcare.mobilehealthplatform.domain.ResourceDirectory;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestPurposeOfVisit;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestStatus;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestType;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestVisitType;
import com.agilex.healthcare.testutility.ResourceDirectoryLoader;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.ModeHelper;
import com.sun.jersey.api.client.UniformInterfaceException;
@Ignore
public class AppointmentRequestsResourceTest {

    private static MobileHealthClientTestVersion client;
    private static URI appointmentRequestsUri;
    private static AppointmentRequest mentalHealthAppointmentRequest1;
    private static AppointmentRequest secondAppointmentRequest;

    @BeforeClass
    public static void fixtureSetup() {
        Assume.assumeTrue(ModeHelper.isMode("dev", "va-veteran-dev"));
        client = TestHelper.createMobileHealthClient();

        appointmentRequestsUri = getAppointmentRequestsUri();

        cleanSubmittedRequests();
    }

    @AfterClass
    public static void tearDown(){

        if (client != null) {
            client.getJerseyClient().destroy();
            client = null;
        }
    }

    @Before
    public void appointmentRequestSetup(){
        mentalHealthAppointmentRequest1 = createValidAppointmentRequest();
        mentalHealthAppointmentRequest1 = client.getJerseyClient().resource(appointmentRequestsUri).type(MediaType.APPLICATION_XML).accept("application/xml").post(AppointmentRequest.class, mentalHealthAppointmentRequest1);
    }

    private static void cleanSubmittedRequests() {
        //get all appointment requests for the patient
        AppointmentRequests requests = retrieveAppointments(appointmentRequestsUri);

        for (AppointmentRequest request : requests) {
            if (request.getStatus().equals(AppointmentRequestStatus.SUBMITTED.getName())){
                cancelAppointmentRequest(request);
            }
        }
    }

    @After public void appointmentRequestTearDown(){
        if (mentalHealthAppointmentRequest1 != null){
            mentalHealthAppointmentRequest1 =  cancelAppointmentRequest(mentalHealthAppointmentRequest1);
        }

        if (secondAppointmentRequest != null){
            secondAppointmentRequest = cancelAppointmentRequest(secondAppointmentRequest);
        }
    }

    private static AppointmentRequest cancelAppointmentRequest(AppointmentRequest appointmentRequest) {
        appointmentRequest.setActiveFlag(false);
        appointmentRequest.setStatus(AppointmentRequestStatus.CANCELLED.getName());
        AppointmentRequest result = client.getJerseyClient().resource(appointmentRequest.getSelfUri()).type(MediaType.APPLICATION_XML).accept("application/xml").put(AppointmentRequest.class, appointmentRequest);
        return result;
    }

    @Test
    public void saveAppointmentRequest(){
        assertNotNull(mentalHealthAppointmentRequest1.getDataIdentifier().getUniqueId());
        assertNotNull(mentalHealthAppointmentRequest1.getSelfUri());

        assertEquals(mentalHealthAppointmentRequest1.isTextMessagingAllowed(), mentalHealthAppointmentRequest1.getPatient().isTextMessagingAllowed());
        assertEquals(mentalHealthAppointmentRequest1.getTextMessagingPhoneNumber(), mentalHealthAppointmentRequest1.getPatient().getTextMessagingPhoneNumber());
    }

    @Test
    public void saveAppointmentRequestWithScriptInjection(){
        AppointmentRequest appointmentRequest = createValidAppointmentRequest();
        appointmentRequest.setEmail("<script>alert(1)</script>@gmail.com");
        appointmentRequest = client.getJerseyClient().resource(appointmentRequestsUri).type(MediaType.APPLICATION_JSON).accept("application/json").post(AppointmentRequest.class, appointmentRequest);
        assertEquals("<(script>alert(1)<(/script>@gmail.com", appointmentRequest.getEmail());
    }

    @Test
    public void retrieveAppointments() {
        AppointmentRequests retrieveAppointments = retrieveAppointments(appointmentRequestsUri);
        assertTrue(retrieveAppointments.size()>0);
    }

    @Test
    public void retrieveAppointmentRequestsUsingDateFilter(){
        UriBuilder uriBuilder = UriBuilder.fromUri(appointmentRequestsUri);
        uriBuilder.queryParam("startDate", DateHelper.formatDate(DateHelper.getYesterday()));
        uriBuilder.queryParam("endDate", DateHelper.formatDate(DateHelper.getTomorrow()));
        URI uri = uriBuilder.build();
        AppointmentRequests retrieveAppointments = retrieveAppointments(uri);
        assertTrue(retrieveAppointments.size() > 0);
    }

    @Test
    public void retrieveEmptyAppointmentRequestsMessages(){
        UriBuilder uriBuilder = UriBuilder.fromUri(appointmentRequestsUri);
        uriBuilder.queryParam("startDate", DateHelper.formatDate(DateHelper.getYesterday()));
        uriBuilder.queryParam("endDate", DateHelper.formatDate(DateHelper.getTomorrow()));
        URI uri = uriBuilder.build();
        AppointmentRequests retrieveAppointments = retrieveAppointments(uri);
        URI relatedMessageUri = retrieveAppointments.get(0).getLink().getUriByTitle(LinkTitles.AppointmentRequestMessages);


        UriBuilder messageUriBuilder = UriBuilder.fromUri(relatedMessageUri);
        URI messageUri = messageUriBuilder.build();

        AppointmentRequestMessages messages = retrieveAppointmentRequestMessages(messageUri);
        assertEquals(0, messages.size());
        assertNotNull(messages);
    }

    @Test
    public void doNotSaveWithNonMatchingAppointmentRequestId(){

        String appointmentRequestId = mentalHealthAppointmentRequest1.getDataIdentifier().getUniqueId();
        String nonMatchingAppointmentRequestId = "0";
        int returnStatus = 0;
        int HTTP_STATUS_NULL_CONTENT=204;

        AppointmentRequestMessage message = new AppointmentRequestMessage();
        message.setAppointmentRequestId(nonMatchingAppointmentRequestId);
        message.setMessageText("Please call me");
        message.setSenderId("D123401");
        message.setPatientIdentifier("EDIPI", "D123401");
        message.setMessageDateTime(DateHelper.getToday());

        URI appointmentRequestMessagesUri = getAppointmentRequestMessagesUri(appointmentRequestId);
        if (appointmentRequestMessagesUri == null){
            Assert.fail("Unable to determine appointmentRequestMessagesUri");
        }

        AppointmentRequestMessage savedAppointmentRequestMessage1 = null;
        try{
            savedAppointmentRequestMessage1 = client.getJerseyClient().resource(appointmentRequestMessagesUri).type(MediaType.APPLICATION_XML).accept("application/xml").post(AppointmentRequestMessage.class, message);
        } catch(UniformInterfaceException e){
            returnStatus = e.getResponse().getStatus();
        }

        assertEquals(HTTP_STATUS_NULL_CONTENT, returnStatus);
        assertNull(savedAppointmentRequestMessage1);

    }

    @Test
    public void saveAppointmentRequestMessage(){
        String appointmentRequestId = mentalHealthAppointmentRequest1.getDataIdentifier().getUniqueId();

        AppointmentRequestMessage message = new AppointmentRequestMessage();
        message.setAppointmentRequestId(appointmentRequestId);
        message.setMessageText("Please call me");
        message.setSenderId("D123401");
        message.setMessageDateTime(DateHelper.getToday());

        URI appointmentRequestMessagesUri = getAppointmentRequestMessagesUri(appointmentRequestId);
        if (appointmentRequestMessagesUri == null){
            Assert.fail("Unable to determine appointmentRequestMessagesUri");
        }
        AppointmentRequestMessage result = client.getJerseyClient().resource(appointmentRequestMessagesUri).type(MediaType.APPLICATION_XML).accept("application/xml").post(AppointmentRequestMessage.class, message);

        assertNotNull(result);
        assertNotNull(result.getDataIdentifier());
        assertNotNull(result.getDataIdentifier().getUniqueId());
    }

    @Test
    public void retrieveAppointmentRequestMessages(){

        String appointmentRequestId = mentalHealthAppointmentRequest1.getDataIdentifier().getUniqueId();

        AppointmentRequestMessage message = new AppointmentRequestMessage();
        message.setAppointmentRequestId(appointmentRequestId);
        message.setMessageText("Please call me");
        message.setSenderId("D123401");
        message.setMessageDateTime(DateHelper.getToday());

        URI appointmentRequestMessagesUri = getAppointmentRequestMessagesUri(appointmentRequestId);
        if (appointmentRequestMessagesUri == null){
            Assert.fail("Unable to determine appointmentRequestMessagesUri");
        }
        AppointmentRequestMessage savedAppointmentRequestMessage1 = client.getJerseyClient().resource(appointmentRequestMessagesUri).type(MediaType.APPLICATION_XML).accept("application/xml").post(AppointmentRequestMessage.class, message);
        assertNotNull(savedAppointmentRequestMessage1);

        AppointmentRequestMessages messages = retrieveAppointmentRequestMessages(appointmentRequestMessagesUri);

        assertNotNull(messages);
        assertEquals(1, messages.size());

        message = new AppointmentRequestMessage();
        message.setAppointmentRequestId(appointmentRequestId);
        message.setMessageText("Please call me again");
        message.setSenderId("D60000");
        message.setMessageDateTime(DateHelper.getToday());

        AppointmentRequestMessage savedAppointmentRequestMessage2 = client.getJerseyClient().resource(appointmentRequestMessagesUri).type(MediaType.APPLICATION_XML).accept("application/xml").post(AppointmentRequestMessage.class, message);

        assertNotNull(savedAppointmentRequestMessage2);

        messages = retrieveAppointmentRequestMessages(appointmentRequestMessagesUri);

        assertNotNull(messages);
        assertEquals(messages.size(), 2);

    }


    @Test
    public void updateAppointmentRequest(){
        mentalHealthAppointmentRequest1.setStatus(AppointmentRequestStatus.BOOKED.getName());
        addARDetailCode(mentalHealthAppointmentRequest1, "DETCODE1");
        Date beforeUpdate = DateHelper.getYesterday();
        mentalHealthAppointmentRequest1.setLastUpdatedDate(beforeUpdate);
        mentalHealthAppointmentRequest1 = client.getJerseyClient().resource(mentalHealthAppointmentRequest1.getSelfUri()).type(MediaType.APPLICATION_XML).accept("application/xml").put(AppointmentRequest.class, mentalHealthAppointmentRequest1);
        assertNotNull(mentalHealthAppointmentRequest1);
        assertNotNull(AppointmentRequestStatus.BOOKED.getName(), mentalHealthAppointmentRequest1.getStatus());
        assertTrue(mentalHealthAppointmentRequest1.getLastUpdatedDate().after(beforeUpdate));
    }

    @Test
    public void updateAppointmentRequestWithMultipleDetailCodes(){
        mentalHealthAppointmentRequest1.setStatus(AppointmentRequestStatus.CANCELLED.getName());
        addARDetailCode(mentalHealthAppointmentRequest1, "DETCODE8");
        addARDetailCode(mentalHealthAppointmentRequest1, "DETCODE7");
        Date beforeUpdate = DateHelper.getYesterday();
        mentalHealthAppointmentRequest1.setLastUpdatedDate(beforeUpdate);
        mentalHealthAppointmentRequest1 = client.getJerseyClient().resource(mentalHealthAppointmentRequest1.getSelfUri()).type(MediaType.APPLICATION_XML).accept("application/xml").put(AppointmentRequest.class, mentalHealthAppointmentRequest1);
        assertNotNull(mentalHealthAppointmentRequest1);
        assertNotNull(AppointmentRequestStatus.CANCELLED.getName(), mentalHealthAppointmentRequest1.getStatus());
        assertTrue(mentalHealthAppointmentRequest1.getLastUpdatedDate().after(beforeUpdate));
    }

    @Test
    public void validateSaveSecondAppointmentRequest(){
        secondAppointmentRequest = createValidAppointmentRequest();
        secondAppointmentRequest.setParentRequest(mentalHealthAppointmentRequest1);
        secondAppointmentRequest.setSecondRequest(true);
        mentalHealthAppointmentRequest1.setSecondRequestSubmitted(true);

        secondAppointmentRequest = 	client.getJerseyClient().resource(appointmentRequestsUri).type(MediaType.APPLICATION_XML).accept("application/xml").post(AppointmentRequest.class, secondAppointmentRequest);

        assertEquals(true, secondAppointmentRequest.isSecondRequest());
        assertNull(secondAppointmentRequest.getParentRequest());

        AppointmentRequest parentRequest = client.getJerseyClient().resource(mentalHealthAppointmentRequest1.getSelfUri()).type(MediaType.APPLICATION_XML).accept("application/xml").get(AppointmentRequest.class);
        assertEquals(true, parentRequest.isSecondRequestSubmitted());
    }

    @Test
    public void validateSetProviderSeenAppointmentRequestResourceIsAvailable(){
    	//Not possible to test Provider specific resource in va-veteran-dev mode. Verifying if the Resource is available or not.
    	//There is seperate unit test at DataLayer level.
    	assertNotNull(mentalHealthAppointmentRequest1.getLink().getUriByTitle(LinkTitles.AppointmentRequestProviderSeenFlag));
    }

	private void addARDetailCode(AppointmentRequest appointmentRequest, String code){

        DetailCode detailCode = new DetailCode();
        detailCode.setCode(code);

        List<AppointmentRequestDetailCode> arDetailCodes = appointmentRequest.getAppointmentRequestDetailCode();
        AppointmentRequestDetailCode arDetailCode = new AppointmentRequestDetailCode();
        arDetailCode.setCreatedDate(new Date());
        arDetailCode.setDetailCode(detailCode);
        arDetailCode.setUserId("zztest.staff01");
        arDetailCodes.add(arDetailCode);
        appointmentRequest.setAppointmentRequestDetailCode(arDetailCodes);
    }

    private static AppointmentRequests retrieveAppointments(URI appointmentsBaseUri) {
        AppointmentRequests appointmentRequests = client.getJerseyClient().resource(appointmentsBaseUri).get(AppointmentRequests.class);

        return appointmentRequests;
    }

    private static URI getAppointmentRequestsUri() {
        ResourceDirectory resourceDirectory = ResourceDirectoryLoader.loadResourceDirectory();
        URI mhpUserUri = resourceDirectory.getLink().getUriByTitle(LinkTitles.MHPUser);

        MhpUser mhpUser = client.getJerseyClient().resource(mhpUserUri).get(MhpUser.class);

        URI appointmentRequestsUri = mhpUser.getLink().getUriByTitle(LinkTitles.Appointments);

        return appointmentRequestsUri;
    }

    private static URI getAppointmentRequestMessagesUri(String appointmentRequestId) {
        URI appointmentRequestsUri = getAppointmentRequestsUri();
        AppointmentRequests appointmentRequests = client.getJerseyClient().resource(appointmentRequestsUri).get(AppointmentRequests.class);

        for (AppointmentRequest request : appointmentRequests) {
            if(request.getAppointmentRequestId().equals(appointmentRequestId)){
                return request.getLink().getUriByTitle(LinkTitles.AppointmentRequestMessages);
            }
        }
        return null;
    }

    private AppointmentRequestMessages retrieveAppointmentRequestMessages(URI uri) {

        return client.getJerseyClient().resource(uri).get(AppointmentRequestMessages.class);
    }

    private static AppointmentRequest createValidAppointmentRequest() {
        String appointmentType = AppointmentRequestType.MENTAL_HEALTH.getName();
        String email = "test@agilex.com";
        String phoneNumber = "+1 (555) 555-5555 x55.55";
        String visitType = AppointmentRequestVisitType.OFFICE_VISIT.getName();
        String purposeOfVisit = AppointmentRequestPurposeOfVisit.OTHER.getName();
        String otherPurposeOfVisit = "Other purpose of visit specified test";
        String providerId = "PROV1";
        boolean secondRequest = false;

        Calendar cal = Calendar.getInstance();
        cal.add(Calendar.DATE, 7);
        String optionDate1 = DateHelper.formatDate(cal.getTime());
        String optionTime1 = "AM";
        cal.add(Calendar.DATE, 1);
        String optionDate2 = DateHelper.formatDate(cal.getTime());
        String optionTime2 = "PM";
        cal.add(Calendar.DATE, 2);
        String optionDate3 = DateHelper.formatDate(cal.getTime());
        String optionTime3 = "AM";
        HashSet<String> bestTimeToCall = new HashSet<String>();
        bestTimeToCall.add("9 AM - 11 AM");
        bestTimeToCall.add("11 AM - 1 PM");
        String textMessagingPhoneNumber = "555-222-3242";
        boolean textMessagingAllowed = true;

        String facilityAddress = "1201 Broad Block Blvd Richmond, VA 23249";
        String facilityCity = "Richmond";
        String facilityState = "VA";
        String facilityCode = "652";
        String facilityName = "Richmond VAMC";
        String facilityParentSiteCode = "688";
        String facilityType = "VAMC";
        Facility facility = createFacility(facilityAddress, facilityCity, facilityCode, facilityName, facilityParentSiteCode, facilityState, facilityType);

        AppointmentRequest a = new AppointmentRequest();

        a.setAppointmentType(appointmentType);
        a.setVisitType(visitType);
        a.setFacility(facility);
        a.setEmail(email);
        a.setPhoneNumber(phoneNumber);
        a.setOptionDate1(optionDate1);
        a.setOptionTime1(optionTime1);
        a.setOptionDate2(optionDate2);
        a.setOptionTime2(optionTime2);
        a.setOptionDate3(optionDate3);
        a.setOptionTime3(optionTime3);
        a.setBestTimetoCall(bestTimeToCall);
        a.setPurposeOfVisit(purposeOfVisit);
        a.setOtherPurposeOfVisit(otherPurposeOfVisit);
        a.setStatus(AppointmentRequestStatus.SUBMITTED.getName());
        a.setProviderId(providerId);
        a.setLastUpdatedDate(new Date());
        a.setSecondRequest(secondRequest);
        a.setProviderName("zztest staff01");
        a.setTextMessagingAllowed(textMessagingAllowed);
        a.setTextMessagingPhoneNumber(textMessagingPhoneNumber);

        return a;
    }

    private static Facility createFacility(String facilityAddress, String facilityCity, String facilityCode, String facilityName, String facilityParentSiteCode, String facilityState, String facilityType) {
        Facility facility = new Facility();
        facility.setAddress(facilityAddress);
        facility.setCity(facilityCity);
        facility.setFacilityCode(facilityCode);
        facility.setName(facilityName);
        facility.setParentSiteCode(facilityParentSiteCode);
        facility.setState(facilityState);
        facility.setType(facilityType);

        return facility;
    }


}
