package com.agilex.healthcare.mobilehealthplatform.restservice;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.net.URI;
import java.util.Date;

import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.client.AuthenticationInfo;
import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.mobilehealthplatform.domain.AtomLink;
import com.agilex.healthcare.mobilehealthplatform.domain.AtomLinks;
import com.agilex.healthcare.mobilehealthplatform.domain.LinkTitles;
import com.agilex.healthcare.mobilehealthplatform.domain.MhpUser;
import com.agilex.healthcare.mobilehealthplatform.domain.ResourceDirectory;
import com.agilex.healthcare.mobilehealthplatform.domain.RightOfAccessInfo;
import com.agilex.healthcare.testutility.IntegrationTestConfiguration;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.ModeHelper;
import com.sun.jersey.api.client.Client;

public class RightOfAccessResourceTest {
	
	private static ResourceDirectory resourceDirectory = null;
	private static MobileHealthClientTestVersion client;
	private static Client jerseyClient;
	private static MhpUser mhpUser = null;
	private static URI RightOfAccessUri = null;
	
	@BeforeClass
	public static void setUpResourceDirectoryAndUser() {
		Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev"));
		client = TestHelper.createMobileHealthClient();
		jerseyClient = new Client();
		URI resourceDirectoryURI = IntegrationTestConfiguration.getHealthAdapterResourceDirectoryUri();
		resourceDirectory = jerseyClient.resource(resourceDirectoryURI).accept("application/xml").get(ResourceDirectory.class);
		
		URI mhpUserResourceUri = getMhpUserResourceUri();
		
		mhpUser = getMhpUser(mhpUserResourceUri);
		RightOfAccessUri = getRightOfAccessURI(mhpUser);
	}
	
	@AfterClass
	public static void destroy() {
	//      Set ROA so other resource tests can execute
	      if(mhpUser != null){
	          RightOfAccessInfo originalRightOfAccessInfo = new RightOfAccessInfo(mhpUser.getId(), true, new Date());
	          clientPut(originalRightOfAccessInfo);
	      }
	      if(jerseyClient != null){
	          jerseyClient.destroy();
	          jerseyClient = null;
	      }
	      if(client != null){
	          client.getJerseyClient().destroy();
	          client = null;
	      }
	}

	private static URI getMhpUserResourceUri() {
		URI mhpUserResourceUri = null;
		AtomLinks links = resourceDirectory.getLink();
		mhpUserResourceUri = getUriForTitle(LinkTitles.MHPUser, links);
		return mhpUserResourceUri;
	}

	private static URI getUriForTitle(String title, AtomLinks links) {
		URI targetUri = null;
		for (AtomLink atomLink : links) {
			if(atomLink.getTitle().equals(title)){
				targetUri = atomLink.getHref();
				break;
			}
		}
		
		return targetUri;
	}
	
	private static URI getRightOfAccessURI(MhpUser mhpUser){
		return getUriForTitle(LinkTitles.RightOfAccess, mhpUser.getLink());
	}

	@Test
	public void checkAcceptingRightOfAccess() {
		RightOfAccessInfo originalRightOfAccessInfo = new RightOfAccessInfo(mhpUser.getId(), true, new Date());
		clientPut(originalRightOfAccessInfo);

		RightOfAccessInfo retrievedRightOfAccessInfo = getRightOfAccessInfo();
		assertEquals(true, retrievedRightOfAccessInfo.isRightOfAccessAccepted());
		assertTrue(DateHelper.calculateDeltaInDays(originalRightOfAccessInfo.getRightOfAccessDate(), retrievedRightOfAccessInfo.getRightOfAccessDate())==0);
	}

	@Test
	public void checkCancellingRightOfAccessForm() {
		RightOfAccessInfo originalRightOfAccessInfo = new RightOfAccessInfo(mhpUser.getId(), false, new Date());
		clientPut(originalRightOfAccessInfo);

		RightOfAccessInfo retrievedRightOfAccessInfo = getRightOfAccessInfo();
		
		assertEquals(false, retrievedRightOfAccessInfo.isRightOfAccessAccepted());
		assertTrue(DateHelper.calculateDeltaInDays(originalRightOfAccessInfo.getRightOfAccessDate(), retrievedRightOfAccessInfo.getRightOfAccessDate())==0);
	}
	
	@Test
	public void verifyRightOfAccessPDFisReturned(){
		String response = client.getJerseyClient().resource(RightOfAccessUri).accept("application/pdf").get(String.class);
		assertTrue(response.startsWith("%PDF"));
	}

	@Test
	public void verifyRightOfAccessImageisReturned(){
		URI rightOfAccessImageURI = getUriForTitle(LinkTitles.RightOfAccessImage, mhpUser.getLink());
		String response = client.getJerseyClient().resource(rightOfAccessImageURI).accept("image/png").get(String.class);
		assertTrue(response.startsWith("PNG", 1));
	}
	
	
	@Test(expected=Exception.class)
	public void valideErrorIsReturnedIfRightOfAccessIsNotAvailable(){
		String uri = IntegrationTestConfiguration.getMobileHealthServerBaseUri();
		AuthenticationInfo authenticationInfo = IntegrationTestConfiguration.getAuthenticationInformation();
		authenticationInfo.setUsername("zztest.patient26");
		authenticationInfo.setPassword("pass");
		MobileHealthClientTestVersion mobileHealthClient = new MobileHealthClientTestVersion(uri, authenticationInfo);
		
		mobileHealthClient.getJerseyClient().resource(RightOfAccessUri).get(Response.class);
	}
	
	private static MhpUser getMhpUser(URI mhpUserResourceUri) {
		return client.getJerseyClient().resource(mhpUserResourceUri).get(MhpUser.class);		
	}
	
	private RightOfAccessInfo getRightOfAccessInfo() {
		return client.getJerseyClient().resource(RightOfAccessUri).get(RightOfAccessInfo.class);
	}

	private static RightOfAccessInfo clientPut(RightOfAccessInfo rightOfAccessInfo) {
		return client.getJerseyClient().resource(RightOfAccessUri).type(MediaType.APPLICATION_XML).accept("application/xml").put(RightOfAccessInfo.class, rightOfAccessInfo);
	}
	
}