package com.agilex.healthcare.mobilehealthplatform.restservice;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.util.Calendar;
import java.util.Date;

import javax.ws.rs.core.MediaType;

import org.junit.Ignore;
import org.junit.Test;
import org.w3c.dom.Document;
import org.w3c.dom.Node;

import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.StreamHelper;
import com.agilex.healthcare.utility.XpathHelper;
import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.ClientResponse.Status;
import com.sun.jersey.api.client.WebResource;

public abstract class GeneralResourceTestBase {
	private boolean defaultDebug = true;
	private static final int number_of_runs_for_perf_test = 10000;
	private WebResource cachedResource = null;
	
	private Client jerseyClient;
	
	private final String GET = "GET";
	private final String HEAD = "HEAD";

	abstract protected String getResourcePath();

	protected WebResource getResourceHandle() {
		if (cachedResource == null) {
			String uri = getResourcePath();
			if (defaultDebug)
				System.out.println("initializing resource with URI " + uri);
			cachedResource = jerseyClient.resource(uri);
		}
		return cachedResource;
	}

	protected String getCanonicalResourceUri() {
		return getResourcePath();
	}

	private void runTest(String httpAction, String mediaType, boolean debug) {
		if (debug)
			System.out.println("running test for media type " + mediaType);


		jerseyClient = TestHelper.createMobileHealthClient().getJerseyClient();
		ClientResponse response = getResourceHandle().accept(mediaType).get(ClientResponse.class);
		if (debug) {
			System.out.println(response.toString());
			System.out.println((StreamHelper.streamToString(response.getEntityInputStream(), false)));
		}
		jerseyClient.destroy();
		jerseyClient = null;
		if (response.getClientResponseStatus() != Status.OK)
			throw new RuntimeException(response.toString());
	}

	private void runPerfTest(String mediaType, int runs) {
		this.defaultDebug = false;
		System.out.println("running perf test for media type " + mediaType);
		Calendar start = Calendar.getInstance();
		start.setTime(new Date());

		for (int i = 0; i < runs; i++) {
			runTest("GET", mediaType, false);
		}
		Calendar end = Calendar.getInstance();
		end.setTime(new Date());
		double deltaMs = (end.getTimeInMillis() - start.getTimeInMillis()) / 1000.0;
		System.out.println("delta:" + deltaMs + " (" + deltaMs / ((double) runs) + " avg)");
	}

	private void runMediaTest(String httpAction, String mediaType) {
		this.defaultDebug = true;
		runTest(httpAction, mediaType, false);
	}

	@Test
	public void getAsXml() {
		runMediaTest(GET, MediaType.APPLICATION_XML);
	}

	@Test
	public void headAsXml() {
		runMediaTest(HEAD, MediaType.APPLICATION_XML);
	}

	@Test
	public void getAsJson() {
		runMediaTest(GET, MediaType.APPLICATION_JSON);
	}

	@Test
	public void headAsJson() {
		runMediaTest(HEAD, MediaType.APPLICATION_JSON);
	}

	@Ignore
	public void getAsText() {
		runMediaTest(GET, MediaType.TEXT_PLAIN);
	}

	@Ignore
	public void headAsText() {
		runMediaTest(HEAD, MediaType.TEXT_PLAIN);
	}

	@Ignore
	public void getAsRdf() {
		runMediaTest(GET, "application/rdf+xml");
	}

	@Ignore
	public void headAsRdf() {
		runMediaTest(HEAD, "application/rdf+xml");
	}

	@Ignore
	public void getAsLightXml() {
		runMediaTest(GET, "application/light+xml");
	}

	@Ignore
	public void headAsLightXml() {
		runMediaTest(HEAD, "application/light+xml");
	}

	@Test(expected = Exception.class)
	public void getUnknownMediaType() {
		runMediaTest(GET, "unknown");
	}

	@Test(expected = Exception.class)
	public void headUnknownMediaType() {
		runMediaTest(HEAD, "unknown");
	}

	@Ignore
	public void perfTestAsXml() {
		runPerfTest(MediaType.APPLICATION_XML, number_of_runs_for_perf_test);
	}

	@Ignore
	public void perfTestAsJson() {
		runPerfTest(MediaType.APPLICATION_JSON, number_of_runs_for_perf_test);
	}

	@Ignore
	public void perfTestAsRdf() {
		runPerfTest("application/rdf+xml", number_of_runs_for_perf_test);
	}

	@Ignore
	public void checkThatResultContainsSelfLink() {
		String selfLink = getSelfLink();
		assertTrue(NullChecker.isNotNullish(selfLink));
	}

	@Ignore
	public void checkThatSelfLinkMatchesCanonicalUri() {
		String selfLink = getSelfLink();
		assertEquals(this.getCanonicalResourceUri().toLowerCase(), selfLink.toLowerCase());
	}

	private String getSelfLink() {
		String mediaType = MediaType.APPLICATION_XML;

		jerseyClient = TestHelper.createMobileHealthClient().getJerseyClient();
		Document response = getResourceHandle().accept(mediaType).get(Document.class);

		String selfLink = null;
		selfLink = XpathHelper.getString(response, "child::*/atom:link[@rel='self']/@href");
		jerseyClient.destroy();
		jerseyClient = null;
		return selfLink;
	}

	@Ignore
	public void checkThatEachElementHasNamespace() {
		String mediaType = MediaType.APPLICATION_XML;

		jerseyClient = TestHelper.createMobileHealthClient().getJerseyClient();
		Document response = getResourceHandle().accept(mediaType).get(Document.class);
		jerseyClient.destroy();
		assertThatElementAndChildrenHaveNamespaces(response.getFirstChild());
	}

	private void assertThatElementAndChildrenHaveNamespaces(Node node) {
		System.out.println("checking namespace for node " + node.getNodeName() + "; namespace=" + node.getNamespaceURI());
		if (node != null) {
			assertTrue("Node " + node.getNodeName() + " does not have a namespace", NullChecker.isNotNullish(node.getNamespaceURI()));
			System.out.println("good for this node, checking children");
			for (int i = 0; i < node.getChildNodes().getLength(); i++) {
				Node childNode = node.getChildNodes().item(i);
				assertThatElementAndChildrenHaveNamespaces(childNode);
			}
			System.out.println("checked namespace for node " + node.getNodeName() + " and children");
		}
	}
}
