package com.agilex.healthcare.mobilehealthplatform;

import java.io.BufferedWriter;
import java.io.Console;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriBuilder;

import org.codehaus.jackson.JsonGenerationException;
import org.codehaus.jackson.JsonNode;
import org.codehaus.jackson.JsonParser;
import org.codehaus.jackson.map.JsonMappingException;
import org.codehaus.jackson.map.ObjectMapper;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import cc.plural.jsonij.parser.ParserException;

import com.agilex.healthcare.mobilehealthplatform.client.AuthenticationInfo;
import com.agilex.healthcare.mobilehealthplatform.clientapi.MobileHealthClientTestVersion;
import com.agilex.healthcare.testutility.TestHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.XmlHelper;
import com.agilex.healthcare.utility.XpathHelper;
import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.ClientResponse;

public class TestDataCreator {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(TestDataCreator.class);

	private Client client;
	private String outputPath = "/mhp/testdata";
	private String xmlPath = "/mhp/testdata/xml";
	private String jsonPath = "/mhp/testdata/json";
	private HashMap<String, Boolean> retrievedUris = new HashMap<String, Boolean>();
	private URI baseUri;
	private List<String> hrefs = new ArrayList<String>();
	private List<String> vitalsSections = new ArrayList<String>();
	private boolean doVitalsLongitudinalOnce = true;
	private boolean doVitalsOperationalOnce = true;
	
	public TestDataCreator(URI baseUri, String outputPath) {

		this.outputPath = outputPath;
		this.xmlPath = this.outputPath + "/xml";
		this.jsonPath = this.outputPath + "/json";
		this.vitalsSections.add("BP");
		this.vitalsSections.add("pulse");
		this.vitalsSections.add("respiration");
		this.vitalsSections.add("temperature");
		this.vitalsSections.add("weight");
		this.vitalsSections.add("pain");
		this.vitalsSections.add("pulse-ox");
		
		File xmlDirectory = new File(xmlPath);
		if (xmlDirectory.exists() == false) {
			xmlDirectory.mkdir();
		}

		File jsonDirectory = new File(jsonPath);
		if (jsonDirectory.exists() == false) {
			jsonDirectory.mkdir();
		}
		AuthenticationInfo authInfo = new AuthenticationInfo("zztest.patient01","pass","oauth");

		MobileHealthClientTestVersion mhpClient = TestHelper.createMobileHealthClient(authInfo);
		this.client = mhpClient.getJerseyClient();
		this.baseUri = baseUri;
	}

	public void createXMLData(String uri, boolean getChildren) throws IOException {

		URI uriO;

		try {

			if (uri.startsWith("http://")) {
				uriO = new URI(uri);
			} else {
				uriO = UriBuilder.fromUri(baseUri).path(uri).build();
			}

		} catch (URISyntaxException e) {
			throw new RuntimeException("unable to create uri from " + uri);
		}

		createXMLData(uriO, getChildren);
	}

	public void createJSONData(String uri, boolean getChildren) throws IOException, ParserException, URISyntaxException {

		URI uriO;

		try {

			if (uri.startsWith("http://")) {
				uriO = new URI(uri);
			} else {
				uriO = UriBuilder.fromUri(baseUri).path(uri).build();
			}

		} catch (URISyntaxException e) {
			throw new RuntimeException("unable to create uri from " + uri);
		}
			
		findHrefInResponse(uriO);
		
	}

	public void createXMLData(URI uri, boolean getChildren) throws IOException {

		client.setConnectTimeout(30000);
		client.setReadTimeout(30000);

		logger.info("creating data for uri " + uri.toString());

		if (uri.toString().contains("document")) {
			logger.info("skipping documents " + uri.toString());
			return;
		}

		if (retrievedUris.containsKey(uri.toString())) {
			logger.info("abort - have already processed uri " + uri.toString());
			return;
		}
		retrievedUris.put(uri.toString(), true);

		if (!uriReturnsType(uri, MediaType.APPLICATION_XML)) {
			logger.info("abort - XML not returned for uri " + uri.toString());
			return;
		}

		logger.debug("retrieve " + uri.toString());

		Document response = client.resource(uri).accept(MediaType.APPLICATION_XML).get(Document.class);

		logger.debug("retrieved " + uri.toString());

		NodeList linkNodes = XpathHelper.getNodeList(response, "//atom:link");
		for (int i = 0; i < linkNodes.getLength(); i++) {

			Node linkNode = linkNodes.item(i);
			Node hrefAttibute = linkNode.getAttributes().getNamedItem("href");
			String href = hrefAttibute.getNodeValue();

			logger.debug("found child link " + href.toString());

			createXMLData(href, false);

			String updatedHref = determineFileName(href);
			hrefAttibute.setNodeValue(updatedHref);
		}

		writeResponseXMLToFile(response, uri);
	//	writeResponseJsonToFile(uri);
	}

	

	private void writeJsonToFile(String json, String fileName) throws IOException {

		File f = new File(this.jsonPath, fileName);

		logger.debug("writing out to file " + f.getPath());

		FileWriter fstream = new FileWriter(f);

		BufferedWriter out = new BufferedWriter(fstream);
		out.write(json);
		out.close();
	}

	private void writeResponseXMLToFile(Node response, URI uri) throws IOException {

		String serializedResponse = XmlHelper.xmlToString(response);

		File f = new File(this.xmlPath, determineFileName(uri) + ".xml");

		logger.debug("writing out to file " + f.getPath());

		FileWriter fstream = new FileWriter(f);

		BufferedWriter out = new BufferedWriter(fstream);
		out.write(serializedResponse);
		out.close();
	}

	private boolean uriReturnsType(URI uri, String mediaType) {

		try {

			logger.info("invoking head " + uri.toString());

			ClientResponse response = client.resource(uri).accept(mediaType).head();

			String mediaTypeReturned = response.getType().toString();
			logger.info("invoked head, returned mediatype " + mediaTypeReturned + " status " + response.getStatus());

			return (mediaTypeReturned.contains(mediaType));

		} catch (Exception e) {

			logger.error("invoked head, FAILED");
			return false;
		}
	}

	public String determineFileName(URI uri) {
		return determineFileName(uri.toString());
	}

	private String determineFileName(String uri) {

		String filename = uri.toString();

		filename = removeBase(uri);
		filename = filename.replace("http://", "");
		filename = filename.replace("/", "_");
		filename = filename.replace("?", "_");
		filename = filename.replace("=", "_");

		return filename;
	}

	private String removeBase(String uri) {

		uri = remove(uri, "^.*/rest/");
		uri = remove(uri, "^.*/rest");

		return uri;
	}

	private String remove(String subject, String patternToRemove) {

		Pattern pattern = Pattern.compile(patternToRemove, Pattern.CASE_INSENSITIVE);
		Matcher matcher = pattern.matcher(subject);

		String result = matcher.replaceFirst("");

		return result;
	}
	
	public void findHrefInResponse(URI uri) throws JsonGenerationException, JsonMappingException, IOException, URISyntaxException {
		
		client.setConnectTimeout(30000);
		client.setReadTimeout(30000);


		logger.info("creating data for uri " + uri.toString());

		if (uri.toString().contains("vitals/scope/longitudinal/graphdata")) {
			if (doVitalsLongitudinalOnce) {
				doVitalsLongitudinalOnce = false;
				for (String vitalSection : vitalsSections) {
					findHrefInResponse(UriBuilder.fromUri(uri).queryParam("section",vitalSection).build());	
				}
			}
		} 
		
		if (uri.toString().contains("vitals/scope/operational/graphdata")) {
			if (doVitalsOperationalOnce) {
				doVitalsOperationalOnce = false;
				for (String vitalSection : vitalsSections) {
					findHrefInResponse(UriBuilder.fromUri(uri).queryParam("section",vitalSection).build());	
				}
			}
		} 
		 
		
		if (uri.toString().contains("document")) {
			logger.info("skipping documents " + uri.toString());
			return;
		}

		if (retrievedUris.containsKey(uri.toString())) {
			logger.info("abort - have already processed uri " + uri.toString());
			return;
		}
		retrievedUris.put(uri.toString(), true);

		logger.debug("finding hrefs in " + uri.toString());
		
		if (!uriReturnsType(uri, MediaType.APPLICATION_JSON)) {
			logger.info("abort - JSON not returned for uri " + uri.toString());
			return;
		}
		
		
		if (uri.toString().equalsIgnoreCase("http://hadev.agilexhealth.com:8080/MobileHealthPlatformWeb/rest/metrics")) {
			logger.debug("FAILED that specific uri " + uri.toString() + "has been flagged with invalid data");
			return;
		}
		
		logger.debug("retrieving " + uri.toString());

		String response = client.resource(uri).accept(MediaType.APPLICATION_JSON).get(String.class);
		String responseString = response;
		
		logger.debug("retrieved " + uri.toString());

		
		ObjectMapper m = new ObjectMapper();
		JsonNode rootNode = m.readTree(responseString);
		findHref(rootNode);
		logger.debug("root " + rootNode);
		
		String jsonString = m.writeValueAsString(rootNode);
		
		for (int i = 0; i < hrefs.size(); i++) {
			
			String hrefValue = hrefs.get(i);
			jsonString = jsonString.replace("\"" + hrefValue + "\"", "\"" + determineFileName(new URI(hrefValue)) + ".json\"");
		}		

		writeJsonToFile(jsonString, determineFileName(uri) + ".json");
		
		System.out.println(jsonString);
	}

	private void findHref(JsonNode node) throws JsonGenerationException, JsonMappingException, IOException, URISyntaxException {
		if (node != null) {
			System.out.println("process node; " + node);

			if (NullChecker.isNotNullish(node.asText())) {
				System.out.println("text=" + node.asText());
			}

			
			JsonParser parser = node.traverse();
			while (parser.nextToken() != null) {
				String value = parser.getText();
				if (value.equalsIgnoreCase("href")) {
					String hrefValue = parser.nextTextValue();
					hrefs.add(hrefValue);
					findHrefInResponse(new URI(hrefValue));
					
				}
			}
		}

	}
}