package com.agilex.healthcare.mobilehealthplatform.ovid.util;

import java.util.Calendar;
import java.util.Date;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.agilex.healthcare.mobilehealthplatform.domain.DOBDate;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.medsphere.fileman.FMUtil;

public class OvidDateHelper {
	private static final Log logger = LogFactory.getLog(OvidDateHelper.class);

	/**
	 * Convert HL7 date/time/zone format to Fileman date.
	 */
	public static String hl7DateToFMDate(String hl7Date) {
		logger.debug("begin hl7DateToFMDate with hl7Date=" + hl7Date);
		if ("-1".contentEquals(hl7Date))
			return "";
		Date jDate = null;
		String noZoneDate = hl7Date;

		// remove timezone (if any)
		int zoneIdx = -1;
		if (hl7Date.indexOf("-") > 0)
			zoneIdx = hl7Date.indexOf("-");
		else if (hl7Date.indexOf("+") > 0)
			zoneIdx = hl7Date.indexOf("+");

		if (zoneIdx > 0)
			noZoneDate = hl7Date.substring(0, zoneIdx);

		jDate = DateHelper.parseHL7DateFormat(noZoneDate);

		logger.debug(String.format("hl7DateToFMDate %s->%s", hl7Date, jDate));
		String fmdate = FMUtil.dateToFMDate(jDate);
		logger.debug(String.format("hl7DateToFMDate %s->%s->%s", hl7Date, jDate, fmdate));
		return fmdate;
	}

	/**
	 * Converts Fileman date to a date string e.g. 3/4/1975 If the month or year
	 * was not set, those values are set to 0. Based off of FMUtil.
	 * 
	 * @param FMDate
	 * @return
	 */
	public static String fmDateToDobString(String FMDate) {
		try {
			Calendar cal = Calendar.getInstance();
			cal.set(Calendar.MILLISECOND, 0);
			int year = 0, month = 0, day = 0; // , hour = 0, minute = 0, second
												// = 0;

			String[] parts = FMDate.split("\\.");
			String dateString = parts[0];
			double dateNum = Double.parseDouble(dateString);
			if (dateNum < 1410102.0 || dateNum > 4141015.235959) {
				throw new RuntimeException("Date out of range. Must be between 1410102.0 and 4141015.235959");
			}
			year = 1700 + Integer.parseInt(dateString.substring(0, 3));
			month = Integer.parseInt(dateString.substring(3, 5));
			day = Integer.parseInt(dateString.substring(5, 7));

			return month + "/" + day + "/" + year;
		} catch (NumberFormatException ex) {
			throw new RuntimeException(ex.getMessage());
		}
	}

	public static DOBDate createDOBDate(String dob) {
		DOBDate dobDate = null;

		String[] months = { "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

		// Possible date formats: 1/1/1950, 1/0/1950, 0/0/1950
		try {
			if (isValidDate(dob)) {
				String[] dobParts = dob.split("/");
				int month = Integer.parseInt(dobParts[0]);
				int day = Integer.parseInt(dobParts[1]);
				String year = dobParts[2];

				if (month < 0 || month > 12 || day < 0 || day > 31) {
					throw new RuntimeException("Invalid date");
				}

				if (day == 0 && month == 0) {
					dobDate = new DOBDate("yyyy", year);
				} else if (day == 0 && month != 0) {
					dobDate = new DOBDate("MMM yyyy", months[month - 1] + " " + year);
				} else {
					dobDate = new DOBDate("MMM dd, yyyy", months[month - 1] + " " + day + ", " + year);
				}
			}
		} catch (Exception e) {
			logger.error("Failed to parse date: " + dob, e);
			dobDate = null;
		}
		return dobDate;
	}

	public static Date transformToJavaDate(String date) {
		String dateTimeFormat = "MM/dd/yyyy hh:mm";
		String dateTimeFormatRegEx = "##/##/#### ##:##";

		String dateFormat = "MM/dd/yyyy";
		String dateFormatRegEx = "##/##/####";

		Date javaDate = null;
		if (isFormat(date, dateTimeFormatRegEx))
			javaDate = DateHelper.parse(date, dateTimeFormat);
		else if (isFormat(date, dateFormatRegEx))
			javaDate = DateHelper.parse(date, dateFormat);

		if (javaDate == null)
			logger.warn("unable to determine format for date " + date);

		return javaDate;
	}

	// TODO: move this to date utils
	private static boolean isFormat(String dt, String format) {
		String numberFormat = "[0-9]";
		format = format.replace("#", numberFormat);
		Pattern pattern = Pattern.compile(format, Pattern.CASE_INSENSITIVE);
		return pattern.matcher(dt).matches();
	}

	private static boolean isValidDate(String dob) {
		return NullChecker.isNotNullish(dob) && (!dob.equalsIgnoreCase("*sensitive*"));
	}

}
