package com.agilex.healthcare.mobilehealthplatform.ovid;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;

import com.medsphere.fileman.FMUtil;
import com.medsphere.ovid.domain.ov.IsAVitalSign;
import com.medsphere.ovid.domain.ov.OvidDomainException;
import com.medsphere.ovid.domain.ov.OvidSecureRepository;
import com.medsphere.ovid.domain.ov.VitalSign;
import com.medsphere.vistarpc.RPCConnection;
import com.medsphere.vistarpc.RPCException;
import com.medsphere.vistarpc.RPCResponse;
import com.medsphere.vistarpc.RPCResponse.ResponseType;
import com.medsphere.vistarpc.VistaRPC;

public class VitalSignRepository extends OvidSecureRepository {

	private final static org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(VitalSignRepository.class);

	private final String CPRS_GUI_CONTEXT = "OR CPRS GUI CHART";
	private final String ORQQVI_VITALS = "ORQQVI VITALS";

	public VitalSignRepository(RPCConnection connection) {
		super(connection);
	}

	public VitalSignRepository(RPCConnection connection, RPCConnection serverConnection) {
		super(connection, serverConnection);
	}

	public Collection<IsAVitalSign> getMostRecentVitalSigns(String patientId) throws OvidDomainException {
		logger.info("retrieve vital signs from repo for patient id " + patientId);
		Collection<IsAVitalSign> vitalSignList = new ArrayList<IsAVitalSign>();

		RPCConnection rpcConnection;

		try {
			rpcConnection = getConnection();
			rpcConnection.setContext(CPRS_GUI_CONTEXT);
		} catch (RPCException e) {
			logger.error("Exception while setting the context", e);
			throw new OvidDomainException(e);
		}

		VistaRPC activeRpc = new VistaRPC(ORQQVI_VITALS, ResponseType.ARRAY);
		activeRpc.setParam(1, patientId);

		RPCResponse activeResponse;
		try {
			activeResponse = rpcConnection.execute(activeRpc);
		} catch (RPCException e) {
			logger.error("Exception while invoking rpc: " + ORQQVI_VITALS, e);
			throw new OvidDomainException(e);
		}

		String items[] = activeResponse.getArray();
		if (isEmptyResult(items)) {
			return vitalSignList;
		}

		for (String item : items) {
			VitalSign vitalSign = parseVitalSign(item);
			vitalSignList.add(vitalSign);
		}
		logger.info("completed with retrieve vital signs from repo for patient id " + patientId + ".  Returning "
				+ vitalSignList.size() + " record(s).");

		return vitalSignList;
	}	

	public Collection<IsAVitalSign> getVitalSigns(String patientId, Date beginDate, Date endDate) throws OvidDomainException {
		logger.info("retrieve vital signs from repo for patient id " + patientId);
		Collection<IsAVitalSign> vitalSignList = new ArrayList<IsAVitalSign>();

		RPCConnection rpcConnection;

		try {
			rpcConnection = getConnection();
			rpcConnection.setContext(CPRS_GUI_CONTEXT);
		} catch (RPCException e) {
			logger.error("Exception while setting the context", e);
			throw new OvidDomainException(e);
		}

		VistaRPC activeRpc = new VistaRPC(ORQQVI_VITALS, ResponseType.ARRAY);
		activeRpc.setParam(1, patientId);
		activeRpc.setParam(2, FMUtil.dateToFMDate(beginDate));
		activeRpc.setParam(3, FMUtil.dateToFMDate(endDate));
		
		RPCResponse activeResponse;
		try {
			activeResponse = rpcConnection.execute(activeRpc);
		} catch (RPCException e) {
			logger.error("Exception while invoking rpc: " + ORQQVI_VITALS, e);
			throw new OvidDomainException(e);
		}

		String items[] = activeResponse.getArray();
		if (isEmptyResult(items)) {
			return vitalSignList;
		}

		for (String item : items) {
			VitalSign vitalSign = parseVitalSign(item);
			vitalSignList.add(vitalSign);
		}
		logger.info("completed with retrieve vital signs from repo for patient id " + patientId + ".  Returning "
				+ vitalSignList.size() + " record(s).");

		return vitalSignList;
	}

	private VitalSign parseVitalSign(String item) throws OvidDomainException {
		logger.info("parsing vital sign: " + item);
		String parts[] = item.split("\\^");			
		//ien^vital type^rate without units^date/time taken^rate with units^metric rate^notes
		Date dateTaken;
		Date dateEntered = null;
		try {
			dateTaken = FMUtil.fmDateToDate(parts[3]);
		} catch (ParseException e) {
			logger.error("Cannot parse date " + parts[3]);
			throw new OvidDomainException(e);
		}
		String ien = parts[0];
	    String id = parts[1];
	    String name = parts[1];
	    String value;
	    String units = null;
	    if (parts.length < 5) {
	    	//vital sign does not have units, take value from 3rd field
	    	value = parts[2];
	    } else {
	    	//vital sign has units, take value from 5th field and parse the units.
		    value = parts[4];
			String valueParts[] = value.split(" ");
			value = valueParts[0];						
			if (valueParts.length > 1) {
				units = valueParts[valueParts.length-1];
			}
	    }	    
	    VitalSign vitalSign = new VitalSign(ien, id, name, value, dateTaken, dateEntered);	    
		if (units != null) {
			vitalSign.setUnits(units);			
		}
		System.err.println("origVitalSign="+vitalSign.toString());
		return vitalSign;
	}
}
