package com.agilex.healthcare.mobilehealthplatform.ovid;

import org.w3c.dom.Document;

import com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.connectionmanagement.ConnectionManagerFactory;
import com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.connectionmanagement.IOvidConnectionManager;
import com.agilex.healthcare.mobilehealthplatform.domain.MhpUser;
import com.agilex.healthcare.mobilehealthplatform.security.AppAuthenticator;
import com.agilex.healthcare.utility.XmlHelper;
import com.agilex.healthcare.utility.XpathHelper;
import com.medsphere.ovid.domain.ov.OvidDomainException;
import com.medsphere.vistarpc.RPCConnection;
import com.medsphere.vistarpc.RPCException;
import com.medsphere.vistarpc.RPCResponse;
import com.medsphere.vistarpc.RPCResponse.ResponseType;
import com.medsphere.vistarpc.VistaRPC;

public class VistaAuthenticator implements AppAuthenticator {

	private final static org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(VistaAuthenticator.class);

	private final String MSC_PATIENT_DASHBOARD = "MSC PATIENT DASHBOARD";
	private final String RPC_XHD_AUTHORIZE = "XHD AUTHORIZE";

	/**
	 * Calls RPC "XHD AUTHORIZE" and perform user authentication.
	 * 
	 * @param accessCode
	 * @param verifyCode
	 * @return
	 * @throws OvidDomainException
	 */
	@Override
	public MhpUser authenticate(String accessCode, String verifyCode, String vistaLocation) throws OvidDomainException {
		IOvidConnectionManager connManager = null;
		RPCConnection serverConnection = null;

		try {
			logger.debug("Vista authenticator getting connection manager instance");
			connManager = ConnectionManagerFactory.getInstance();

			logger.debug("Vista authenticator getting server connection instance");
			serverConnection = connManager.getServerConnection();
			logger.debug("Vista authenticator setting connection context");
			serverConnection.setContext(MSC_PATIENT_DASHBOARD);

			logger.debug("Vista authenticator initializing RPC ");
			VistaRPC rpc = new VistaRPC(RPC_XHD_AUTHORIZE, ResponseType.GLOBAL_ARRAY);
			rpc.setParam(1, accessCode);
			rpc.setParam(2, verifyCode);

			logger.debug("Vista authenticator executing RPC");
			RPCResponse response = serverConnection.execute(rpc);
			logger.debug("Vista authenticator executed RPC");

			if (response.getError() == null || response.getError().isEmpty()) {
				String[] items = response.getArray();
				if (items.length == 0) {
					logger.info("No data!!");
					throw new OvidDomainException("No response for RPC:XHD AUTHORIZE");
				} else {
					StringBuffer userInfo = new StringBuffer();
					for (int idx = 0; idx < items.length; ++idx) {
						userInfo.append(items[idx]);
					}

					Document userInfoAsDocument = XmlHelper.loadXml(userInfo.toString());

					// check response for error message
					String errorText = checkResponseForError(userInfoAsDocument);

					if (errorText != null && !errorText.isEmpty()) {
						throw new OvidDomainException(errorText);
					}

					return getUserFromResponse(userInfoAsDocument);
				}
			} else {
				throw new OvidDomainException(response.getError());
			}
		} catch (RPCException e1) {
			throw new OvidDomainException(e1);
		} catch (RuntimeException e2) {
			throw new OvidDomainException(e2);
		} finally {
			closeConnection(serverConnection);
		}
	}

	// TODO: combine with Connection Manager code
	private void closeConnection(RPCConnection connection) {
		ConnectionManagerFactory.getInstance().releaseConnection(connection);
	}

	/**
	 * Parse response for error.
	 * 
	 * Example: <result xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
	 * xsi:noNamespaceSchemaLocation="AuthenticateSchema.xsd">
	 * <errorText>Invalid Access/Verify Code Pair</errorText> </result>
	 * 
	 * @param source
	 *            response in XML format
	 * @return error message
	 */
	private String checkResponseForError(Document source) {
		String expression = "/result/errorText";
		return XpathHelper.getString(source, expression);
	}

	/**
	 * Parse response for user info.
	 * 
	 * Example: <result xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
	 * xsi:noNamespaceSchemaLocation="AuthenticateSchema.xsd"> <userInfo>
	 * <uniqueId>148</uniqueId> <firstName>KMR</firstName>
	 * <lastName>USER</lastName> <middleInitial/> <title/> <lastFourSSN/>
	 * </userInfo> </result>
	 * 
	 * @param source
	 *            response in XML format
	 * @return user info
	 */
	private MhpUser getUserFromResponse(Document source) {
		MhpUser user = new MhpUser();

		String expression = "/result/userInfo/uniqueId";
		user.setId(XpathHelper.getString(source, expression));

		expression = "/result/userInfo/firstName";
		user.setFirstName(XpathHelper.getString(source, expression));

		expression = "/result/userInfo/lastName";
		user.setLastName(XpathHelper.getString(source, expression));

		expression = "/result/userInfo/middleInitial";
		user.setMiddleInitials(XpathHelper.getString(source, expression));

		return user;
	}
}
