package com.agilex.healthcare.mobilehealthplatform.ovid;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import com.agilex.healthcare.mobilehealthplatform.ovid.domain.OvidTopographyCache;
import com.agilex.healthcare.mobilehealthplatform.ovid.fmdomain.FMTopographyField;
import com.medsphere.common.util.TimeKeeperDelegate;
import com.medsphere.fileman.FMQueryByIENS;
import com.medsphere.fileman.FMQueryList;
import com.medsphere.fileman.FMResultSet;
import com.medsphere.fileman.FMScreen;
import com.medsphere.fileman.FMScreenEquals;
import com.medsphere.fileman.FMScreenField;
import com.medsphere.fileman.FMScreenValue;
import com.medsphere.fileman.FMUtil;
import com.medsphere.ovid.domain.ov.OvidDomainException;
import com.medsphere.resource.ResException;
import com.medsphere.rpcresadapter.RPCResAdapter;
import com.medsphere.vistarpc.RPCConnection;
import com.medsphere.vistarpc.RPCException;

public class TopographyFieldRepository extends OvidConnectionAwareRepository {
	
	private final static org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(TopographyFieldRepository.class);

	// List of Fileman globals
	private final String TOPOGRAPHY_FIELD_FILE = "61";
	
	public TopographyFieldRepository(RPCConnection connection) {
		super(connection);
	}

	public Map<String, FMTopographyField> getAllTopographies() throws OvidDomainException {
		Map<String, FMTopographyField> topographyMap = 
			Collections.synchronizedMap(new HashMap<String, FMTopographyField>());
		
		RPCConnection rpcConnection;
		RPCResAdapter adapter = null;
		
		try {
			// setup connection
			rpcConnection = getConnection();
			rpcConnection.setContext(FMUtil.FM_RPC_CONTEXT);
    		adapter = new RPCResAdapter(rpcConnection, FMUtil.FM_RPC_NAME);
    		
    		TimeKeeperDelegate.start("Getting Topography");
    		
			FMQueryList query = new FMQueryList(adapter, FMTopographyField.getFileInfoForClass());   
				
            FMResultSet results = query.execute();
            if (results != null && (results.getError() == null || results.getError().isEmpty())) {
            	FMTopographyField topography = null;
                while (results.next()) {
                	topography = new FMTopographyField(results);
                	topographyMap.put(topography.getIEN(), topography);
                }
            }
            else {
            	logger.error(results.getError());
				throw new OvidDomainException(results.getError());
            }
            
            return topographyMap;
        } catch (ResException e1) {
        	logger.error("Exception querying " + TOPOGRAPHY_FIELD_FILE, e1);
			throw new OvidDomainException(e1);
		} catch (RPCException e2) {
			logger.error("Exception querying " + TOPOGRAPHY_FIELD_FILE, e2);
			throw new OvidDomainException(e2);
		} finally {
            TimeKeeperDelegate.stopAndLog("Getting Topography");
        }		
	}	
	
	public Map<String, FMTopographyField> getTopographiesByIEN(Collection<String> iens) throws OvidDomainException {
		Map<String, FMTopographyField> topographyMap = 
			Collections.synchronizedMap(new HashMap<String, FMTopographyField>());
		
		RPCConnection rpcConnection;
		RPCResAdapter adapter = null;
		
		try {
			// setup connection
			rpcConnection = getConnection();
			rpcConnection.setContext(FMUtil.FM_RPC_CONTEXT);
    		adapter = new RPCResAdapter(rpcConnection, FMUtil.FM_RPC_NAME);
    		
    		TimeKeeperDelegate.start("Getting Topography");    		

            FMQueryByIENS query = new FMQueryByIENS(adapter, FMTopographyField.getFileInfoForClass());    		
            query.setIENS(iens);

            FMResultSet results = query.execute();
            if (results != null && (results.getError() == null || results.getError().isEmpty())) {
            	FMTopographyField fmrecord = null;
                while (results.next()) {
                	fmrecord = new FMTopographyField(results);                	
            		topographyMap.put(fmrecord.getIEN(), fmrecord);
                }
            }            
            else {
            	logger.error(results.getError());
				throw new OvidDomainException(results.getError());
            }
            
            return topographyMap;
        } catch (ResException e1) {
        	logger.error("Exception querying " + TOPOGRAPHY_FIELD_FILE, e1);
			throw new OvidDomainException(e1);
		} catch (RPCException e2) {
			logger.error("Exception querying " + TOPOGRAPHY_FIELD_FILE, e2);
			throw new OvidDomainException(e2);
		} finally {
            TimeKeeperDelegate.stopAndLog("Getting Topography");
        }		
	}	
	
    /**
     * Get ward by name
     * @param name
     * @return
     * @throws OvidDomainException
     */
    public void addTopographyToCache(String name) throws OvidDomainException {
    	     	
    	FMTopographyField fmrecord = null;    	
    	RPCConnection rpcConnection;
		RPCResAdapter adapter = null;
		
		if (name == null || name.isEmpty()) {
			logger.error("Exception querying " + TOPOGRAPHY_FIELD_FILE + ": name can not be null or empty");
		}
		
        try {
        	// setup connection
			rpcConnection = getConnection();
			rpcConnection.setContext(FMUtil.FM_RPC_CONTEXT);
    		adapter = new RPCResAdapter(rpcConnection, FMUtil.FM_RPC_NAME);
    		
            FMQueryList query = new FMQueryList(adapter, FMTopographyField.getFileInfoForClass());
            FMScreen byName = new FMScreenEquals(new FMScreenField(".01"), new FMScreenValue(name.trim()));
            query.setScreen(byName);
            FMResultSet result = query.execute();
            if (result != null) {
                if (result.getError() != null) {
                    throw new OvidDomainException(result.getError());
                }
                if (result.next()) {
                	fmrecord = new FMTopographyField(result);
                	OvidTopographyCache.getInstance().addToCache(fmrecord.getName(), fmrecord);
                	logger.debug(String.format("adding topography item to cache [key=%s][value=%s]" , fmrecord.getName() , fmrecord ) );
                }
            }        	
        } catch (ResException e1) {
        	logger.error("Exception querying " + TOPOGRAPHY_FIELD_FILE, e1);
			throw new OvidDomainException(e1);
		} catch (RPCException e2) {
			logger.error("Exception querying " + TOPOGRAPHY_FIELD_FILE, e2);
			throw new OvidDomainException(e2);
		} finally {
            TimeKeeperDelegate.stopAndLog("Getting Topography");
        }		
    }

}
