package com.agilex.healthcare.mobilehealthplatform.ovid;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;

import com.agilex.healthcare.mobilehealthplatform.MHPConstants;
import com.agilex.healthcare.mobilehealthplatform.ovid.domain.OvidPatient;
import com.medsphere.fileman.FMQueryByIENS;
import com.medsphere.fileman.FMResultSet;
import com.medsphere.fmdomain.FMPatient;
import com.medsphere.fmdomain.FMPatientMovement;
import com.medsphere.ovid.domain.ov.OvidDomainException;
import com.medsphere.ovid.domain.ov.OvidSecureRepository;
import com.medsphere.ovid.domain.ov.PatientMovementRepository;
import com.medsphere.resource.ResAdapter;
import com.medsphere.resource.ResException;
import com.medsphere.vistarpc.RPCArray;
import com.medsphere.vistarpc.RPCConnection;
import com.medsphere.vistarpc.RPCException;
import com.medsphere.vistarpc.RPCResponse;
import com.medsphere.vistarpc.RPCResponse.ResponseType;
import com.medsphere.vistarpc.VistaRPC;

public class OvidPatientRepository extends OvidSecureRepository {

	private final static org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(OvidPatientRepository.class);
	
	private final static String RPC_CONTEXT = "OR CPRS GUI CHART";
	private final static String RPC_ORWPT_ID_INFO = "ORWPT ID INFO";
	protected RPCConnection serverConnection;
	protected PatientMovementRepository repo; 	
	public OvidPatientRepository(RPCConnection connection) {
		super(connection);
		this.serverConnection = connection;
		repo = new PatientMovementRepository(serverConnection);
	}

	public OvidPatientRepository(RPCConnection connection, RPCConnection serverConnection) {
		super(connection, serverConnection);
		this.serverConnection = serverConnection;
		repo = new PatientMovementRepository(serverConnection);
	}

	/**
	 * lookup by name (partial names are ok (e.g. LASTNAME). Uses SC PATIENT
	 * LOOKUP RPC
	 * 
	 * @param name
	 * @return
	 * @throws OvidDomainException
	 */
	public Collection<OvidPatient> lookupPatientsByName(String name) throws OvidDomainException {
		Collection<String> nameList = new ArrayList<String>();
		nameList.add(name);
		return patientLookup(nameList);
	}

	/**
	 * Returns information about a patient using RPC "ORWPT ID INFO".
	 * @param ien
	 * @return
	 * @throws OvidDomainException
	 */
	public String getDateOfBirth(String ien) throws OvidDomainException {
		String dob = null;
		
		RPCConnection rpcConnection;
		try {
			rpcConnection = super.getConnection();
			if (rpcConnection == null) {
				super.getServerConnection();
			}
			rpcConnection.setContext(RPC_CONTEXT);
		} catch (RPCException ex) {
			throw new OvidDomainException(ex);
		}
	
		VistaRPC rpc = new VistaRPC(RPC_ORWPT_ID_INFO, ResponseType.SINGLE_VALUE);
		rpc.setParam(1, ien);
		
		try {
			RPCResponse response = rpcConnection.execute(rpc);
			String data = response.getString();
			if (data == null || data.isEmpty()) {
				logger.debug("empty data");
			}
			else {			
				logger.debug("data=" + data);
				String parts[] = data.split("\\^", -1);
				dob = parts[1];
			}
		} catch (RPCException ex) {
			logger.error("Exception while executing RPC \"" + RPC_ORWPT_ID_INFO + "\"", ex);
			throw new OvidDomainException(ex);
		}
		
		return dob;
	}
	
	protected Collection<OvidPatient> patientLookup(Collection<String> values) throws OvidDomainException {
		Collection<FMPatient> fmPatients = new ArrayList<FMPatient>();
		Collection<OvidPatient> ovidPatients = new ArrayList<OvidPatient>();
		HashMap<String, String> iensDobMap = new HashMap<String, String>();

		logger.debug("looking up patients by name: " + values);
		RPCConnection rpcConnection;

		try {
			rpcConnection = getConnection();
			rpcConnection.setContext(RPC_CONTEXT);
		} catch (RPCException ex) {
			throw new OvidDomainException(ex);
		}

		VistaRPC rpc = new VistaRPC("SC PATIENT LOOKUP", ResponseType.ARRAY);
		RPCArray valArray = new RPCArray();
		for (String value : values) {
			valArray.put("\"VALUE\"", value);
		}

		rpc.setParam(1, valArray);
		try {
			RPCResponse response = rpcConnection.execute(rpc);
			String items[] = response.getArray();
			if (isEmptyResult(items)) {
				logger.debug("empty list");
				return ovidPatients;
			}
			Collection<String> iens = new ArrayList<String>();
			for (String item : items) {
				String parts[] = item.split("\\^", -1);
				if (parts.length > 0) {
					String id = parts[0];
					if (id.matches("\\d+")) {
						iens.add(id);
						String dobString = parts[2];
						iensDobMap.put(id, dobString);
					}
				}
			}
			fmPatients = getPatientsForIENS(iens);
			for (FMPatient fmPatient : fmPatients) {
				String dob = iensDobMap.get(fmPatient.getIEN());
				OvidPatient ovidPatient = new OvidPatient(fmPatient);
				ovidPatient.setDob(dob);
				String status = getPatientStatus(fmPatient.getIEN());
				ovidPatient.setStatus(status);
				ovidPatients.add(ovidPatient);
			}
		} catch (RPCException ex) {
			logger.error("RPC exception", ex);
		}

		return ovidPatients;
	}

	/**
	 * Get a list of patients for a list of iens
	 * 
	 * @param iens
	 * @return
	 * @throws com.medsphere.ovid.connection.OvidDomainException
	 */
	protected Collection<FMPatient> getPatientsForIENS(Collection<String> iens) throws OvidDomainException {
		return getPatientsForIENS(iens, false);
	}

	/**
	 * Get a list of patients for a list of iens
	 * 
	 * @param iens
	 * @return
	 * @throws com.medsphere.ovid.connection.OvidDomainException
	 */
	protected Collection<FMPatient> getPatientsForIENS(Collection<String> iens, Boolean getInternal)
			throws OvidDomainException {
		Collection<FMPatient> list = new ArrayList<FMPatient>();
		try {
			ResAdapter adapter = obtainServerRPCAdapter();

			FMQueryByIENS query = new FMQueryByIENS(adapter, FMPatient.getFileInfoForClass());
			query.setIENS(iens);
			query.getField("ATTENDING PHYSICIAN").setInternal(getInternal);
			query.getField("CURRENT ROOM").setInternal(getInternal);
			query.getField("CURRENT MOVEMENT").setInternal(getInternal);
			query.getField("PROVIDER").setInternal(getInternal);
			query.getField("WARD LOCATION");
			query.getField("ROOM-BED");

			FMResultSet results = query.execute();
			if (results != null) {
				if (results.getError() != null) {
					throw new OvidDomainException(results.getError());
				}
				while (results.next()) {
					list.add(new FMPatient(results));
				}
			}
		} catch (ResException e) {
			throw new OvidDomainException(e);
		}

		return list;
	}

	protected String getPatientStatus(String dfn) throws OvidDomainException {
		Collection<FMPatientMovement> list = new ArrayList<FMPatientMovement>();
		list = repo.getPatientMovementByPatientDFN(dfn);
		if (list.size() == 0) {
			return MHPConstants.OUTPATIENT;
		}
				
		FMPatientMovement patientMovement = list.iterator().next();
		if (list.size() > 1) {
			//find the latest movement
			for (FMPatientMovement tempMovement : list) {
				if (patientMovement.getDateTime().getTime() < tempMovement.getDateTime().getTime()) {
					patientMovement = tempMovement;
				}
			}
		}
		if (patientMovement.getTransaction().equals("ADMISSION")) {	// ADMISSION = 1
			return MHPConstants.INPATIENT; //active
		} else {
			return MHPConstants.OUTPATIENT; //inactive
		}
	}
}
