package com.agilex.healthcare.mobilehealthplatform.ovid;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;

import com.agilex.healthcare.mobilehealthplatform.ovid.domain.OvidMedication;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.ListHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.medsphere.ovid.domain.ov.OvidDomainException;
import com.medsphere.ovid.domain.ov.OvidSecureRepository;
import com.medsphere.vistarpc.RPCArray;
import com.medsphere.vistarpc.RPCConnection;
import com.medsphere.vistarpc.RPCException;
import com.medsphere.vistarpc.RPCResponse;
import com.medsphere.vistarpc.RPCResponse.ResponseType;
import com.medsphere.vistarpc.VistaRPC;

public class MedicationRepository extends OvidSecureRepository {
	private final static org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory
			.getLog(MedicationRepository.class);

	private String context;
	private static final String ORWPS_ACTIVE = "ORWPS ACTIVE";
	private static final String ORWPS_DETAIL = "ORWPS DETAIL";
	private static final String ORWPS_MEDHIST = "ORWPS MEDHIST";
	private static String ORRC_ORDERS_BY_ID = "ORRC ORDERS BY ID";
	private final String MSC_PATIENT_DASHBOARD = "MSC PATIENT DASHBOARD";
	
	public MedicationRepository(RPCConnection connection, String context) {
		super(connection);
		this.context = context;
	}

	/**
	 * Get a single ovid medication with the medicationDetail that combines the
	 * medication detail and medication history
	 * 
	 * @param patientId
	 * @param orderId
	 * @return
	 * @throws OvidDomainException
	 */
	public OvidMedication getOvidMedicationDetail(String patientId, String orderId) throws OvidDomainException {
		// reuse a single connection for all 3 rpc calls
		RPCConnection rpcConnection = null;
		rpcConnection = getConnection();
		try {
			rpcConnection.setContext(context);
		} catch (RPCException e) {
			logger.error("Exception while setting the context", e);
			throw new OvidDomainException(e);
		}

		Collection<OvidMedication> medications = getAllMedications(patientId, rpcConnection);
		for (OvidMedication ovidMedication : medications) {
			if (ovidMedication.getOrderNumber().equals(orderId)) {
				// Medication detail is obtained by concatenating medication
				// detail and history
				String medicationDetail = getMedicationDetail(patientId, ovidMedication.getPrescriptionId(),
						rpcConnection);
				String medicationHistory = getMedicationHistory(patientId, orderId, rpcConnection);
				String combinedDetail = medicationDetail + "\n\n" + medicationHistory;
				ovidMedication.setMedicationDetail(combinedDetail);
				return ovidMedication;
			}
		}
		return null;
	}

	/**
	 * Get all medications for a patient without the medicationDetail
	 * 
	 * @param patientId
	 * @return
	 * @throws OvidDomainException
	 */
	public Collection<OvidMedication> getAllMedications(String patientId) throws OvidDomainException {
		RPCConnection rpcConnection = null;
		rpcConnection = getConnection();
		try {
			rpcConnection.setContext(context);
		} catch (RPCException e) {
			logger.error("Exception while setting the context", e);
			throw new OvidDomainException(e);
		}
		return getAllMedications(patientId, rpcConnection);
	}

	/**
	 * Get only the medication history
	 * 
	 * @param patientId
	 * @param orderId
	 * @return
	 * @throws OvidDomainException
	 */
	public String getMedicationHistory(String patientId, String orderId) throws OvidDomainException {
		RPCConnection rpcConnection = null;
		rpcConnection = getConnection();
		try {
			rpcConnection.setContext(context);
		} catch (RPCException e) {
			logger.error("Exception while setting the context", e);
			throw new OvidDomainException(e);
		}
		return getMedicationHistory(patientId, orderId, rpcConnection);
	}

	/**
	 * Get only the medication detail without the medication history
	 * 
	 * @param patientId
	 * @param prescriptionId
	 * @return
	 * @throws OvidDomainException
	 */
	public String getMedicationDetail(String patientId, String prescriptionId) throws OvidDomainException {
		RPCConnection rpcConnection = null;
		rpcConnection = getConnection();
		try {
			rpcConnection.setContext(context);
		} catch (RPCException e) {
			logger.error("Exception while setting the context", e);
			throw new OvidDomainException(e);
		}
		return getMedicationDetail(patientId, prescriptionId, rpcConnection);
	}

	private Collection<OvidMedication> getAllMedications(String patientId, RPCConnection rpcConnection)
			throws OvidDomainException {
		final int MEDICATIONS_RESULTS_ORDERID_INDEX = 8;
		final int PRESCRIPTION_ID_INDEX = 1;
		final int MEDICATIONS_RESULTS_STATUS_INDEX = 9;
		final int ENDDATE_INDEX = 4;
		final int ORDERDATE_INDEX = 15;
		final int MEDICATIONNAME_INDEX = 2;
		final int LASTFILLEDDATE_INDEX = 10;
		final int REFILLS_INDEX = 5;
		final int SUPPLY_INDEX = 11;
		final int QUANTITY_INDEX = 12;

		logger.info("retrieve medications from repo for patient id " + patientId);

		Collection<OvidMedication> medicationList = new ArrayList<OvidMedication>();

		VistaRPC activeRpc = new VistaRPC(ORWPS_ACTIVE, ResponseType.ARRAY);
		activeRpc.setParam(1, patientId);

		RPCResponse activeResponse;
		try {
			activeResponse = rpcConnection.execute(activeRpc);
		} catch (RPCException e) {
			logger.error("Exception while invoking rpc: " + ORWPS_ACTIVE, e);
			throw new OvidDomainException(e);
		}

		String items[] = activeResponse.getArray();
		if (isEmptyResult(items)) {
			return medicationList;
		}

		// ~IV^9V;I^ceFAZolin INJ^INFUSE OVER 30 MIN.^3100104.24^^1
		// GM^^599^EXPIRED^^^^^0^3091230.0935^
		// Line 1: ~<type of med, inpatient/outpatient/non-va>^<prescription
		// id>^med
		// name>^^<expiration date>^<# of refills
		// remaining>^^^<??>^<status>^<last filled date>^<days
		// supply>^<quantity>^^^<??>^<???>^<??>
		// Line 2: <free text description>
		// Line 3+: \ <sig>
		int medicationIndex = 0;
		int lineNumber = 0;
		for (String item : items) {
			// logger.debug(lineNumber + ": " + item);
			lineNumber++;
			if (item.startsWith("~")) {
				medicationIndex++;
				String parts[] = item.split("\\^");
				String orderID = ListHelper.getValue(parts, MEDICATIONS_RESULTS_ORDERID_INDEX);
				String prescriptionId = ListHelper.getValue(parts, PRESCRIPTION_ID_INDEX);
				String endDateText = ListHelper.getValue(parts, ENDDATE_INDEX);
				String lastFilledDateText = ListHelper.getValue(parts, LASTFILLEDDATE_INDEX);
				String orderDateText = ListHelper.getValue(parts, ORDERDATE_INDEX);
				String medName = ListHelper.getValue(parts, MEDICATIONNAME_INDEX);
				String refills = ListHelper.getValue(parts, REFILLS_INDEX);
				String supply = ListHelper.getValue(parts, SUPPLY_INDEX);
				String quantity = ListHelper.getValue(parts, QUANTITY_INDEX);
				String sourceCode = ListHelper.getValue(parts, 0);
				String source = calculateMedicationSource(sourceCode);
				String status = ListHelper.getValue(parts, MEDICATIONS_RESULTS_STATUS_INDEX);
				status = status.toUpperCase();
				if (status.startsWith("DISCONTINUED")) {
					//remove /edit or (edit) after discontinued
					status = "DISCONTINUED";
				}

				
				// dates for external (home meds) is not reliable
				if ("EXTERNAL".contentEquals(source)) {
					orderDateText = null;
					endDateText = null;
				}

				logger.debug(String.format("%s: m=%s; sd=%s; ed=%s; lfd=%s s=%s [%s]", medicationIndex, medName,
						orderDateText, endDateText, lastFilledDateText, source, item));

				OvidMedication ovidMedication = new OvidMedication();
				ovidMedication.setMedication(medName);
				ovidMedication.setOrderNumber(orderID);
				ovidMedication.setMedicationId("unknown");
				ovidMedication.setPrescriptionId(prescriptionId);
				ovidMedication.setStartDate(parseDateFromMedicationFileFormat(orderDateText));
				ovidMedication.setStopDate(parseDateFromMedicationFileFormat(endDateText));
				ovidMedication.setStatus(status);
				ovidMedication.setMedType(source);
				ovidMedication.setRefills(refills);
				ovidMedication.setSupply(supply);
				ovidMedication.setQuantity(quantity);
				ovidMedication.setLastFilled(parseDateFromMedicationFileFormat(lastFilledDateText));

				StringBuilder sig = null;
				boolean stopSigParsing = false;
				int readAheadLineNumber = lineNumber + 1;
				// skip any lines that start don't start with /
				for (; readAheadLineNumber < items.length; readAheadLineNumber++) {
					String line = items[readAheadLineNumber];
					if (line.startsWith("~")) {
						stopSigParsing = true;
						break;
					}
					if (line.startsWith("\\")) {
						break;
					}
				}
				if (!stopSigParsing) {
					// read first line of sig, don't add a new line at the
					// beginning
					sig = new StringBuilder();
					sig.append(items[readAheadLineNumber].substring(1));
					readAheadLineNumber++;
					int numberOfTabLines = 0;
					// read rest of the lines
					for (; readAheadLineNumber < items.length; readAheadLineNumber++) {
						String line = items[readAheadLineNumber];
						if (line.startsWith("~")) {
							break;
						}
						if (line.startsWith("\\")) {
							// write on a new line
							sig.append("  ").append(line.substring(1));
						} else if (line.startsWith("t")) {
							numberOfTabLines++;
							// show up to two tab lines, ignore the rest
							if (numberOfTabLines > 2) {
								break;
							}
							// ignore lines that contain HOME
							if (line.trim().contains("HOME")) {
								continue;
							}
							// add spaces then append
							sig.append("  ").append(line.substring(1));
						} else {
							// append
							sig.append(line);
						}
					}
				}

				if (sig != null) {
					logger.debug("sig=" + sig.toString());
					ovidMedication.setSigDetail(sig.toString());
				}
				medicationList.add(ovidMedication);
			}
		}

		logger.info("complete with retrieve medications from repo for patient id " + patientId + ".  Returning "
				+ medicationList.size() + " record(s).");
		return medicationList;
	}

	/**
	 * 
	 * @param patientId
	 * @param orderId
	 * @return
	 * @throws OvidDomainException
	 */
	public OvidMedication getMedicationLite(String orderId) throws OvidDomainException {

		HashMap<String, String> prescriptionIdMap = new HashMap<String, String>();
		RPCConnection rpcConnection;

		try {
			rpcConnection = getConnection();
			rpcConnection.setContext(MSC_PATIENT_DASHBOARD);
		} catch (RPCException e) {
			logger.error("Exception while setting the context", e);
			throw new OvidDomainException(e);
		}
		RPCArray orders = new RPCArray();
		orders.put(Integer.toString(0), "OR1:" + orderId);
		VistaRPC ordersRpc = new VistaRPC(ORRC_ORDERS_BY_ID, ResponseType.ARRAY);
		ordersRpc.setParam(1, orders);
		RPCResponse orderResponse;
		try {
			orderResponse = rpcConnection.execute(ordersRpc);
		} catch (RPCException e) {
			logger.error("Exception while invoking rpc: " + ORRC_ORDERS_BY_ID, e);
			throw new OvidDomainException(e);
		} finally {
			try {
				rpcConnection.setContext(context);
			} catch (RPCException e) {
				logger.error("Exception while setting the context", e);
				throw new OvidDomainException(e);
			}
			
		}
		if (orderResponse.getError() != null && !orderResponse.getError().isEmpty()) {
			logger.error(orderResponse.getError());
			throw new OvidDomainException(orderResponse.getError());
		}						
				
		String orderItems[] = orderResponse.getArray();

		int i = 0;
		for (String orderItem : orderItems) {
			i++;
			logger.debug(orderItem);

			if (orderItem.startsWith("Item")) {
				String parts[] = orderItem.split("\\^", -1);
				if (parts[1].length() == 0) {
					return null;
				}
				String idParts[] = parts[0].split(":", -1);
				String orderID = idParts[1];
				String medName = parts[1].replace("...","");
				String datePart = parts[2];
				while (datePart.length() < 14) {
					datePart += "0";
				}
				Date date = null;

				try {
					date = parseMedTimeFromOrderFormat(datePart);
				} catch (ParseException ex) {
					date = null;
				}
				String status = parts[3].toUpperCase();
				if (status.startsWith("DISCONTINUED")) {
					//remove /edit or (edit) after discontinued
					status = "DISCONTINUED";
				}

				logger.debug(String.format("%s: m=%s; sd=%s; ed=%s; lfd=%s [%s]", i, medName, datePart,
						"?", "?", orderItem));

				OvidMedication ovidMedication = new OvidMedication();
				ovidMedication.setMedication(medName);
				ovidMedication.setOrderNumber(orderID);
				ovidMedication.setMedicationId(orderID);
				ovidMedication.setStartDate(date);
				ovidMedication.setStatus(status);
				String prescriptionId = prescriptionIdMap.get(orderID);
				ovidMedication.setPrescriptionId(prescriptionId);
				return ovidMedication;
			}
		}
		return null;
	}

	private String getMedicationHistory(String patientId, String orderId, RPCConnection rpcConnection)
			throws OvidDomainException {
		logger.info("get medication history for patient id " + patientId + " order id " + orderId);

		VistaRPC rpc = new VistaRPC(ORWPS_MEDHIST, ResponseType.GLOBAL_ARRAY);

		rpc.setParam(1, patientId);
		rpc.setParam(2, orderId);
		RPCResponse response = null;
		try {
			response = rpcConnection.execute(rpc);
		} catch (RPCException e) {
			logger.error("Exception while executing RPC \"" + ORWPS_MEDHIST + "\"", e);
			throw new OvidDomainException(e);
		}

		if (response.getError() == null || response.getError().isEmpty()) {
			StringBuilder builder = new StringBuilder();
			String items[] = response.getArray();
			if (isEmptyResult(items) || items.length < 4) {
				logger.info("Empty medication history");
				return null;
			} else {
				for (String item : items) {
					builder.append(item).append("\n");
				}
			}
			return builder.toString();
		} else {
			logger.error(response.getError());
			throw new OvidDomainException(response.getError());
		}
	}

	private String getMedicationDetail(String patientId, String prescriptionId, RPCConnection rpcConnection)
			throws OvidDomainException {
		logger.info("get medication detail for patient id " + patientId + " prescription id " + prescriptionId);

		VistaRPC rpc = new VistaRPC(ORWPS_DETAIL, ResponseType.GLOBAL_ARRAY);

		rpc.setParam(1, patientId);
		rpc.setParam(2, prescriptionId);
		RPCResponse response = null;
		try {
			response = rpcConnection.execute(rpc);
		} catch (RPCException e) {
			logger.error("Exception while executing RPC \"" + ORWPS_DETAIL + "\"", e);
			throw new OvidDomainException(e);
		}

		if (response.getError() == null || response.getError().isEmpty()) {
			StringBuilder builder = new StringBuilder();
			String items[] = response.getArray();
			if (isEmptyResult(items)) {
				logger.info("Empty medication detail");
				return null;
			} else if (items.length <= 10 || items[10].equals("Order #0")) {
				logger.info("Empty medication detail");
				return null;
			} else {
				for (String item : items) {
					builder.append(item).append("\n");
				}
			}
			return builder.toString();
		} else {
			logger.error(response.getError());
			throw new OvidDomainException(response.getError());
		}
	}

	private String normalizeDate(String dateString) {
		if (dateString.length() >= 14) {
			return dateString;
		}
		StringBuilder sb = new StringBuilder(dateString);
		for (int i = 14 - dateString.length(); i > 0; --i) {
			sb.append("0");
		}
		return sb.toString();
	}

	private Date parseMedTimeFromOrderFormat(String dateString) throws ParseException {
		return new SimpleDateFormat("yyyyMMddHHmmss").parse(normalizeDate(dateString));
	}

	private Date parseDateFromMedicationFileFormat(String dateString) {
		return DateHelper.parseVistaDate(dateString);
	}

	private String calculateMedicationSource(String sourceCode) {
		String source = null;

		if (NullChecker.isNullish(sourceCode))
			source = null;
		else {
			sourceCode = sourceCode.replace("~", "");
			if (sourceCode.contentEquals("IV"))
				source = "INPATIENT";
			else if (sourceCode.contentEquals("UD"))
				source = "INPATIENT";
			else if (sourceCode.contentEquals("OP"))
				source = "OUTPATIENT";
			else if (sourceCode.contentEquals("NV"))
				source = "EXTERNAL";
			else
				source = "UNKNOWN";
		}
		return source;
	}
}
