package com.agilex.healthcare.mobilehealthplatform.ovid;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import com.agilex.healthcare.mobilehealthplatform.ovid.domain.OvidLabTestCache;
import com.agilex.healthcare.mobilehealthplatform.ovid.domain.OvidTopographyCache;
import com.agilex.healthcare.mobilehealthplatform.ovid.fmdomain.FMLabReport;
import com.agilex.healthcare.mobilehealthplatform.ovid.fmdomain.FMLabReportHeader;
import com.agilex.healthcare.mobilehealthplatform.ovid.fmdomain.FMLabReportLabTest;
import com.agilex.healthcare.mobilehealthplatform.ovid.fmdomain.FMLabReportTestGroup;
import com.agilex.healthcare.mobilehealthplatform.ovid.fmdomain.FMTopographyField;
import com.medsphere.common.cache.GenericCacheException;
import com.medsphere.fileman.FMFile;
import com.medsphere.fileman.FMQueryList;
import com.medsphere.fileman.FMRecord;
import com.medsphere.fileman.FMResultSet;
import com.medsphere.fileman.FMUtil;
import com.medsphere.fmdomain.FMLaboratoryTest;
import com.medsphere.ovid.domain.ov.OvidDomainException;
import com.medsphere.ovid.domain.ov.OvidSecureRepository;
import com.medsphere.resource.ResException;
import com.medsphere.rpcresadapter.RPCResAdapter;
import com.medsphere.vistarpc.RPCConnection;
import com.medsphere.vistarpc.RPCException;

public class LabTestRepository extends OvidSecureRepository {

	private final static org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(LabTestRepository.class);

	private final static String LABORATORY_TEST_FILE = "60";
	private final static String CUMMULATIVE_REPORTS = "CUMULATIVE REPORTS";

	public LabTestRepository(RPCConnection connection) {
		super(connection);

		try {
			if (OvidLabTestCache.getInstance().getCacheValues().isEmpty()) {
				setLabTestsCache();
			}
		} catch (GenericCacheException e) {
			throw new RuntimeException(e);
		} catch (OvidDomainException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * Query LABORATORY TEST file and returns an unsorted map of all available
	 * tests.
	 * 
	 * @return
	 * 
	 * @return An unsorted map of FMLaboratoryTest objects with the lab test ids
	 *         as it keys and each FMLaboratoryTest carrying the full name of
	 *         the lab test and a short (print) name of the lab test.
	 * @throws OvidDomainException
	 */
	protected void setLabTestsCache() throws OvidDomainException {
		RPCConnection rpcConnection;
		RPCResAdapter adapter = null;

		try {
			OvidLabTestCache.getInstance().flush();

			logger.debug("create lab test cache");

			// setup connection
			rpcConnection = getConnection();
			rpcConnection.setContext(FMUtil.FM_RPC_CONTEXT);
			adapter = new RPCResAdapter(rpcConnection, FMUtil.FM_RPC_NAME);

			FMFile labFile = new FMFile(LABORATORY_TEST_FILE);
			labFile.addField(".01");
			labFile.addField("PRINT NAME");

			FMQueryList query = new FMQueryList(adapter, labFile);
			FMResultSet results = query.execute();

			FMLaboratoryTest result = null;
			while (results.next()) {
				result = new FMLaboratoryTest(results);
				if (!result.getName().startsWith("ZZ")) {
					OvidLabTestCache.getInstance().addToCache(result.getIEN(), result);
				}
			}

			logger.debug("created lab test cache");
		} catch (ResException e1) {
			logger.error("Exception querying FILE " + LABORATORY_TEST_FILE, e1);
			throw new OvidDomainException(e1);
		} catch (RPCException e2) {
			logger.error("Exception querying FILE " + LABORATORY_TEST_FILE, e2);
			throw new OvidDomainException(e2);
		}
	}

	/**
	 * Retrieve lab test info.
	 * 
	 * @param name
	 *            identifier for the lab test
	 * @param conn
	 *            a valid connection
	 * @return a FMLaboratoryTest object
	 */
	public FMLaboratoryTest getLabTestById(String id) {

		FMLaboratoryTest fmLabTest = null;

		if (id != null && !id.isEmpty()) {
			try {
				fmLabTest = OvidLabTestCache.getInstance().getByKey(id);
			} catch (GenericCacheException gce) {
				logger.error("Exception in getLabTestById()", gce);
			}
		}
		logger.debug(String.format("Retrieve lab test by id to get short name [id=%s][result=%s]", id, fmLabTest));

		return fmLabTest;
	}

	/**
	 * Retrieve specimens info for a list of orders.
	 * 
	 * @param orderIENs
	 *            a collection of order numbers.
	 * @param conn
	 *            a valid connection to vista.
	 * @return
	 */
	protected Map<String, FMTopographyField> getSpecimensInfo(Collection<String> ids) {

		Map<String, FMTopographyField> specimens = null;
		TopographyFieldRepository repo = null;

		if (ids != null && !ids.isEmpty()) {
			try {
				repo = new TopographyFieldRepository(getConnection());
				specimens = repo.getTopographiesByIEN(ids);
			} catch (OvidDomainException e) {
				logger.error("Exception in getSpecimensInfo()", e);
			}
		}

		return specimens;
	}

	/**
	 * Retrieve specimen info.
	 * 
	 * @param name
	 *            specimen name
	 * @param conn
	 *            a valid connection
	 * @return
	 */
	protected void addSpecimensByNameToCache(Collection<String> names) {

		TopographyFieldRepository repo = null;

		if (names != null && !names.isEmpty()) {
			try {
				repo = new TopographyFieldRepository(getConnection());

				// query Topography global for info and add record to cache.
				for (String name : names) {
					repo.addTopographyToCache(name);
				}
			} catch (OvidDomainException e) {
				logger.error("Exception in addSpecimenToCache()", e);
			}
		}
	}

	protected FMTopographyField getSpecimenFromCache(String name) {
		FMTopographyField specimen = null;

		try {
			specimen = OvidTopographyCache.getInstance().getByKey(name);
		} catch (GenericCacheException e) {
			logger.error("Exception in getSpecimenFromCache()", e);
		}

		return specimen;
	}

	/**
	 * Extracts lab tests groupings from CUMMULATIVE REPORTS stored in LAB
	 * REPORTS global and returns an unsorted map of lab tests and associated
	 * lab groups. A lab test can be associates with one or more lab groups.
	 * 
	 * @return An unsorted map of lab tests and lab groups with the lab test id
	 *         as key and collection of lab groups as its values.
	 * 
	 * @throws OvidDomainException
	 */
	public Map<String, Collection<String>> getLabGroups() throws OvidDomainException {
		Map<String, Collection<String>> map = Collections.synchronizedMap(new HashMap<String, Collection<String>>());

		RPCConnection rpcConnection;
		RPCResAdapter adapter = null;

		try {
			// setup connection
			rpcConnection = getConnection();
			rpcConnection.setContext(FMUtil.FM_RPC_CONTEXT);
			adapter = new RPCResAdapter(rpcConnection, FMUtil.FM_RPC_NAME);

			FMQueryList query = new FMQueryList(adapter, FMLabReport.getFileInfoForClass());

			FMResultSet results = query.execute();
			if (results != null && (results.getError() == null || results.getError().isEmpty())) {
				FMLabReport report = null;
				while (results.next()) {
					report = new FMLabReport(results);

					if (report.getName().trim().equalsIgnoreCase(CUMMULATIVE_REPORTS)) {

						// get report headings
						getReportHeadings(report, adapter);

						if (logger.isDebugEnabled())
							log(report);

						// each headings can have one or more lab groupings and
						// each
						// lab grouping can contain one or more lab tests.
						for (FMLabReportHeader h : report.getMajorHeaders()) {
							for (FMLabReportTestGroup group : h.getMinorHeaders()) {
								String labTestId = null;

								try {
									String labGroupLabel = group.getMinorHeader();
									for (FMLabReportLabTest t : group.getLabTests()) {
										if (t.getLabTest() != null) {
											labTestId = t.getLabTest().toString();

											Collection<String> values = null;

											// build lab test and lab groupings
											// relationships
											if (!map.containsKey(labTestId)) {
												values = new ArrayList<String>();
											} else {
												values = map.get(labTestId);
											}

											values.add(labGroupLabel);
											map.put(labTestId, values);
										}
									}
								} catch (NumberFormatException ignore) {
								}
							}
						}

						break;
					}
				}
			} else {
				logger.error(results.getError());
				throw new OvidDomainException(results.getError());
			}

			return map;
		} catch (ResException e1) {
			logger.error("Exception querying " + CUMMULATIVE_REPORTS, e1);
			throw new OvidDomainException(e1);
		} catch (RPCException e2) {
			logger.error("Exception querying " + CUMMULATIVE_REPORTS, e2);
			throw new OvidDomainException(e2);
		}
	}

	/**
	 * 
	 * @param report
	 *            a FMRecord representing a entry from the
	 */
	private void getReportHeadings(FMLabReport report, RPCResAdapter adapter) throws OvidDomainException {

		FMLabReportHeader majorHeaderField = report.getMajorHeader();
		try {
			FMQueryList secondaryQuery = new FMQueryList(adapter, majorHeaderField.getFile());

			FMResultSet results = secondaryQuery.execute();

			if (results != null && (results.getError() == null || results.getError().isEmpty())) {
				while (results.next()) {
					// extract report heading
					FMLabReportHeader d = new FMLabReportHeader(results);

					// extract all lab groupings under this report heading
					d.setParent(report);
					d.setMinorHeaders(getLabGroupings(d, adapter));

					// add to lab report
					report.addMajorHeader(d);
				}
			} else {
				logger.error(results.getError());
				throw new OvidDomainException(results.getError());
			}
		} catch (ResException e) {
			logger.error("Exception querying subfile MAJOR HEADER of LAB REPORTS", e);
			throw new OvidDomainException(e);
		}
	}

	/**
	 * 
	 * @param header
	 * @return
	 */
	private Collection<FMLabReportTestGroup> getLabGroupings(FMLabReportHeader header, RPCResAdapter adapter) throws OvidDomainException {

		Collection<FMLabReportTestGroup> labGroups = new ArrayList<FMLabReportTestGroup>();

		try {
			FMLabReportTestGroup minorHeaderField = header.getMinorHeader();

			// extract lab groups
			FMQueryList mhQuery = new FMQueryList(adapter, minorHeaderField.getFile());
			FMResultSet mhQueryResults = mhQuery.execute();

			if (mhQueryResults != null && (mhQueryResults.getError() == null || mhQueryResults.getError().isEmpty())) {

				FMLabReportLabTest labTestField = null;
				FMLabReportLabTest labTestResult = null;

				FMLabReportTestGroup labGroupResult = null;

				FMQueryList labTestQuery = null;
				FMResultSet labTestQueryResults = null;

				while (mhQueryResults.next()) {
					labGroupResult = new FMLabReportTestGroup(mhQueryResults);

					// extract lab tests in this lab group
					labGroupResult.setParent(header);
					labTestField = labGroupResult.getLabTest();
					labTestQuery = new FMQueryList(adapter, labTestField.getFile());
					labTestQueryResults = labTestQuery.execute();

					if (labTestQueryResults != null && (labTestQueryResults.getError() == null || labTestQueryResults.getError().isEmpty())) {
						while (labTestQueryResults.next()) {
							labTestResult = new FMLabReportLabTest(labTestQueryResults);
							labGroupResult.addLabTest(labTestResult);
						}
					} else {
						logger.error(labTestQueryResults.getError());
						throw new OvidDomainException(mhQueryResults.getError());
					}

					labGroups.add(labGroupResult);
				}
			} else {
				logger.error(mhQueryResults.getError());
				throw new OvidDomainException(mhQueryResults.getError());
			}
			return labGroups;
		} catch (ResException e) {
			logger.error("Exception querying subfile MINOR HEADER of LAB REPORTS", e);
			throw new OvidDomainException(e);
		}
	}

	private void log(FMRecord record) {
		if (record == null)
			return;

		try {
			if (record instanceof FMLabReport) {
				FMLabReport d = (FMLabReport) record;

				logger.debug("[FMLabReport] name=[" + d.getName() + "] ien=[" + d.getIEN() + "]");
				logger.debug("  # of MAJOR HEADERS=[" + d.getMajorHeaders().size() + "]");
				for (FMLabReportHeader h1 : d.getMajorHeaders()) {
					logger.debug("  |-- " + h1);
					logger.debug("    # of MINOR HEADERS=[" + h1.getMinorHeaders().size() + "]");
					for (FMLabReportTestGroup h2 : h1.getMinorHeaders()) {
						logger.debug("    |-- " + h2);
						logger.debug("      # of LAB TESTS=[" + h2.getLabTests() + "]");
						for (FMLabReportLabTest lt : h2.getLabTests()) {
							logger.debug("      |-- " + lt);
						}
					}
				}
			} else {
				logger.debug("[FMRecord] " + record);
			}
		} catch (Exception ignore) {
		}
	}
}
