package com.agilex.healthcare.mobilehealthplatform.ovid;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import com.agilex.healthcare.mobilehealthplatform.ovid.domain.OvidBatteryLabResult;
import com.agilex.healthcare.mobilehealthplatform.ovid.domain.OvidLabResult;
import com.agilex.healthcare.mobilehealthplatform.ovid.domain.OvidLabTest;
import com.agilex.healthcare.mobilehealthplatform.ovid.domain.OvidLabTestCache;
import com.agilex.healthcare.mobilehealthplatform.ovid.domain.OvidMicroLabResult;
import com.agilex.healthcare.mobilehealthplatform.ovid.domain.OvidTestResult;
import com.agilex.healthcare.mobilehealthplatform.ovid.fmdomain.FMTopographyField;
import com.agilex.healthcare.mobilehealthplatform.ovid.util.OvidDateHelper;
import com.agilex.healthcare.utility.ListHelper;
import com.medsphere.common.cache.GenericCacheException;
import com.medsphere.fileman.FMUtil;
import com.medsphere.fmdomain.FMLaboratoryTest;
import com.medsphere.fmdomain.FMOrder;
import com.medsphere.fmdomain.FMOrderResponse;
import com.medsphere.ovid.domain.ov.OrderRepository;
import com.medsphere.ovid.domain.ov.OvidDomainException;
import com.medsphere.ovid.model.domain.patient.IsAPatientItem;
import com.medsphere.ovid.model.domain.patient.PatientResult;
import com.medsphere.ovid.model.domain.patient.ResultDetail;
import com.medsphere.vistarpc.RPCConnection;
import com.medsphere.vistarpc.RPCException;
import com.medsphere.vistarpc.RPCResponse;
import com.medsphere.vistarpc.RPCResponse.ResponseType;
import com.medsphere.vistarpc.VistaRPC;

public class LabResultRepository extends LabTestRepository {

	private final static org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(LabResultRepository.class);

	// private String context;
	private final static boolean detailsIndicator = true;
	//public final static String labDataGlobal = "63";
	public final static String chemLabSubcript = "CH";
	public final static String hemeLabSubcript = "HE";
	public final static String cprsGUIContext = "OR CPRS GUI CHART";

	private final static String RPC_ORRC_RESULTS_BY_DATE = "ORRC RESULTS BY DATE";
	private final static String RPC_ORWGRPC_ITEMS = "ORWGRPC ITEMS";
	private final static String RPC_ORWLRR_INTERIMG = "ORWLRR INTERIMG";
	private final static String RPC_ORWGRPC_ITEMDATA = "ORWGRPC ITEMDATA";
	private final static String RPC_ORWLRR_INFO = "ORWLRR INFO";
	private final static String RPC_ORWRP_REPORT_TEXT = "ORWRP REPORT TEXT";
	private final static String HEME_CHEM_REPORT_TYPE = "OR_CH:CHEM & HEMATOLOGY~CH;ORDV02;3;999";
	private final static String MICRO_REPORT_TYPE = "OR_MIC:MICROBIOLOGY~MI;ORDV05;38;999";
	
	public enum LabGlobal {
		 HEMECHEM("63"), MICRO("63MI");

		 private String globalFile;

		 private LabGlobal(String gf) {
			 globalFile = gf;
		 }

		 public String getGlobalFile() {
		   return globalFile;
		 }
	}
	
	public LabResultRepository(RPCConnection connection) {
		super(connection);
	}

	/**
	 * Returns a list of lab results for a patient within a specified date
	 * ranges.
	 * 
	 * @param patientId
	 *            unique identifier of a patient
	 * @param beginDate
	 *            start date/time
	 * @param endDate
	 *            end date/time
	 * 
	 * @return a collection of OvidLabResult objects
	 * @throws OvidDomainException
	 */
	public Collection<OvidLabResult> getResultsByDateOld(String patientId, Date beginDate, Date endDate) throws OvidDomainException {

		Collection<OvidLabResult> ovidLabResults = new ArrayList<OvidLabResult>();

		// get list of lab tests for this patient
		Map<String, OvidLabTest> labtests = getLabTestsByPatient(patientId, LabGlobal.HEMECHEM);

		// get lab results
		Collection<IsAPatientItem> items = getResults(patientId, getConnection().getDUZ(), beginDate, endDate, detailsIndicator);

		OvidLabResult labResult = null;
		String mapKey = "";
		for (IsAPatientItem item : items) {
			PatientResult ptItem = (PatientResult) item;

			if (ptItem.getDetails() != null && !ptItem.getDetails().isEmpty()) {
				// there seemed to be a bug in VistA's RPC
				// "ORRC_RESULTS_BY_DATE" where results outside
				// the specified date ranges are being returned. Must do a check
				// to filter these results out.
				OvidLabTest labtest;
				if (isResultWithinDateRanges(ptItem.getDateTime(), beginDate, endDate)) {
					Date resultDTM = ptItem.getDateTime();

					if (ptItem.getDetails().size() > 1) {
						labResult = new OvidBatteryLabResult();
						labResult.setResultedDate(resultDTM);
						((OvidBatteryLabResult) labResult).setBatteryTestName(ptItem.getMessage());
						// labResult.setAccessionNumber(ptItem.getId());
						labResult.setOrderId(ptItem.getId());
					} else {
						// single test
						labResult = new OvidLabResult();
						labResult.setResultedDate(resultDTM);
						// labResult.setAccessionNumber(ptItem.getId());
						labResult.setOrderId(ptItem.getId());
					}

					for (ResultDetail detail : ptItem.getDetails()) {
						mapKey = detail.getTestName();
						labtest = labtests.get(mapKey);

						if (labtest != null) {
							labResult.getResults().add(
									new OvidTestResult(((labtest.getShortName() != null && !labtest.getShortName().isEmpty()) ? labtest.getShortName() : labtest.getTestName()), labtest.getId(),
											detail.getValue(), detail.getUnits(), detail.getReferenceRange(), detail.getIndicator()));
						} else {
							logger.error("No matching lab test found - " + mapKey);
						}
					}

//					logger.debug(labResult.toString());

					ovidLabResults.add(labResult);
				}
			}
		}

		// retrieve specimens info for each order
		if (ovidLabResults != null && !ovidLabResults.isEmpty()) {
			// temporary list of order numbers
			Collection<String> orderIds = new ArrayList<String>();
			String orderid = "";
			for (OvidLabResult ovidLabResult : ovidLabResults) {
				orderid = ovidLabResult.getOrderId();
				orderIds.add(orderid.replace("ORR:", ""));
			}

			// get a hash map of <order id, specimen id>
			Map<String, String> orderSpecimenIdList = getSpecimenIdsByOrder(orderIds, getConnection());

			// retrieve details about each specimen
			if (orderSpecimenIdList != null && !orderSpecimenIdList.isEmpty()) {
				// build a list of unique specimen id
				Map<String, String> specimenIds = Collections.synchronizedMap(new HashMap<String, String>());
				for (String specimenId : orderSpecimenIdList.values()) {
					if (!specimenIds.containsKey(specimenId)) {
						specimenIds.put(specimenId, specimenId);
					}
				}

				// get a hash map of <specimen id, specimen details info>
				Map<String, FMTopographyField> specimensInfo = getSpecimensInfo(specimenIds.values());

				if (specimensInfo != null && !specimensInfo.isEmpty()) {
					for (OvidLabResult ovidLabResult : ovidLabResults) {
						orderid = ovidLabResult.getOrderId();
						orderid = orderid.replace("ORR:", "");

						if (orderSpecimenIdList.containsKey(orderid)) {
							// set specimen id
							ovidLabResult.setSpecimenId(orderSpecimenIdList.get(orderid));
							// set specimen name
							ovidLabResult.setSpecimen(specimensInfo.get(ovidLabResult.getSpecimenId()).getName());
						} else {
							logger.error("No specimen identified in ORDER " + orderid);
						}
					}
				}
			}
		}

		return ovidLabResults;
	}

	/**
	 * Calls RPC "ORRC RESULTS BY DATE" to returns a list of orders for a
	 * patient with results within a given time frame.
	 * 
	 * @param patientDfn
	 *            unique identifier of a patient
	 * @param duz
	 *            unique identifier of a user requesting this info
	 * @param startDate
	 *            date to start searching for orders, in the form YYYYMMDDHHMMSS
	 * @param stopDate
	 *            date to stop searching for orders, in the form YYYYMMDDHHMMSS
	 * @param details
	 *            this flag indicates whether to return the results with each
	 *            order (true) or just the list of order ID's (false).
	 * 
	 * @return This call returns a list of orders for a patient that were placed
	 *         between <i>startDate<i> and <i>stopDate<i> and now have results
	 *         available that have not yet been acknowledged by USER, in the
	 *         form: "Item=" _ ID ^ Text ^ Result Date in the form
	 *         YYYYMMDDHHMMSS where ID is the order number in the form
	 *         "ORR:###". If the RESULTS parameter was true, other entries will
	 *         also be included as: "Data=" _ Test ^ Value ^ Units ^ Reference
	 *         Range ^ Critical flag, or "Text=" _ line of report text repeated
	 *         as needed until the next "Item" entry.
	 * @throws OvidDomainException
	 */
	public Collection<IsAPatientItem> getResults(String patientDfn, String duz, Date startDate, Date stopDate, boolean details) throws OvidDomainException {
		Collection<IsAPatientItem> collection = new ArrayList<IsAPatientItem>();
		RPCConnection rpcConnection;
		String fmResultedDTM = null;

		try {
			rpcConnection = getConnection();
			rpcConnection.setContext(cprsGUIContext);
		} catch (RPCException e) {
			throw new OvidDomainException(e);
		}

		VistaRPC rpc = new VistaRPC(RPC_ORRC_RESULTS_BY_DATE, ResponseType.ARRAY);

		rpc.setParam(1, patientDfn);
		rpc.setParam(2, duz);
		rpc.setParam(3, FMUtil.dateToFMDate(startDate));
		rpc.setParam(4, FMUtil.dateToFMDate(stopDate));
		if (details) {
			rpc.setParam(5, "1");
		} else {
			rpc.setParam(5, "0");
		}
		String items[] = null;
		try {
			RPCResponse response = rpcConnection.execute(rpc);
			items = response.getArray();
			if (isEmptyResult(items)) {
				return collection;
			}
			IsAPatientItem item = null;

			for (int idx = 0; idx < items.length; ++idx) {
				if (items[idx].startsWith("Item=")) {
					if (item != null) {
						collection.add(item);
						item = null;
					}

//					logger.debug(items[idx]);
					String parts[] = items[idx].split("\\^", -1);

					// get resulted date/time
					if (parts.length > 2 && !parts[2].isEmpty() && !parts[2].equals("-1")) {
						fmResultedDTM = transformToFMDate(parts[2]);

						// there seemed to be a bug in VistA's RPC
						// "ORRC_RESULTS_BY_DATE" where
						// results outside the specified date ranges are being
						// returned. Must check
						// and filter these results out.
						if (isResultWithinDateRanges(FMUtil.fmDateToDate(fmResultedDTM), startDate, stopDate)) {
							String id = parts[0].substring(5, parts[0].length());
							String message = parts[1];

							item = new PatientResult(id, message, "", FMUtil.fmDateToDate(fmResultedDTM));

							for (int itemIdx = idx + 1; itemIdx < items.length && idx < items.length; ++itemIdx) {
//								logger.debug(items[itemIdx]);

								if (items[itemIdx].startsWith("Text=")) {
									((PatientResult) item).addText(items[itemIdx].substring(5, items[itemIdx].length()));
								} else if (items[itemIdx].startsWith("Cmnt=")) {
									((PatientResult) item).addComment(items[itemIdx].substring(5, items[itemIdx].length()));
								} else if (items[itemIdx].startsWith("Data=")) {
									String dataParts[] = items[itemIdx].split("\\^", -1);
									String testName = dataParts[0].substring(5, dataParts[0].length());
									String value = dataParts[1];
									String units = dataParts[2];
									String referenceRange = dataParts[3];
									String indicator = dataParts[4];
									((PatientResult) item).addDetail(new ResultDetail(testName, value, units, referenceRange, indicator));
								} else if (items[itemIdx].startsWith("Item=")) {
									idx = itemIdx - 1;
									break;
								}
							}
						}
					}
				}
			}

			if (item != null) {
				collection.add(item);
			}

		} catch (RPCException e1) {
			throw new OvidDomainException(e1);
		} catch (ParseException e2) {
			throw new OvidDomainException(e2);
		} catch (Exception e3) {
			logger.error("fmResultedDTM=" + fmResultedDTM);
			e3.printStackTrace();
		}

		return collection;
	}

	/**
	 * Calls RPC "ORWLRR INTERIMG" (context "OR CPRS GUI CHART") and returns
	 * most recent lab result for a patient.
	 * 
	 * @param patientId
	 *            unique identifier of patient
	 * 
	 * @return
	 * @throws OvidDomainException
	 */
	public OvidLabResult getMostRecentResult(String patientId) throws OvidDomainException {

		RPCConnection rpcConnection;
		OvidLabResult labResult = new OvidLabResult();

		try {
			rpcConnection = getConnection();
			rpcConnection.setContext(cprsGUIContext);
		} catch (RPCException e) {
			logger.error("Exception while setting the context", e);
			throw new OvidDomainException(e);
		}

		VistaRPC rpc = new VistaRPC(RPC_ORWLRR_INTERIMG, ResponseType.ARRAY);

		// set patient
		rpc.setParam(1, patientId);
		// set end date
		rpc.setParam(2, FMUtil.dateToFMDate(new Date()));
		// set descending order
		rpc.setParam(3, "1");
		rpc.setParam(4, "1");

		try {
			RPCResponse response = rpcConnection.execute(rpc);
			if (response.getError() == null || response.getError().isEmpty()) {
				String items[] = response.getArray();
				if (isEmptyResult(items)) {
					logger.debug("No data!!");
				} else {
					// only parse the first two lines for data
					// Examples:
					// item 1: 1^CH^3110208.080003^72^SERUM^CH 0208
					// 590^LABTECH,SEVENTEEN^^^3110208.080003
					// item 2: 173^CREATININE^ 2.3^H^mg/dL^.9 - 1.4
					for (int idx = 0; (idx < items.length) || (idx < 3); ++idx) {
						// only the first two items are relevant
						String parts[] = items[idx].split("\\^", -1);
						if (idx == 0) {
							labResult.setLabType(parts[1]);
							labResult.setAccessionNumber(parts[5]);
							labResult.setSpecimen(parts[4]);
							labResult.setSpecimenId(parts[3]);

							String datePart = parts[2];
							Date resultedDate = null;
							if (datePart != null && !datePart.isEmpty()) {
								try {
									resultedDate = FMUtil.fmDateToDate(datePart);
								} catch (ParseException ignore) {
									resultedDate = null;
								}
							}
							labResult.setResultedDate(resultedDate);
						} else if (idx == 1) {
							OvidTestResult result = new OvidTestResult();
							result.setTestId(parts[0]);
							result.setTestName(parts[1].trim());
							result.setValue(parts[2].trim());
							result.setIndicator(parts[3]);
							result.setUnits(parts[4]);
							result.setReferenceRange(parts[5].trim());

							labResult.getResults().add(result);
						}
					}
				}

				return labResult;
			} else {
				logger.error(response.getError());
				throw new OvidDomainException(response.getError());
			}
		} catch (RPCException e) {
			logger.error("Exception while executing RPC \"" + RPC_ORWLRR_INTERIMG + "\"", e);
			throw new OvidDomainException(e);
		}
	}

	/**
	 * Calls RPC "ORWGRPC ITEMDATA" (context "OR CPRS GUI CHART") and returns
	 * all results of a specific lab test for a patient.
	 * 
	 * @param patientId
	 *            unique identifier of patient
	 * @param testId
	 *            unique identifier of lab test
	 * @return
	 * @throws OvidDomainException
	 */
	public Collection<OvidLabResult> getResultsByTest(String patientId, String testId) throws OvidDomainException {

		RPCConnection rpcConnection;
		Collection<OvidLabResult> labResults = new ArrayList<OvidLabResult>();

		try {
			rpcConnection = getConnection();
			rpcConnection.setContext(cprsGUIContext);
		} catch (RPCException e) {
			logger.error("Exception while setting the context", e);
			throw new OvidDomainException(e);
		}

		try {
			// obtains information about this test including its reference
			// ranges.
			Map<String, OvidLabTest> testInfoMap = getInfoByTest(testId, rpcConnection);

			VistaRPC rpc = new VistaRPC(RPC_ORWGRPC_ITEMDATA, ResponseType.ARRAY);

			// set pointer to lab data file
			rpc.setParam(1, "63" + "^" + testId);
			// set end date
			rpc.setParam(2, FMUtil.dateToFMDate(new Date()));
			// set patient
			rpc.setParam(3, patientId);

			RPCResponse response = rpcConnection.execute(rpc);
			if (response.getError() == null || response.getError().isEmpty()) {
				String items[] = response.getArray();
				if (isEmptyResult(items)) {
					logger.debug("No data!!");
				} else {
					// Result sample: 63^173^2950901.131534^^1.1^^72^SERUM^1
					OvidLabResult labResult = null;
					OvidLabTest testInfo = null;
					for (int idx = 0; (idx < items.length) || (idx < 3); ++idx) {
						// logger.debug(items[idx]);
						// response sample:
						// 63^173^2950901.131534^^1.1^^72^SERUM^1
						String parts[] = items[idx].split("\\^", -1);
						if (testInfoMap != null)
							testInfo = testInfoMap.get(parts[7]);

						labResult = new OvidLabResult();
						labResult.setSpecimenId(parts[6]);
						labResult.setSpecimen(parts[7]);
						String datePart = parts[2];
						if (datePart != null && !datePart.isEmpty()) {
							try {
								labResult.setResultedDate(FMUtil.fmDateToDate(datePart));
							} catch (ParseException ignore) {
								logger.error(ignore);
							}
						}

						if (testInfo != null) {
							OvidTestResult result = new OvidTestResult(testInfo.getTestName(), parts[1], parts[4], testInfo.getUnits(), testInfo.getLowReferenceRange() + " - "
									+ testInfo.getHiReferenceRange(), parts[5]);
							result.setRefLow(testInfo.getLowReferenceRange());
							result.setRefHi(testInfo.getHiReferenceRange());
							labResult.getResults().add(result);
						} else {
							labResult.getResults().add(new OvidTestResult("", parts[1], parts[4], "", "", parts[5]));
						}

						labResults.add(labResult);
					}
				}

				return labResults;
			} else {
				logger.error(response.getError());
				throw new OvidDomainException(response.getError());
			}
		} catch (RPCException e) {
			logger.error("Exception while executing RPC \"" + RPC_ORWGRPC_ITEMDATA + "\"", e);
			throw new OvidDomainException(e);
		}
	}

	/**
	 * Calls RPC "ORWLRR INFO" (context "OR CPRS GUI CHART") and returns all
	 * information about this test.
	 * 
	 * @param testId
	 * @return
	 * @throws OvidDomainException
	 */
	private Map<String, OvidLabTest> getInfoByTest(String testId, RPCConnection rpcConnection) throws OvidDomainException {
		Map<String, OvidLabTest> labTests = new HashMap<String, OvidLabTest>();

		try {
			VistaRPC rpc = new VistaRPC(RPC_ORWLRR_INFO, ResponseType.ARRAY);
			rpc.setParam(1, testId);

			RPCResponse response = rpcConnection.execute(rpc);
			if (response.getError() == null || response.getError().isEmpty()) {
				String[] items = response.getArray();
				if (isEmptyResult(items)) {
					logger.debug("No data!!");
				}

				OvidLabTest labTest = null;

				for (int idx1 = 1; idx1 < items.length; idx1++) {
					if (items[idx1].startsWith("Site/Specimen:")) {
						String[] parts = items[idx1].split("\\:", -1);

						labTest = new OvidLabTest(testId, items[0].trim());
						labTest.setSpecimen(parts[1].trim());

						for (int idx2 = idx1 + 1; idx2 < items.length; idx2++) {
							if (items[idx2].contains("Reference range low")) { // vavista
								parts = items[idx2].split("\\:", -1);
								labTest.setLowReferenceRange(parts[1].trim());
							} else if (items[idx2].contains("Reference range high")) { // vavista
								parts = items[idx2].split("\\:", -1);
								labTest.setHiReferenceRange(parts[1].trim());
							} else if (items[idx2].contains("Reference range:")) { // open
																					// vista
								parts = items[idx2].split("\\:", -1);
								parts = parts[1].split(" - "); // to avoid
																// splitting on
																// a negative
																// number
								// handle cases with only a low or only a high
								// reference range
								if (parts[0].trim().length() > 0) {
									labTest.setLowReferenceRange(parts[0].replace("\"", "").trim());
								}
								if (parts.length > 1 && parts[1].trim().length() > 0) {
									labTest.setHiReferenceRange(parts[1].replace("\"", "").trim());
								}

							} else if (items[idx2].contains("Units")) {
								parts = items[idx2].split("\\:", -1);
								labTest.setUnits(parts[1].trim());
							} else if (items[idx2].startsWith("Site/Specimen:")) {
								idx1 = idx2 - 1;
								break;
							}
						}

						labTests.put(labTest.getSpecimen(), labTest);
					}
				}
			} else {
				logger.error(response.getError());
				throw new OvidDomainException(response.getError());
			}
		} catch (RPCException ex) {
			logger.error("Exception while executing RPC \"" + RPC_ORWLRR_INFO + "\"", ex);
			throw new OvidDomainException(ex);
		}

		return labTests;
	}

	/**
	 * Calls RPC "ORWGRPC ITEMS" to get a list of unique lab tests of a patient.
	 * 
	 * @param patientId
	 *            unique identifier for a patient
	 * @return Map a map of resulted lab tests where each key is the test name.
	 * @throws OvidDomainException
	 */
	public Map<String, OvidLabTest> getLabTestsByPatient(String patientId, LabGlobal labType) throws OvidDomainException {
		final int HEMECHEM_LABTYPENAME_INDEX = 8;
		//final int LABTESTDTM_INDEX = 5;
		final int LABTESTNAME_INDEX = 3;
		final int LABTESTID_INDEX = 1;

		RPCConnection rpcConnection;
		Map<String, OvidLabTest> labTests = new HashMap<String, OvidLabTest>();

		try {
			rpcConnection = getConnection();
			rpcConnection.setContext(cprsGUIContext);
		} catch (RPCException e) {
			logger.error("Exception while setting the context", e);
			throw new OvidDomainException(e);
		}

		try {
			VistaRPC rpc = new VistaRPC(RPC_ORWGRPC_ITEMS, ResponseType.GLOBAL_ARRAY);

			// set patient
			rpc.setParam(1, patientId);
			// set lab data global
			//rpc.setParam(2, "63");
			rpc.setParam(2, labType.getGlobalFile());

			RPCResponse response = rpcConnection.execute(rpc);
			if (response.getError() == null || response.getError().isEmpty()) {
				String items[] = response.getArray();
				if (isEmptyResult(items)) {
					logger.debug("No data!!");
				} else {
					FMLaboratoryTest fmLabTest = null;
					OvidLabTest labTest = null;
					String parts[];
					String subparts[];
					String tmpStr;
					for (int idx = 0; idx < items.length; ++idx) {
						parts = items[idx].split("\\^");
//						logger.debug("Line: " + idx + ": " + items[idx]);

						if (labType.equals(LabGlobal.HEMECHEM)) {
							// 63^<lab id>^^<test name>^^<? date>^<lab type id>^lab - <lab type long label>^<lab type short label>
							// Example: 63^1^^WBC^^3080130.080011^10^lab - HEMATOLOGY^HE
							labTest = new OvidLabTest(ListHelper.getValue(parts, LABTESTID_INDEX), ListHelper.getValue(parts, LABTESTNAME_INDEX));
							labTest.setLabType(ListHelper.getValue(parts, HEMECHEM_LABTYPENAME_INDEX));
						}
						else if (labType.equals(LabGlobal.MICRO)) {
							// 63MI^M;T;<test id>^^<test name>^^<? date>
							// Example:
							//   63MI^M;S;360^^SPECIMEN: SPUTUM^^2970227.074742
							//   63MI^M;T;548^^TEST: CULTURE & SUSCEPTIBILITY^^2970227.074742
							if (ListHelper.getValue(parts, LABTESTID_INDEX).startsWith("M;T")) {
								subparts = parts[LABTESTID_INDEX].split("\\;");
								tmpStr = ListHelper.getValue(parts, LABTESTNAME_INDEX);
								tmpStr = tmpStr.replace("TEST:", "");							
								labTest = new OvidLabTest(ListHelper.getValue(subparts, 2), tmpStr.trim());
								labTest.setLabType("MI");
							}
						}

						// check for duplication
						if (labTest != null) {
							if (labTests.containsKey(labTest.getTestName())) {
								logger.error("Duplicate test name [" + labTest.getTestName() + "]");
							} else {
								fmLabTest = getLabTestById(labTest.getId());
								if (fmLabTest != null)
									labTest.setShortName(fmLabTest.getValue("PRINT NAME"));
								else
									labTest.setShortName(labTest.getTestName());
	
								labTests.put(labTest.getTestName(), labTest);
							}
						}
					}
				}

				return labTests;
			} else {
				logger.error(response.getError());
				throw new OvidDomainException(response.getError());
			}
		} catch (RPCException e) {
			logger.error("Exception while executing RPC \"" + RPC_ORWGRPC_ITEMS + "\"", e);
			throw new OvidDomainException(e);
		}
	}
	
	private boolean isResultWithinDateRanges(Date resultDate, Date startDate, Date stopDate) {
		if (resultDate != null && startDate != null && stopDate != null) {
			if (resultDate.getTime() >= startDate.getTime() && resultDate.getTime() <= stopDate.getTime())
				return true;
		}

		return false;
	}

	private String transformToFMDate(String hl7date) {
		// Pad zeros for missing HH, mm, or ss
		StringBuffer sb = new StringBuffer(hl7date.substring(0, hl7date.length() - 5));
		for (int i = (19 - hl7date.length()); i > 0; --i) {
			sb.append("0");
		}
		sb.append(hl7date.substring(hl7date.length() - 5));

		return OvidDateHelper.hl7DateToFMDate(sb.toString());
	}

	/**
	 * Retrieve specimen id for each order.
	 * 
	 * @param orderIENs
	 *            a collection of order numbers.
	 * @param conn
	 *            a valid connection to vista.
	 * @return a hash map of <order id,specimen id>
	 */
	private Map<String, String> getSpecimenIdsByOrder(Collection<String> orderIENs, RPCConnection conn) {

		Map<String, String> orderSpecimenIdList = null;

		if (orderIENs != null && !orderIENs.isEmpty() && conn != null) {
			try {
				OrderRepository orderRepo = new OrderRepository(conn);
				Collection<FMOrder> orders = orderRepo.getOrdersByIEN(orderIENs);
				orderRepo.resolveOrderResponses(orders);

				orderSpecimenIdList = Collections.synchronizedMap(new HashMap<String, String>());

				for (FMOrder order : orders) {
					Collection<FMOrderResponse> responses = order.getOrderResponses();
					for (FMOrderResponse response : responses) {
						if (response.getId().equalsIgnoreCase("SPECIMEN")) {
							orderSpecimenIdList.put(order.getIEN(), response.getResponseValue());
							break;
						}
					}
				}
			} catch (OvidDomainException e) {
				logger.error("Exception in getSpecimenIdsByOrder()", e);
			}
		}

		return orderSpecimenIdList;
	}

	private OvidLabTest getMatchingLabTest(String testname, Map<String, OvidLabTest> labtests) {
		if (testname == null || testname.isEmpty())
			return null;

		// check for full name match
		if (labtests.containsKey(testname)) {
			return labtests.get(testname);
		}

		// not matching on full name
		Set<Entry<String, OvidLabTest>> set = labtests.entrySet();
		Iterator<Entry<String, OvidLabTest>> iter = set.iterator();
		while (iter.hasNext()) {
			Entry<String, OvidLabTest> entry = iter.next();
			OvidLabTest labtest = (OvidLabTest) entry.getValue();
			if (labtest.getShortName().equalsIgnoreCase(testname)) {
				return labtest;
			}
		}
		
		return null;
	}

	/**
	 * Calls RPC "ORWRP REPORT TEXT" retrieves the report text for a report
	 * selected on the Report tab.
	 * 
	 * @param patientId
	 *            unique identifier for a patient
	 * @return Map a map of resulted lab tests where each key is the test name.
	 * @throws OvidDomainException
	 */
	public Collection<OvidLabResult> getResultsByDate(String patientId, Date startDate, Date stopDate) throws OvidDomainException {

		RPCConnection rpcConnection;
		Collection<OvidLabResult> ovidLabResults = new ArrayList<OvidLabResult>();

		// get list of lab tests for this patient
		Map<String, OvidLabTest> labtests = getLabTestsByPatient(patientId, LabGlobal.HEMECHEM);

		// build lab tests cache
		try {
			if (OvidLabTestCache.getInstance().getCacheValues().isEmpty()) {
				//TODO: I am pretty sure this is not needed since the class init does this
				setLabTestsCache();
			}
		} catch (GenericCacheException e) {
			logger.error("Exception while building OvidLabTestCache", e);
			throw new OvidDomainException(e);
		}

		try {
			rpcConnection = getConnection();
			rpcConnection.setContext(cprsGUIContext);
		} catch (RPCException e) {
			logger.error("Exception while setting the context", e);
			throw new OvidDomainException(e);
		}

		try {
			VistaRPC rpc = new VistaRPC(RPC_ORWRP_REPORT_TEXT, ResponseType.GLOBAL_ARRAY);

			// set patient
			rpc.setParam(1, patientId);
			// set report type
			//rpc.setParam(2, "OR_CH:CHEM & HEMATOLOGY~CH;ORDV02;3;999");
			rpc.setParam(2, HEME_CHEM_REPORT_TYPE);
			rpc.setParam(3, "");
			rpc.setParam(4, "");
			rpc.setParam(5, "");
			// set start date
			rpc.setParam(6, FMUtil.dateToFMDate(startDate));
			// set stop date
			rpc.setParam(7, FMUtil.dateToFMDate(stopDate));

			// Format of response:
			// 1^<facility>
			// 2^<specimen date>
			// 3^<lab test>
			// 4^<specimen>
			// 5^<result>
			// 6^<result indicator>
			// 7^<result unit>
			// 8^<low reference range>
			// 9^<high reference range>
			RPCResponse response = rpcConnection.execute(rpc);
			if (response.getError() == null || response.getError().isEmpty()) {
				String items[] = response.getArray();
				if (isEmptyResult(items)) {
					logger.debug("No data!!");
				} else {
					OvidLabResult labResult = null;
					OvidTestResult testResult = null;
					OvidLabTest labtest = null;
					FMTopographyField specimen = null;
					Collection<String> specimenNames = new ArrayList<String>();

					String parts[];

					for (int idx = 0; idx < items.length; ++idx) {
						parts = items[idx].split("\\^", -1);
//						logger.debug("line " + idx + ": " + items[idx]);

						// get resulted date/time
						if (parts.length > 1 && !parts[1].isEmpty()) {
							if (parts[0].equals("1")) {
								// single test
								labResult = new OvidLabResult();
								testResult = new OvidTestResult();

								for (int rdx = idx + 1; rdx < items.length; ++rdx) {
									parts = items[rdx].split("\\^", -1);
//									logger.debug("line " + rdx + ": " + items[rdx]);

									idx++;

									if (parts[0].equals("2")) {
										labResult.setResultedDate(OvidDateHelper.transformToJavaDate(parts[1]));
									}
									if (parts[0].equals("3")) {
										if (parts[1] != null && !parts[1].isEmpty()) {
											labtest = getMatchingLabTest(parts[1], labtests);

											if (labtest != null) {
												testResult.setTestName((labtest.getShortName() != null && !labtest.getShortName().isEmpty()) ? labtest.getShortName() : labtest.getTestName());
												testResult.setTestId(labtest.getId());
												labResult.setLabType(labtest.getLabType());
											} else {
												logger.error("No matching lab test found - " + parts[1]);
												testResult.setTestName(parts[1]);
											}
										}
									}
									if (parts[0].equals("4")) {
										if (parts[1] != null && !parts[1].isEmpty()) {
											labResult.setSpecimen(parts[1].trim());
											specimen = getSpecimenFromCache(parts[1].trim());
											if (specimen != null)
												labResult.setSpecimenId(specimen.getIEN());
											else {
												if (!specimenNames.contains(labResult.getSpecimen()))
													specimenNames.add(labResult.getSpecimen());
											}
										}
									}
									if (parts[0].equals("5")) {
										testResult.setValue(parts[1]);
									}
									if (parts[0].equals("6")) {
										testResult.setIndicator(parts[1]);
									}
									if (parts[0].equals("7")) {
										testResult.setUnits(parts[1]);
									}
									if (parts[0].equals("8")) {
										testResult.setRefLow(parts[1]);
									}
									if (parts[0].equals("9")) {
										testResult.setRefHi(parts[1]);
										break;
									}
								}

								labResult.getResults().add(testResult);
								ovidLabResults.add(labResult);
								logger.debug(labResult);
							}
						}
					}

					// resolve any missing specimen ids
					logger.info("resolve any missing specimen ids");
					if (specimenNames.size() > 0) {
						logger.info("add specimens to cache");
						addSpecimensByNameToCache(specimenNames);
						for (OvidLabResult entry : ovidLabResults) {
							logger.debug(String.format("checking entry for specimen [id=%s][name=%s]", entry.getSpecimenId(), entry.getSpecimen()));
							if (entry.getSpecimenId() == null || entry.getSpecimenId().isEmpty()) {
								logger.debug(String.format("entry does not have a specimen id, retrieve specimen by name [id=%s][name=%s]", entry.getSpecimenId(), entry.getSpecimen()));
								specimen = getSpecimenFromCache(entry.getSpecimen());
								logger.debug(String.format("retrieved specimen by name [name=%s][specimen=%s]", entry.getSpecimen(), specimen));
							
								//TODO: need test to cover this condition
								if (specimen == null) {
									entry.setSpecimenId(entry.getSpecimen());
								} else {
									entry.setSpecimenId(specimen.getIEN());
								}
								logger.debug(String.format("update rest test specimen to [id=%s][entry=%s]", entry.getSpecimenId() , entry));
							}
						}
						logger.info("end resolve any missing specimen ids");
					}
				}
			} else {
				logger.error(response.getError());
				throw new OvidDomainException(response.getError());
			}
			return ovidLabResults;
		} catch (RPCException e) {
			logger.error("Exception while executing RPC \"" + RPC_ORWRP_REPORT_TEXT + "\"", e);
			throw new OvidDomainException(e);
		}
	}
	
	/**
	 * Calls RPC "ORWRP REPORT TEXT" retrieves the report text for a report
	 * selected on the Report tab.
	 * 
	 * @param patientId
	 *            unique identifier for a patient
	 * @return Map a map of resulted lab tests where each key is the test name.
	 * @throws OvidDomainException
	 */
	public Collection<OvidMicroLabResult> getMicroResultsByDate(String patientId, Date startDate, Date stopDate) throws OvidDomainException {

		RPCConnection rpcConnection;
		Collection<OvidMicroLabResult> ovidLabResults = new ArrayList<OvidMicroLabResult>();

		// get list of lab tests for this patient
		Map<String, OvidLabTest> labtests = getLabTestsByPatient(patientId, LabGlobal.MICRO);
		
		try {
			rpcConnection = getConnection();
			rpcConnection.setContext(cprsGUIContext);
		} catch (RPCException e) {
			logger.error("Exception while setting the context", e);
			throw new OvidDomainException(e);
		}

		try {
			VistaRPC rpc = new VistaRPC(RPC_ORWRP_REPORT_TEXT, ResponseType.GLOBAL_ARRAY);

			// set patient
			rpc.setParam(1, patientId);
			// set report type
			rpc.setParam(2, MICRO_REPORT_TYPE);
			rpc.setParam(3, "");
			rpc.setParam(4, "");
			rpc.setParam(5, "");
			// set start date
			rpc.setParam(6, FMUtil.dateToFMDate(startDate));
			// set stop date
			rpc.setParam(7, FMUtil.dateToFMDate(stopDate));

			// Format of response:
			// 1^<facility>
			// 2^<specimen collection date>
			// 3^<microbiology test name>
			// 4^<collection sample>
			// 5^<specimen>
			// 6^<accession #>
			// 7^<report text> (multiples)
			// 8^<not used>
			RPCResponse response = rpcConnection.execute(rpc);
			if (response.getError() == null || response.getError().isEmpty()) {
				String items[] = response.getArray();
				if (isEmptyResult(items)) {
					logger.debug("No data!!");
				} else {
					OvidMicroLabResult labResult = null;
					OvidLabTest labtest = null;
					FMTopographyField specimen = null;
					Collection<String> specimenNames = new ArrayList<String>();

					Date jDate = null;
					String parts[];

					for (int idx = 0; idx < items.length; ++idx) {
						parts = items[idx].split("\\^", -1);
//						logger.debug("line " + idx + ": " + items[idx]);

						// get resulted date/time
						if (parts.length > 1 && !parts[1].isEmpty()) {

							if (parts[0].equals("1")) {
								// single test
								labResult = new OvidMicroLabResult();

								for (int rdx = idx + 1; rdx < items.length; ++rdx) {
									parts = items[rdx].split("\\^", -1);
//									logger.debug("line " + rdx + ": " + items[rdx]);

									idx++;

									if (parts[0].equals("2")) {
										jDate = OvidDateHelper.transformToJavaDate(parts[1]);
										labResult.setResultedDate(jDate);
										labResult.setCollectionDTM(jDate);
									}
									else if (parts[0].equals("3")) {
										if (parts[1] != null && !parts[1].isEmpty()) {
											labtest = getMatchingLabTest(parts[1], labtests);
											
											if (labtest != null) {
												labResult.setTestName((labtest.getShortName() != null && !labtest.getShortName().isEmpty()) ? labtest.getShortName() : labtest.getTestName());
												labResult.setTestId(labtest.getId());
											} else {
												// no match found from patient's short list, check lab tests cache
												try {
													FMLaboratoryTest fmLabtest = OvidLabTestCache.getInstance().getByKey(parts[1]);
													if (fmLabtest != null) {				
														labResult.setTestId(fmLabtest.getIEN());
														labResult.setTestName(fmLabtest.getValue("PRINT NAME"));
													}
													else {
														logger.error("No matching lab test found - " + parts[1]);
														
														labResult.setTestName(parts[1]);
													}
												} catch (GenericCacheException ignore) {}
												
												//labResult.setTestName(parts[1]);
											}
											
											labResult.setLabType("MI");
										}
									}
									else if (parts[0].equals("4")) {
										labResult.setCollectionSample(parts[1]);
									}
									else if (parts[0].equals("5")) {
										if (parts[1] != null && !parts[1].isEmpty()) {
											labResult.setSpecimen(parts[1].trim());
											specimen = getSpecimenFromCache(parts[1].trim());
											if (specimen != null)
												labResult.setSpecimenId(specimen.getIEN());
											else {
												if (!specimenNames.contains(labResult.getSpecimen()))
													specimenNames.add(labResult.getSpecimen());
											}
										}										
									}
									else if (parts[0].equals("6")) {
										labResult.setAccessionNumber(parts[1]);
									}
									else if (parts[0].equals("7")) {
										labResult.appendText(parts[1]);
									}
									//if (parts[0].equals("8")) {
									else {
										break;
									}
								}

								ovidLabResults.add(labResult);
								//logger.debug(labResult);
							}
						}
					}

					// resolve any missing specimen ids
					logger.info("resolve any missing specimen ids");
					if (specimenNames.size() > 0) {
						logger.info("add specimens to cache");
						addSpecimensByNameToCache(specimenNames);
						for (OvidMicroLabResult entry : ovidLabResults) {
							logger.debug(String.format("checking entry for specimen [id=%s][name=%s]", entry.getSpecimenId(), entry.getSpecimen()));
							if (entry.getSpecimenId() == null || entry.getSpecimenId().isEmpty()) {
								logger.debug(String.format("entry does not have a specimen id, retrieve specimen by name [id=%s][name=%s]", entry.getSpecimenId(), entry.getSpecimen()));
								specimen = getSpecimenFromCache(entry.getSpecimen());
								logger.debug(String.format("retrieved specimen by name [name=%s][specimen=%s]", entry.getSpecimen(), specimen));
								
								//TODO: need test to cover this condition
								if (specimen == null) {
									entry.setSpecimenId(entry.getSpecimen());
								} else {
									entry.setSpecimenId(specimen.getIEN());
								}
								logger.debug(String.format("update rest test specimen to [id=%s][entry=%s]", entry.getSpecimenId() , entry));
							}
						}
						logger.info("end resolve any missing specimen ids");
					}								
				}
			} else {
				logger.error(response.getError());
				throw new OvidDomainException(response.getError());
			}
			return ovidLabResults;
		} catch (RPCException e) {
			logger.error("Exception while executing RPC \"" + RPC_ORWRP_REPORT_TEXT + "\"", e);
			throw new OvidDomainException(e);
		}
	}	
}
