package com.agilex.healthcare.mobilehealthplatform.ovid;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import com.agilex.healthcare.mobilehealthplatform.ovid.domain.OvidAppointment;
import com.medsphere.fileman.FMField.FIELDTYPE;
import com.medsphere.fileman.FMFile;
import com.medsphere.fileman.FMQueryByIENS;
import com.medsphere.fileman.FMQueryList;
import com.medsphere.fileman.FMRecord;
import com.medsphere.fileman.FMResultSet;
import com.medsphere.fileman.FMRetrieve;
import com.medsphere.fileman.FMUtil;
import com.medsphere.fmdomain.FMHospitalLocation;
import com.medsphere.fmdomain.FMPatient;
import com.medsphere.ovid.domain.ov.OvidDomainException;
import com.medsphere.resource.ResAdapter;
import com.medsphere.resource.ResException;
import com.medsphere.rpcresadapter.RPCResAdapter;
import com.medsphere.vistarpc.RPCConnection;
import com.medsphere.vistarpc.RPCException;
import com.medsphere.vistarpc.RPCResponse;
import com.medsphere.vistarpc.RPCResponse.ResponseType;
import com.medsphere.vistarpc.VistaRPC;

public class HospitalLocationRepository extends OvidConnectionAwareRepository {

	private final static org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(HospitalLocationRepository.class);

	// private final String CPRS_GUI_CONTEXT = "OR CPRS GUI CHART";
	private final String MSC_PATIENT_DASHBOARD = "MSC PATIENT DASHBOARD";

	// List of Fileman files
	private final String HOSPITAL_LOCATION_FILE = "44";
	private final String APPOINTMENT_SUBFILE = "1900";
	private final String PROVIDER_SUBFILE = "2600";

	// List of RPCs
	private final String RPC_GET_CLINICS = "MSC GET CLINICS";
	// private final String RPC_ORQPT_CLINIC_PATIENTS = "ORQPT CLINIC PATIENTS";
	private final String RPC_MSCQPT_CLINIC_PATIENTS = "MSCQPT CLINIC PATIENTS";
	protected boolean useRPC = true;

	public HospitalLocationRepository(RPCConnection connection) {
		super(connection);
	}

	/**
	 * Returns a list of appointments of a clinic between start date and stop
	 * date
	 * 
	 * @param clinicId
	 *            unique identifier of a clinic
	 * @param beginDate
	 *            start date
	 * @param endDate
	 *            end date
	 * @return a collection of appointments
	 * @throws OvidDomainException
	 */
	public Collection<OvidAppointment> getClinicAppointments(String clinicId, Date beginDate, Date endDate) throws OvidDomainException {
		Collection<OvidAppointment> list = new ArrayList<OvidAppointment>();
		RPCConnection rpcConnection = null;

		try {
			rpcConnection = getConnection();
			rpcConnection.setContext(FMUtil.FM_RPC_CONTEXT);

			// get information about a clinic, including the total number of
			// appointment entries available
			logger.debug(String.format("retrieve list of appointments for clinic [clinic ien=%s]", clinicId));
			FMRecord clinic = getClinicInfo(clinicId, rpcConnection);
			logger.debug(String.format("retrieved clinic info [clinic ien=%s][clinic name=%s][appt records=%s]", clinic.getIEN(), clinic.getValue("NAME"), clinic.getValue("APPOINTMENT")));
			
			//if (clinic.getValue("APPOINTMENT") != null && !clinic.getValue("APPOINTMENT").isEmpty()) {
				list = getClinicPatients(clinic, beginDate, endDate);
			//}

			logger.debug(String.format("retrieved appointment list for clinic.  [clinic=%s][records=%s]", clinicId, list.size()));

			return list;
		} catch (RPCException e2) {
			throw new OvidDomainException(e2);
		}
	}

	/**
	 * Returns all appointments of a clinic
	 * 
	 * @param clinicId
	 *            unique identifier of a clinic
	 * @return a collection of appointments
	 * @throws OvidDomainException
	 */
	public Collection<OvidAppointment> getClinicAppointments(String clinicId) throws OvidDomainException {
		Collection<OvidAppointment> list = new ArrayList<OvidAppointment>();
		RPCConnection rpcConnection = null;

		try {
			rpcConnection = getConnection();
			rpcConnection.setContext(FMUtil.FM_RPC_CONTEXT);

			// get information about a clinic, including the total number of
			// appointment entries available
			FMRecord clinic = getClinicInfo(clinicId, rpcConnection);

			// logger.debug("clinic ien=[" + clinic.getIEN() + "] name=[" +
			// clinic.getValue("NAME") +
			// "] abbreviation=[" + clinic.getValue("ABBREVIATION") +
			// "] # of appts=[" +
			// clinic.getValue("APPOINTMENT") + "]");

			if (clinic.getValue("APPOINTMENT") != null && !clinic.getValue("APPOINTMENT").isEmpty()) {
				FMFile appointmentSubfile = new FMFile(APPOINTMENT_SUBFILE);
				appointmentSubfile.addField(".01");

				// set parent
				appointmentSubfile.setParentRecord(clinic);

				// extract appointment entries
				RPCResAdapter adapter = new RPCResAdapter(rpcConnection, FMUtil.FM_RPC_NAME);
				FMQueryList secondaryQuery = new FMQueryList(adapter, appointmentSubfile);
				secondaryQuery.addField("APPOINTMENT DATE/TIME", FIELDTYPE.DATE);
				secondaryQuery.getField("APPOINTMENT DATE/TIME").setInternal(false);
				FMResultSet results = secondaryQuery.execute();

				FMRecord appointment = null;
				Date firstApptDate = null;

				// get date/time of first appointment (ever) in this clinic
				while (results.next()) {
					appointment = new FMRecord(results);
					logger.debug("   Appointment DTM=[" + appointment.getIEN() + "," + appointment.getValue("APPOINTMENT DATE/TIME") + "]");
					try {
						firstApptDate = FMUtil.fmDateToDate(appointment.getIEN());
						break;
					} catch (ParseException ignore) {
					}
				}

				// get date/time of last appointment in this clinic
				Date lastApptDate = null;
				while (results.next()) {
					appointment = new FMRecord(results);
					logger.debug("   Appointment DTM=[" + appointment.getIEN() + "," + appointment.getValue("APPOINTMENT DATE/TIME") + "]");
					try {
						lastApptDate = FMUtil.fmDateToDate(appointment.getIEN());
					} catch (ParseException ignore) {
					}
				}

				// has first date but no last date (one appointment only)
				if (lastApptDate == null && firstApptDate != null) {
					lastApptDate = firstApptDate;
				}

				list = getClinicPatients(clinic, firstApptDate, lastApptDate);
			}

			return list;
		} catch (ResException e1) {
			throw new OvidDomainException(e1);
		} catch (RPCException e2) {
			throw new OvidDomainException(e2);
		}
	}

	/**
	 * Returns a list of appointments for a provider between start date and stop
	 * date
	 * 
	 * @param providerId
	 *            unique identifier of a provider
	 * @param beginDate
	 *            start date
	 * @param endDate
	 *            end date
	 * @return a collection of appointments
	 * @throws OvidDomainException
	 */
	public Collection<OvidAppointment> getProviderAppointments(String providerId, Date beginDate, Date endDate) throws OvidDomainException {
		logger.debug(String.format("begin getProviderAppointments [useRpc=%s][provider=%s][startdate=%s][enddate=%s]", useRPC, providerId, beginDate, endDate));

		Collection<OvidAppointment> list = new ArrayList<OvidAppointment>();

		try {
			Collection<FMHospitalLocation> clinicsList = null;
			
			
			logger.debug("determine number of clinics associated with provider # of clinics for provider " + providerId);
			if (isUseRPC()) {
				clinicsList = getAssociatedClinics(providerId);
			} else {
				clinicsList = getAssociatedClinicsFM(providerId);
			}
			logger.debug("provider associated with " + clinicsList.size() + " clinic(s)");

			// get information about a clinic, including the total number of
			// appointment entries available
			int cnt = 0;
			for (FMHospitalLocation clinic : clinicsList) {
				cnt++;
				logger.debug(String.format("get appointments for provider's clinic [clinic=%s][progress=%s of %s]", clinic.getIEN(), cnt, clinicsList.size()));
				list.addAll(getClinicAppointments(clinic.getIEN(), beginDate, endDate));
			}

			logger.debug(String.format("end getProviderAppointments [rc=%s][useRpc=%s][provider=%s][startdate=%s][enddate=%s]", list.size(), useRPC, providerId, beginDate, endDate));
			return list;
		} catch (Exception e1) {
			throw new OvidDomainException(e1);
		}
	}

	/**
	 * Returns all appointments of a provider
	 * 
	 * @param providerId
	 *            unique identifier of a provider
	 * @return a collection of appointments
	 * @throws OvidDomainException
	 */
	public Collection<OvidAppointment> getProviderAppointments(String providerId) throws OvidDomainException {
		Collection<OvidAppointment> list = new ArrayList<OvidAppointment>();

		try {
			Collection<FMHospitalLocation> clinicsList = null;
			if (isUseRPC()) {
				clinicsList = getAssociatedClinics(providerId);
			} else {
				clinicsList = getAssociatedClinicsFM(providerId);
			}

			// logger.debug("# of clinics of provider=[" + providerId + "]: " +
			// clinicsList.size());

			// get information about a clinic, including the total number of
			// appointment entries available
			for (FMHospitalLocation clinic : clinicsList) {
				list.addAll(getClinicAppointments(clinic.getIEN()));
			}
			return list;
		} catch (Exception e1) {
			throw new OvidDomainException(e1);
		}
	}

	/**
	 * Calls RPC "MSCQPT CLINIC PATIENTS" (context "MSC PATIENT DASHBOARD") and
	 * returns patients with appointments at a clinic between start and stop
	 * dates.
	 * 
	 * @param
	 * @return
	 * @throws OvidDomainException
	 */
	private Collection<OvidAppointment> getClinicPatients(FMRecord clinic, Date beginDate, Date endDate) throws OvidDomainException {

		RPCConnection rpcConnection;
		Collection<OvidAppointment> appointmentList = new ArrayList<OvidAppointment>();
		Collection<String> iens = null;
		
		try {
			rpcConnection = getConnection();
			rpcConnection.setContext(MSC_PATIENT_DASHBOARD);
		} catch (RPCException e) {
			logger.error("Exception while setting the context", e);
			throw new OvidDomainException(e);
		}

		try {
			VistaRPC rpc = new VistaRPC(RPC_MSCQPT_CLINIC_PATIENTS, ResponseType.ARRAY);

			// set patient
			rpc.setParam(1, clinic.getIEN());
			// set start date
			rpc.setParam(2, getSimpleDateTimeFormat().format(beginDate));
			// set end date
			rpc.setParam(3, getSimpleDateTimeFormat().format(endDate));
			
			logger.debug(String.format("retrieve clinic's appointment details [clinic=%s][startdate=%s][enddate=%s]", clinic.getIEN(), beginDate, endDate));

			RPCResponse response = rpcConnection.execute(rpc);			
			if (response.getError() == null || response.getError().isEmpty()) {

				String items[] = response.getArray();
				if (isEmptyResult(items) || firstItemHasInvalidResults(items[0])) {
					logger.debug("No data");
				} else {
					// -----------------------------------------------------------------
					// Output format:
					// [patient ien]^[patient name]^[clinic ien]^[appointment
					// date/time]^^^^^[patient status]^[appointment reason]
					// where"
					// format of patient name is <last>,<first> <middle>
					// appointment date/time is in Fileman format
					// patient status has values of "IPT" (inpatient), "OPT"
					// (outpatient)
					//
					// Example: 9^TESTPATIENT,TWO^20^3110414.103^^^^^IPT^HEART
					// ATTACK, NEED FOLLOW-UP
					// -----------------------------------------------------------------
					OvidAppointment appointment = null;					
					iens = new ArrayList<String>(items.length);
					for (int idx = 0; idx < items.length; ++idx) {
						String parts[] = items[idx].split("\\^", -1);
						if (parts[3] != null && !parts[3].isEmpty()) {
							appointment = new OvidAppointment();

							try {
								appointment.setAppointmentDTM(FMUtil.fmDateToDate(parts[3]));

								appointment.setClinicName(clinic.getValue("NAME"));
								appointment.setClinicAbbreviation(clinic.getValue("ABBREVIATION"));
								appointment.setPatientName(parts[1]);
								appointment.setAppointmentType(parts[8]);
								appointment.setDuration(clinic.getValue("LENGTH OF APP'T"));
								//appointment.setPatientSSN(getSSN(parts[0]));
								appointment.setPatientId(parts[0]);
								appointment.setAppointmentReason(parts[9]);
								appointmentList.add(appointment);
								
								// add to list of patient IENs
								if (!iens.contains(parts[0])) {
									iens.add(parts[0]);								
								}
							} catch (ParseException e2) {
								logger.error("Encountered parsing appointment date/time " + parts[3], e2);
							}
						}
					}
				}

				// get SSN for each patient
				if (iens != null && !iens.isEmpty()) {
					Map<String, FMPatient> fmPatients = getPatientsInfo(iens);
					
					for (OvidAppointment appointment: appointmentList) {
						String ien = appointment.getPatientId();
						if (fmPatients.containsKey(ien)) {
							appointment.setPatientSSN(fmPatients.get(ien).getSsn());
						}
					}					
				}
								
				logger.debug(String.format("retrieved clinic's appointment details [clinic=%s][startdate=%s][enddate=%s][appointments=%s]", clinic.getIEN(), beginDate, endDate, appointmentList.size()));
				return appointmentList;
			} else {
				logger.error(response.getError());
				throw new OvidDomainException(response.getError());
			}
		} catch (RPCException e) {
			logger.error("Exception while executing RPC \"MSCQPT CLINIC PATIENTS\"", e);
			throw new OvidDomainException(e);
		}
	}

	private boolean firstItemHasInvalidResults(String messageFromFirstItem) {

		String msg = messageFromFirstItem.trim().toLowerCase();

		/*
		 * These are the possible messages that could come from VA Vista into
		 * the first item returned by RPC, but they are not errors; for the
		 * logic of this class it should could as no data condition:
		 * 
		 * - No clinic identified - Clinic is inactive or Occasion Of Service
		 * Error - in date range. - No appointments
		 */
		if (msg.contains("no appointments") || msg.contains("no clinic identified") || msg.contains("clinic is inactive or occasion of service") || msg.contains("error in date range")) {

			return true;

		} else {

			return false;
		}

	}

	/**
	 * Calls RPC "MSC GET CLINICS" (context "MSC PATIENT DASHBOARD") and returns
	 * a list of clinics linked to a provider
	 * 
	 * @param providerId
	 *            unique identifier of a provider
	 * @return a list of clinics
	 * @throws OvidDomainException
	 */
	public Collection<FMHospitalLocation> getAssociatedClinics(String providerId) throws OvidDomainException {

		RPCConnection rpcConnection;
		Collection<FMHospitalLocation> clinicsList = new ArrayList<FMHospitalLocation>();

		try {
			rpcConnection = getConnection();
			rpcConnection.setContext(MSC_PATIENT_DASHBOARD);
		} catch (RPCException e) {
			logger.error("Exception while setting the context", e);
			throw new OvidDomainException(e);
		}

		try {
			VistaRPC rpc = new VistaRPC(RPC_GET_CLINICS, ResponseType.ARRAY);
			rpc.setParam(1, providerId);

			RPCResponse response = rpcConnection.execute(rpc);

			if (response.getError() == null || response.getError().isEmpty()) {
				String[] items = response.getArray();
				if (isEmptyResult(items)) {
					logger.debug("No data");
				} else {
					// response is of format:
					// line 1: # of clinics
					// line 2-n: clinic id^clinic name
					int noOfClinics = 0;
					if (items[0] != null && !items[0].isEmpty())
						noOfClinics = Integer.parseInt(items[0]);

					FMHospitalLocation clinic = null;
					String[] parts = null;
					for (int idx = 1; idx <= noOfClinics && idx < items.length; ++idx) {
						parts = items[idx].split("\\^", -1);
						if (parts.length > 1) {
							clinic = new FMHospitalLocation();

							// set clinic id
							clinic.setIEN(parts[0]);

							// set clinic name
							clinic.setValue("NAME", parts[1], FIELDTYPE.FREE_TEXT);

							// add to list
							clinicsList.add(clinic);
						}
					}
				}

				return clinicsList;
			} else {
				logger.error(response.getError());
				throw new OvidDomainException(response.getError());
			}
		} catch (RPCException e) {
			logger.error("Exception while executing RPC \"" + RPC_GET_CLINICS + "\"", e);
			throw new OvidDomainException(e);
		}
	}

	/**
	 * Query HOSPITAL LOCATION file and return a list of clinics that the
	 * specified provider is associated with.
	 * 
	 * @param providerId
	 *            provider identifier
	 * @return a list of clinics
	 * @throws OvidDomainException
	 */
	public Collection<FMHospitalLocation> getAssociatedClinicsFM(String providerId) throws OvidDomainException {

		RPCConnection rpcConnection;
		RPCResAdapter adapter = null;
		Collection<FMHospitalLocation> clinicsList = new ArrayList<FMHospitalLocation>();

		try {
			// setup connection
			rpcConnection = getConnection();
			rpcConnection.setContext(FMUtil.FM_RPC_CONTEXT);
			adapter = new RPCResAdapter(rpcConnection, FMUtil.FM_RPC_NAME);

			FMFile hospitalLocationFile = new FMFile(HOSPITAL_LOCATION_FILE);
			hospitalLocationFile.addField(".01");
			
			FMQueryList query = new FMQueryList(adapter, hospitalLocationFile);
			FMResultSet results = query.execute();

			FMRecord result = null;
			while (results.next()) {
				result = new FMRecord(results);
				if (hasProviderInClinic(result, providerId, rpcConnection)) {
					FMHospitalLocation clinic = new FMHospitalLocation();
					clinic = new FMHospitalLocation();

					// set clinic id
					clinic.setIEN(result.getIEN());

					// add to list
					clinicsList.add(clinic);
				}
			}

			return clinicsList;
		} catch (ResException e1) {
			logger.error("Exception while searching in \"" + HOSPITAL_LOCATION_FILE + "\"", e1);
			throw new OvidDomainException(e1);
		} catch (RPCException e2) {
			logger.error("Exception while searching in \"" + HOSPITAL_LOCATION_FILE + "\"", e2);
			throw new OvidDomainException(e2);
		}
	}

	/**
	 * Query HOSPITAL LOCATION file and return information about a clinic
	 * including its appointments.
	 * 
	 * @param clinicId
	 * @return
	 * @throws OvidDomainException
	 */
	private FMRecord getClinicInfo(String clinicId, RPCConnection rpcConnection) throws OvidDomainException {
		RPCResAdapter adapter = null;

		try {
			rpcConnection.setContext(FMUtil.FM_RPC_CONTEXT);
			adapter = new RPCResAdapter(rpcConnection, FMUtil.FM_RPC_NAME);

			FMRecord clinic = new FMRecord(HOSPITAL_LOCATION_FILE);
			clinic.setIEN(clinicId); //

			FMRetrieve query = new FMRetrieve(adapter);
			query.setRecord(clinic);
			query.setInternal(false); // we want the text of the field it points
										// to, not the IEN

			query.addField("NAME", FIELDTYPE.POINTER_TO_FILE);
			// element abbreviation
			query.addField("ABBREVIATION", FIELDTYPE.FREE_TEXT);
			// element length of appointment
			query.addField("LENGTH OF APP'T", FIELDTYPE.NUMERIC);
			// element appointment (1900)
			//query.addField("APPOINTMENT", FIELDTYPE.SUBFILE);

			// locate record
			query.execute();
			return clinic;
		} catch (ResException e1) {
			throw new OvidDomainException(e1);
		} catch (RPCException e2) {
			throw new OvidDomainException(e2);
		}
	}

	private boolean hasProviderInClinic(FMRecord clinic, String providerId, RPCConnection rpcConnection) {
		RPCResAdapter adapter = null;

		try {
			rpcConnection.setContext(FMUtil.FM_RPC_CONTEXT);
			adapter = new RPCResAdapter(rpcConnection, FMUtil.FM_RPC_NAME);

			FMFile subfile = new FMFile(PROVIDER_SUBFILE);
			subfile.addField("PROVIDER");
			subfile.addField("DEFAULT PROVIDER", FIELDTYPE.SET_OF_CODES);

			// So the query knows which set of subfile records
			subfile.setParentRecord(clinic);

			FMQueryList secondaryQuery = new FMQueryList(adapter, subfile);
			secondaryQuery.getField("PROVIDER").setInternal(true);

			FMResultSet results = secondaryQuery.execute();

			FMRecord result = null;
			while (results.next()) {
				result = new FMRecord(results);

				// logger.debug("   provider=[#" + result.getIEN() + "] ien=[" +
				// result.getValue("PROVIDER") +
				// "] default=[" + result.getValue("DEFAULT PROVIDER") + "]");

				if (result.getValue("PROVIDER").equalsIgnoreCase(providerId))
					return true;
			}

			return false;
		} catch (ResException e1) {
			logger.error("Exception while querying SUBFILE \"" + PROVIDER_SUBFILE + "\"", e1);
		} catch (RPCException e2) {
			logger.error("Exception while querying SUBFILE \"" + PROVIDER_SUBFILE + "\"", e2);
		}

		return false;
	}

	/**
	 * Return full SSN of a patient. [NOTE: stay here for now, will be move out
	 * to include with patient search implementations!!]
	 * 
	 * @param patientId
	 *            patient unique identifier
	 * @return
	 */
//	private String getSSN(String patientId) {
//		String ssn = "";
//		PatientRepository patientRepo = new PatientRepository(getConnection(), getConnection());
//
//		try {
//			FMPatient patient = patientRepo.getPatientByIEN(patientId);
//			if (patient != null)
//				ssn = patient.getSsn();
//		} catch (OvidDomainException e) {
//			logger.error("Exception while getting SSN of patient " + patientId, e);
//		}
//
//		return ssn;
//	}

	/**
	 * Return full SSN of a patient. [NOTE: stay here for now, will be move out
	 * to include with patient search implementations!!]
	 * 
	 * @param patientId
	 *            patient unique identifier
	 * @return
	 * @throws OvidDomainException 
	 */
	private Map<String, FMPatient> getPatientsInfo(Collection<String> iens) throws OvidDomainException {
		Map<String, FMPatient> list = new HashMap<String, FMPatient>();
		try {
			RPCConnection rpcConnection = getConnection();
			rpcConnection.setContext(FMUtil.FM_RPC_CONTEXT);
			ResAdapter adapter = new RPCResAdapter(getConnection(), FMUtil.FM_RPC_NAME);
			
			FMQueryByIENS query = new FMQueryByIENS(adapter, FMPatient.getFileInfoForClass());
			query.setIENS(iens);				

			FMResultSet results = query.execute();
			FMPatient fmPatient = null;
			if (results != null) {
				if (results.getError() != null) {
					throw new OvidDomainException(results.getError());
				}
				while (results.next()) {
					fmPatient = new FMPatient(results);
					list.put(fmPatient.getIEN(), fmPatient);
				}
			}
		} catch (ResException e1) {
			throw new OvidDomainException(e1);
		} catch (RPCException e2) {
			throw new OvidDomainException(e2);
		}

		return list;
	}
	
	private SimpleDateFormat getSimpleDateTimeFormat() {
		return new SimpleDateFormat("MM/dd/yyyy");
	}

	public boolean isUseRPC() {
		return useRPC;
	}

	public void setUseRPC(boolean useRPC) {
		this.useRPC = useRPC;
	}
}
