package com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.lab;

import java.util.regex.Pattern;

import com.agilex.healthcare.mobilehealthplatform.domain.LabResult;
import com.agilex.healthcare.mobilehealthplatform.ovid.domain.OvidTestResult;
import com.agilex.healthcare.utility.NullChecker;

public class ReferenceRangeFormatter {
	private static final String UPPER_BOUND_INDICATOR = "<";
	private static final String LOWER_BOUND_INDICATOR = ">";
	private static final String REFERENCE_RANGE_DELIM = "-";
//	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(ReferenceRangeFormatter.class);

	public void fillReferenceRange(OvidTestResult ovidLabResultDetail, LabResult labResult) {
		if (doesOvidContainsDiscreteReferenceRangeValues(ovidLabResultDetail)) {
			copyReferenceRangeFromOvid(ovidLabResultDetail, labResult);
		} else {
			fillReferenceRange(ovidLabResultDetail.getReferenceRange(), labResult);
		}

//		logger.debug(String.format("Parsed lab reference range [testid=%s][ovid.low=%s][ovid.high=%s][ovid.range=%s][actual.low=%s][actual.high=%s]", ovidLabResultDetail.getTestId(),
//				ovidLabResultDetail.getRefLow(), ovidLabResultDetail.getRefHi(), ovidLabResultDetail.getReferenceRange(), labResult.getReferenceLow(), labResult.getReferenceHigh()));
	}

	private boolean doesOvidContainsDiscreteReferenceRangeValues(OvidTestResult ovidLabResultDetail) {
		return NullChecker.isNotNullish(ovidLabResultDetail.getRefHi()) || NullChecker.isNotNullish(ovidLabResultDetail.getRefLow());
	}

	private void copyReferenceRangeFromOvid(OvidTestResult ovidLabResultDetail, LabResult labResult) {
		labResult.setReferenceLow(ovidLabResultDetail.getRefLow());
		labResult.setReferenceHigh(ovidLabResultDetail.getRefHi());
	}

	public void fillReferenceRange(String referenceRange, LabResult labResult) {
		String referenceLow = null;
		String referenceHigh = null;

		if (NullChecker.isNotNullish(referenceRange)) {
			referenceRange = referenceRange.replace(" " , "");
			Pattern xDashy = Pattern.compile("[0-9\\.]+-[0-9\\.]+", Pattern.CASE_INSENSITIVE);
			if (xDashy.matcher(referenceRange).matches()) {
				String[] refRangeParts = referenceRange.split(REFERENCE_RANGE_DELIM);
				referenceLow = formatLowerBound(refRangeParts[0]);
				if (refRangeParts.length == 2)
					referenceHigh = formatUpperBound(refRangeParts[1]);
			}
			
			Pattern xDash = Pattern.compile("[0-9\\.]+-", Pattern.CASE_INSENSITIVE);
			if (xDash.matcher(referenceRange).matches() ) { 
				referenceLow = referenceRange.replace("-", "");
			} 
			
			Pattern dashY = Pattern.compile("-[0-9\\.]+", Pattern.CASE_INSENSITIVE);
			if (dashY.matcher(referenceRange).matches() ) { 
				referenceHigh = referenceRange.replace("-", "");
			}
			
			Pattern lowerBoundPattern = Pattern.compile("[-]*>[0-9\\.]+[-]*", Pattern.CASE_INSENSITIVE);
			if (lowerBoundPattern.matcher(referenceRange).matches() ) { 
				referenceLow = formatLowerBound ( referenceRange.replace("-", ""));
			}
			Pattern upperBoundPattern = Pattern.compile("[-]*<[0-9\\.]+[-]*", Pattern.CASE_INSENSITIVE);
			if (upperBoundPattern.matcher(referenceRange).matches() ) { 
				referenceHigh = formatUpperBound ( referenceRange.replace("-", ""));
			}
			
			if (!areReferenceRangesSet(referenceLow, referenceHigh) && isLowerBound(referenceRange)) {
				referenceLow = formatLowerBound(referenceRange);
			}
			if (!areReferenceRangesSet(referenceLow, referenceHigh) && isUpperBound(referenceRange)) {
				referenceHigh = formatUpperBound(referenceRange);
			}
		}
		labResult.setReferenceLow(referenceLow);
		labResult.setReferenceHigh(referenceHigh);
	}

	private boolean areReferenceRangesSet(String referenceLow, String referenceHigh) {
		return (NullChecker.isNotNullish(referenceLow) || NullChecker.isNotNullish(referenceHigh));
	}

	private boolean isUpperBound(String referenceRange) {
		return referenceRange.contains(UPPER_BOUND_INDICATOR);
	}

	private boolean isLowerBound(String referenceRange) {
		return referenceRange.contains(LOWER_BOUND_INDICATOR);
	}

	private String formatLowerBound(String s) {
		return formatReferenceRangeValue(s, LOWER_BOUND_INDICATOR);
	}

	private String formatUpperBound(String s) {
		return formatReferenceRangeValue(s, UPPER_BOUND_INDICATOR);
	}

	private String formatReferenceRangeValue(String s, String irrelevantIndicator) {
		if (s != null) {
			s = s.trim();
			s = s.replace(irrelevantIndicator, "");
			if (NullChecker.isNullish(s))
				s = null;
		}
		return s;
	}
}
