package com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.lab;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.agilex.healthcare.mobilehealthplatform.datalayer.lab.micro.MicrobiologyLabDataLayer;
import com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.connectionmanagement.ConnectionManagerFactory;
import com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.connectionmanagement.IOvidConnectionManager;
import com.agilex.healthcare.mobilehealthplatform.domain.LabResult;
import com.agilex.healthcare.mobilehealthplatform.domain.LabResults;
import com.agilex.healthcare.mobilehealthplatform.domain.LabTestGroup;
import com.agilex.healthcare.mobilehealthplatform.domain.LabTestGroups;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.ovid.LabResultRepository;
import com.agilex.healthcare.mobilehealthplatform.ovid.LabTestRepository;
import com.agilex.healthcare.mobilehealthplatform.ovid.domain.OvidMicroLabResult;
import com.agilex.healthcare.utility.NullChecker;
import com.medsphere.ovid.domain.ov.OvidDomainException;

public class MicrobiologyLabDataLayerOvid implements MicrobiologyLabDataLayer {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(MicrobiologyLabDataLayerOvid.class);

	@Override
	public LabResults getLabResults(PatientIdentifier patientIdentifier, DateFilter dateFilter) {
		String patientId = patientIdentifier.getUniqueId();
		IOvidConnectionManager connectionmanager = ConnectionManagerFactory.getInstance();
		LabResultRepository repo = connectionmanager.getLabResultRepository();
		Collection<OvidMicroLabResult> ovidResults;
		try {
			logger.info(String.format("retrieve micro lab results [patient=%s][startdate=%s][enddate=%s]", patientId, dateFilter.getStartDate(), dateFilter.getEndDate()));
			ovidResults = repo.getMicroResultsByDate(patientId, dateFilter.getStartDate(), dateFilter.getEndDate());
			logger.info(String.format("retrieved %s micro lab results [patient=%s][startdate=%s][enddate=%s]", ovidResults.size(), patientId, dateFilter.getStartDate(), dateFilter.getEndDate()));
		} catch (OvidDomainException e) {
			throw new RuntimeException("Unable to retrieve micro lab results", e);
		} finally {
			connectionmanager.releaseRepositoryConnections(repo);
		}

		LabResults results = convertOvidTestResults(ovidResults, patientId);
		return results;
	}

	@Override
	public LabTestGroups lookupLabGroupsByTestId(String labTestId) {
		Map<String, Collection<String>> ovidLabGroups = getOvidLabGroups();
		LabTestGroups groups = convertOvidLabGroups(ovidLabGroups.get(labTestId));
		return groups;
	}

	@Override
	public Map<String, LabTestGroups> lookupLabGroupsByTestIds(List<String> labTestIds) {
		Map<String, Collection<String>> ovidLabGroups = getOvidLabGroups();
		Map<String, LabTestGroups> map = new HashMap<String, LabTestGroups>();
		for (String labTestId : (String[]) labTestIds.toArray()) {
			LabTestGroups groups = convertOvidLabGroups(ovidLabGroups.get(labTestId));
			map.put(labTestId, groups);
		}
		return map;
	}

	@Override
	public LabResults getLabResultsByTestIdAndSpecimenId(PatientIdentifier patientId, String testId, String specimenId) {
		LabResults labResults = getLabResultsByTestId(patientId, testId);
		LabResults filteredLabResults = new LabResults();
		for (LabResult labResult : labResults) {
			if (specimenId.equals(labResult.getSpecimenId())) {
				filteredLabResults.add(labResult);
			}
		}
		return filteredLabResults;
	}

	@Override
	public LabResults getLabResultsByTestId(PatientIdentifier patientIdentifier, String testId) {
		return null;
	}

	private LabResults convertOvidTestResults(Collection<OvidMicroLabResult> ovidResults, String patientId) {
		logger.debug("start converting lab results");
		LabResults results = new LabResults();
		for (OvidMicroLabResult ovidLabResult : ovidResults) {
			LabResult result = convertOvidLabResultDetail(ovidLabResult, patientId);
			results.add(result);
		}
		logger.debug("end converting lab results");
		return results;
	}

	private LabResult convertOvidLabResultDetail(OvidMicroLabResult ovidLabResult, String patientId) {
		LabResult result = new LabResult();
		result.setTestname(ovidLabResult.getTestName());
		result.setAccessionNumber(ovidLabResult.getAccessionNumber());
		result.setOrderId(ovidLabResult.getOrderId());
		result.setValue(ovidLabResult.getText());
		result.setValueUnits(null);
		result.setIndicator(null);
		result.setStatus(ovidLabResult.getStatus());
		result.setLabType(ovidLabResult.getLabType());
		result.setTestId(ovidLabResult.getTestId());
		result.setPatientId(patientId);
		result.setResultedDate(ovidLabResult.getResultedDate());
		result.setSpecimenId(ovidLabResult.getSpecimenId());
		result.setSpecimenName(ovidLabResult.getSpecimen());

		if (NullChecker.isNullish(result.getTestId()))
			result.setTestId(result.getTestname());

		if (NullChecker.isNullish(result.getSpecimenId()))
			result.setSpecimenId(result.getSpecimenName());

		return result;
	}

	private Map<String, Collection<String>> getOvidLabGroups() {
		IOvidConnectionManager connectionmanager = ConnectionManagerFactory.getInstance();
		LabTestRepository repo = connectionmanager.getLabTestRepository();
		Map<String, Collection<String>> ovidLabGroups;
		try {
			logger.info("retrieve lab groups");
			ovidLabGroups = repo.getLabGroups();
			logger.info("retrieved lab groups");
		} catch (OvidDomainException e) {
			throw new RuntimeException("Unable to retrieve lab results", e);
		} finally {
			connectionmanager.releaseRepositoryConnections(repo);
		}
		return ovidLabGroups;
	}

	private LabTestGroups convertOvidLabGroups(Collection<String> ovidLabGroups) {
		LabTestGroups groups = new LabTestGroups();
		for (String group : ovidLabGroups) {
			LabTestGroup labGroup = new LabTestGroup();
			labGroup.setName(group);
			groups.add(labGroup);
		}
		return groups;
	}

	@Override
	public LabResults getLabResultsLite(String patientId, DateFilter dateFilter) {
		return null;
	}
}
