package com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.connectionmanagement.pooled;

import java.util.Date;

import com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.connectionmanagement.AbstractConnectionManager;
import com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.connectionmanagement.OvidConfiguration;
import com.agilex.healthcare.mobilehealthplatform.healthcheck.SystemProperties;
import com.agilex.healthcare.utility.DateHelper;
import com.medsphere.vistarpc.RPCConnection;
import com.medsphere.vistarpc.RPCException;
import com.medsphere.vistarpc.RPCResponse;
import com.medsphere.vistarpc.VistaRPC;

/**
 * Provides a pool of user connections and server connections that grows as
 * needed.
 * 
 */
public class OvidConnectionManagerWithPooling extends AbstractConnectionManager {

	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(OvidConnectionManagerWithPooling.class);
	protected GenericPool<RPCBrokerPooledConnection> serverPool = new GenericPool<RPCBrokerPooledConnection>();
	protected GenericPool<RPCBrokerPooledConnection> userPool = new GenericPool<RPCBrokerPooledConnection>();
	public static final String SERVER_CONNECTION = "server";
	public static final String USER_CONNECTION = "user";
	private static boolean isPoolInitialized = false;
	private boolean validateReusedConnections = true;

	public OvidConnectionManagerWithPooling(OvidConfiguration configuration) {
		super(configuration);

		ValidateConnectionEvent validateCallback = new ValidateConnectionEvent(this);

		this.validateReusedConnections = configuration.isValidateReusedConnections();
		boolean keepFreeConnectionsAlive = configuration.isKeepFreeConnectionsAlive();
		boolean keepAllConnectionsAlive = configuration.isKeepAllConnectionsAlive();
		int secondsBetweenConnectionRefresh = configuration.getSecondsBetweenConnnectionRefresh();

		if (keepAllConnectionsAlive) {
			serverPool.addTimerCallback(validateCallback, secondsBetweenConnectionRefresh * 1000, true);
			userPool.addTimerCallback(validateCallback, secondsBetweenConnectionRefresh * 1000, true);
		} else if (keepFreeConnectionsAlive) {
			serverPool.addTimerCallback(validateCallback, secondsBetweenConnectionRefresh * 1000, false);
			userPool.addTimerCallback(validateCallback, secondsBetweenConnectionRefresh * 1000, false);
		}
	}

	@Override
	public synchronized RPCConnection getServerConnection() {
		return getConnection(serverPool, configuration.getServer(), configuration.getPort(), configuration.getOvidAccessCode(), configuration.getOvidVerifyCode(), this, SERVER_CONNECTION);
	}

	@Override
	public synchronized RPCConnection getUserConnection() {
		return getConnection(userPool, configuration.getServer(), configuration.getPort(), configuration.getUserAccessCode(), configuration.getUserVerifyCode(), this, USER_CONNECTION);

	}

	private synchronized RPCConnection getConnection(GenericPool<RPCBrokerPooledConnection> pool, String host, int port, String accessCode, String verifyCode, OvidConnectionManagerWithPooling source,
			String connectionType) {
		initializeConnectionsIfNotInitialized();

		logger.info("Connection requested (pre), current stats: " + getStats(connectionType, pool));
		RPCBrokerPooledConnection connection = pool.getObject();
		if (connection != null) {
			logger.debug("Reusing available connection " + getStats(connectionType, pool, connection));

			if (this.validateReusedConnections) {
				logger.debug("Validating new connection");
				boolean valid = validateConnection(connection);
				if (valid) {
					logger.info("Connection requested (post), reusing available connection.  stats: " + getStats(connectionType, pool, connection));
				} else {
					// todo: move removeObject to connection.release
					pool.removeObject(connection);
					connection = createConnection(host, port, accessCode, verifyCode, source, connectionType);
					pool.addObject(connection);
					logger.info("Connection requested (post), pooled connection not valid, created new connection.  stats: " + getStats(connectionType, pool, connection));
				}
			} 
		} else {
			connection = createConnection(host, port, accessCode, verifyCode, source, connectionType);
			pool.addObject(connection);
			logger.info("Connection requested (post), no available connection, created new connection.  stats: " + getStats(connectionType, pool, connection));
		}

		logger.debug("Handing out connection, current stats: " + getStats(connectionType, pool, connection));
		return connection;
	}

	private String getStats(String connectionType, GenericPool<RPCBrokerPooledConnection> pool, RPCConnection connection) {
		reportStats(connectionType, pool);
		return String.format("[pool=%s][free=%s][used=%s][c=%s]", connectionType, pool.getNumObjectsFree(), pool.getNumObjectsUsed(), connection.toString());
	}

	private String getStats(String connectionType, GenericPool<RPCBrokerPooledConnection> pool) {
		reportStats(connectionType, pool);
		return String.format("[pool=%s][free=%s][used=%s]", connectionType, pool.getNumObjectsFree(), pool.getNumObjectsUsed());
	}

	private void reportStats(String connectionType, GenericPool<RPCBrokerPooledConnection> pool) {
		SystemProperties.setValue("ovid-cm-" + connectionType + "-connections-free", Integer.toString(pool.getNumObjectsFree()));
		SystemProperties.setValue("ovid-cm-" + connectionType + "-connections-used", Integer.toString(pool.getNumObjectsUsed()));
	}

	private RPCBrokerPooledConnection createConnection(String host, int port, String accessCode, String verifyCode, OvidConnectionManagerWithPooling source, String connectionType) {

		RPCBrokerPooledConnection connection;
		try {
			logger.info(String.format("initialize connection [s=%s][p=%s][a=%s][v=%s]", host, port, accessCode, verifyCode));
			connection = new RPCBrokerPooledConnection(host, port, accessCode, verifyCode, source, connectionType);
			SystemProperties.setValue("ovid-cm-" + connectionType + "-connections-lastcreated", DateHelper.formatDateTime(new Date()));
			logger.debug("created new RPCBrokerPooledConnection.  " + connection.toString());
		} catch (RPCException e) {
			throw new RuntimeException("Failed to create new RPCBrokerPooledConnection", e);
		}

		return connection;
	}

	@Override
	public void releaseConnection(RPCConnection connection) {
		if (connection != null) {
			if (connection instanceof RPCBrokerPooledConnection) {
				releasePooledConnection((RPCBrokerPooledConnection) connection);
			} else {
				super.releaseConnection(connection);
			}
		} else {
			logger.debug("unable to close connection, connection does not exist");
		}
	}

	private void releasePooledConnection(RPCBrokerPooledConnection connection) {
		GenericPool<RPCBrokerPooledConnection> pool;
		int maxNumberOfConnections;
		if (connection.getType() == SERVER_CONNECTION) {
			pool = serverPool;
			maxNumberOfConnections = configuration.getMaxFreeServerConnections();
		} else if (connection.getType() == USER_CONNECTION) {
			pool = userPool;
			maxNumberOfConnections = configuration.getMaxFreeUserConnections();
		} else {
			throw new RuntimeException("Invalid connection type " + connection.getType());
		}
		releasePooledConnection(connection, connection.getType(), pool, maxNumberOfConnections);
	}

	private void releasePooledConnection(RPCBrokerPooledConnection connection, String connectionType, GenericPool<RPCBrokerPooledConnection> pool, int MaxNumberOfConnections) {
		logger.debug("begin release connection. " + connection.toString());
		try {
			if (pool.getNumObjectsFree() >= MaxNumberOfConnections) {
				connection.terminate();
				pool.removeObject(connection);
				logger.info("Release request (post), max connections exceeded, removing connection from pool, current stats: " + getStats(connectionType, pool, connection));
			} else {
				pool.returnObject(connection);
				logger.info("Release request (post), connection returned to pool, current stats: " + getStats(connectionType, pool, connection));
			}
		} catch (RPCException e) {
			logger.error("Error occurred while releasing connection", e);
			throw new RuntimeException(e);
		}
	}

	public synchronized void terminateAllConnections() {
		terminateAllConnections(serverPool);
		terminateAllConnections(userPool);
		isPoolInitialized = false;
	}

	private synchronized void terminateAllConnections(GenericPool<RPCBrokerPooledConnection> pool) {
		for (RPCBrokerPooledConnection connection : pool.getAllObjects()) {
			try {
				connection.terminate();
				logger.debug("Terminated connection");
			} catch (RPCException e) {
				// ignore
			}
			pool.removeObject(connection);
		}
	}

	public void resetConnections() {
		terminateAllConnections();
		initializeConnections();
	}

	public boolean validateConnection(RPCBrokerPooledConnection connection) {
		if (connection == null) {
			return false;
		}
		try {
			logger.debug("Validating connection " + connection.toString());
			VistaRPC rpc = new VistaRPC("XWB IM HERE");
			RPCResponse response = connection.execute(rpc);
			if (response != null) {
				logger.debug("Validation response: [error] " + response.getError() + " : [type] " + response.getResponseType() + " : [string] " + response.getString());
			} else {
				logger.debug("Validation response is null.");
			}
			if (response == null || response.getError() != null || !"1".equals(response.getString())) {
				logger.info("Validation failed.  Connection will be terminated and a new connection will be created.");
				connection.terminate();
				return false;
			}
			logger.debug("Connection is valid. " + connection.toString());

			return true;
		} catch (RPCException e) {
			try {
				logger.debug("Validation failed.  Connection will be terminated and a new connection will be created.");
				connection.terminate();
			} catch (RPCException e1) {
				logger.error(e1);
			}
			return false;
		}
	}

	private synchronized void initializeConnectionsIfNotInitialized() {
		if (!isPoolInitialized) {
			initializeConnections();
			isPoolInitialized = true;
		}
	}

	private void initializeConnections() {
		try {
			logger.info(String.format("initializing connection pools.  [initial server connections=%s][inital user connections=%s]", configuration.getInitialServerConnections(),
					configuration.getInitialUserConnections()));
			initializeConnections(serverPool, SERVER_CONNECTION, configuration.getInitialServerConnections(), configuration.getServer(), configuration.getPort(), configuration.getOvidAccessCode(),
					configuration.getOvidVerifyCode(), this);
			initializeConnections(userPool, USER_CONNECTION, configuration.getInitialUserConnections(), configuration.getServer(), configuration.getPort(), configuration.getUserAccessCode(),
					configuration.getUserVerifyCode(), this);
			logger.info("initialized connection pools");
		} catch (Throwable e) {
			logger.error("Error initializing connection pool", e);
			throw new RuntimeException(e);
		}
	}

	private void initializeConnections(GenericPool<RPCBrokerPooledConnection> pool, String connectionType, int initialConnections, String host, int port, String accessCode, String verifyCode,
			OvidConnectionManagerWithPooling source) {
		logger.info(String.format("initializing connection pool.  [pool=%s][initial connections=%s]", connectionType, initialConnections));

		for (int i = 0; i < configuration.getInitialServerConnections(); i++) {
			RPCBrokerPooledConnection connection = createConnection(host, port, accessCode, verifyCode, source, connectionType);
			pool.addObject(connection);
			// object is automatically in use, must return object to pool
			pool.returnObject(connection);
			logger.info(String.format("Create connection during init phase, current stats: " + getStats(connectionType, pool, connection)));
		}

		SystemProperties.setValue("ovid-cm-" + connectionType + "-connections-initialized", DateHelper.formatDateTime(new Date()));
	}

	public int getNumServerConnectionsFree() {
		return serverPool.getNumObjectsFree();
	}

	public int getNumServerConnectionsUsed() {
		return serverPool.getNumObjectsUsed();
	}

	public int getNumUserConnectionsFree() {
		return userPool.getNumObjectsFree();
	}

	public int getNumUserConnectionsUsed() {
		return userPool.getNumObjectsUsed();
	}

	public boolean isValidateNewConnections() {
		return validateReusedConnections;
	}

	public void setValidateNewConnections(boolean validateNewConnections) {
		this.validateReusedConnections = validateNewConnections;
	}

}
