package com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.connectionmanagement;

import com.agilex.healthcare.mobilehealthplatform.ovid.HospitalLocationRepository;
import com.agilex.healthcare.mobilehealthplatform.ovid.LabResultRepository;
import com.agilex.healthcare.mobilehealthplatform.ovid.LabTestRepository;
import com.agilex.healthcare.mobilehealthplatform.ovid.MedicationRepository;
import com.agilex.healthcare.mobilehealthplatform.ovid.OvidPatientRepository;
import com.agilex.healthcare.mobilehealthplatform.ovid.TopographyFieldRepository;
import com.agilex.healthcare.mobilehealthplatform.ovid.VitalSignRepository;
import com.medsphere.ovid.domain.ov.OvidSecureRepository;
import com.medsphere.ovid.domain.ov.PatientItemRepository;
import com.medsphere.ovid.domain.ov.PatientMovementRepository;
import com.medsphere.ovid.domain.ov.PatientRepository;
import com.medsphere.ovid.domain.ov.ProblemRepository;
import com.medsphere.vistarpc.RPCBrokerConnection;
import com.medsphere.vistarpc.RPCConnection;
import com.medsphere.vistarpc.RPCException;

public class AbstractConnectionManager implements IOvidConnectionManager {

	protected static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(AbstractConnectionManager.class);

	private static final String ALLERGY_RPC_CONTEXT = "OR CPRS GUI CHART";
	private static final String MEDICATION_RPC_CONTEXT = "OR CPRS GUI CHART";
	private static final String PROBLEM_RPC_CONTEXT = "MSC PATIENT DASHBOARD";

	protected OvidConfiguration configuration = null;

	public AbstractConnectionManager(OvidConfiguration configuration) {
		this.configuration = configuration;
	}

	@Override
	public RPCConnection getUserConnection() {
		return getConnection(configuration.getOvidAccessCode(), configuration.getOvidVerifyCode());
	}

	@Override
	public RPCConnection getServerConnection() {
		return getConnection(configuration.getOvidAccessCode(), configuration.getOvidVerifyCode());
	}

	private RPCConnection getConnection(String accessCode, String verifyCode) {
		RPCBrokerConnection connection;
		try {
			logger.info(String.format("initialize connection [s=%s][p=%s][a=%s][v=%s]", configuration.getServer(), configuration.getPort(), accessCode, verifyCode));
			connection = new RPCBrokerConnection(configuration.getServer(), configuration.getPort(), accessCode, verifyCode);
		} catch (RPCException e) {
			throw new RuntimeException(e);
		}
		return connection;
	}

	@Override
	public PatientRepository getPatientRepository() {
		return new PatientRepository(getUserConnection(), getServerConnection());
	}

	@Override
	public PatientMovementRepository getPatientMovementRepository() {
		return new PatientMovementRepository(getServerConnection());
	}

	@Override
	public OvidPatientRepository getOvidPatientRepository() {
		return new OvidPatientRepository(getUserConnection(), getServerConnection());
	}

	@Override
	public PatientItemRepository getAllergyRepository() {
		return getPatientItemRepository(ALLERGY_RPC_CONTEXT);
	}

	@Override
	public MedicationRepository getMedicationRepository() {
		return new MedicationRepository(getServerConnection(), MEDICATION_RPC_CONTEXT);
	}

	@Override
	public VitalSignRepository getVitalSignRepository() {
		return new VitalSignRepository(getServerConnection());
	}

	@Override
	public LabResultRepository getLabResultRepository() {
		return new LabResultRepository(getServerConnection());
	}

	@Override
	public LabTestRepository getLabTestRepository() {
		return new LabTestRepository(getServerConnection());
	}

	@Override
	public ProblemRepository getProblemRepository() {
		return new ProblemRepository(getServerConnection());
	}

	@Override
	public HospitalLocationRepository getHospitalLocationRepository() {
		return new HospitalLocationRepository(getServerConnection());
	}

	@Override
	public void releaseRepositoryConnections(OvidSecureRepository repository) {
		if (repository != null) {
			logger.debug("preparing to release server connection on repository " + repository.toString());
			releaseConnection(repository.getServerConnection());
			logger.debug("preparing to release user connection on repository " + repository.toString());
			releaseConnection(repository.getConnection());
		} else {
			logger.warn("Was asked to release connections for a repository, but repository was null.  Check to ensure that connections were properly released.");
		}
	}

	@Override
	public void releaseConnection(RPCConnection connection) {
		try {
			if (connection != null) {
				logger.debug("closing connection " + connection.toString());
				connection.close();
				logger.debug("connection close " + connection.toString());
			} else {
				logger.debug("unable to close connection, connection does not exist");
			}
		} catch (RPCException e) {
			logger.error("Error occurred while releasing connection", e);
			throw new RuntimeException("failed to close connection.", e);
		}
	}

	@Override
	public PatientItemRepository getPatientItemRepository(String rpcContext) {
		return new PatientItemRepository(getUserConnection(), getServerConnection(), rpcContext);
	}

	@Override
	public PatientItemRepository getProblemPatientItemRepository() {
		return getPatientItemRepository(PROBLEM_RPC_CONTEXT);
	}

	@Override
	public TopographyFieldRepository getTopographyFieldRepository() {
		return new TopographyFieldRepository(getServerConnection());
	}

}
